const express = require('express');
const path = require('path');
const fs = require("fs");
const mysql = require("mysql2");
const bodyParser = require('body-parser');
const multer = require('multer');
const session = require('express-session');
require('dotenv').config();

const app = express();
const PORT = process.env.PORT || 3001;

app.listen(PORT, () => {
  console.log(`TEC app running on port ${PORT}`);
});

app.use(express.static("public"));
app.use("/pdf", express.static("public/pdf"));
app.use('/pdf', express.static(path.join(__dirname, 'public/pdf')));
app.use('/files', express.static(path.join(__dirname, 'public/files')));
app.use("/uploads", express.static(path.join(__dirname, "public/uploads")));
app.use('/images', express.static(path.join(__dirname, 'public/images')));

// Handle Chrome DevTools .well-known requests (suppress CSP warning)
// app.get('/.well-known/*', (req, res) => {
//   res.status(404).end();
// });

function getUniqueFileName(folderPath, originalName) {
  const ext = path.extname(originalName);
  const baseName = path.basename(originalName, ext);

  let fileName = originalName;
  let counter = 1;

  while (fs.existsSync(path.join(folderPath, fileName))) {
    fileName = `${baseName}-${counter}${ext}`;
    counter++;
  }

  return fileName;
}

const imageStorage = multer.diskStorage({
  destination: function (req, file, cb) {
    cb(null, path.join(__dirname, 'public/images'));
  },
  filename: function (req, file, cb) {
    cb(null, Date.now() + '-' + file.originalname);
  }
});

const uploadImage = multer({
  storage: imageStorage,
  fileFilter: function (req, file, cb) {
    if (
      file.mimetype === 'image/jpeg' ||
      file.mimetype === 'image/png' ||
      file.mimetype === 'image/webp'
    ) {
      cb(null, true);
    } else {
      cb(new Error('Only image files allowed'));
    }
  }
});

/* committee pdf upload */
const uploadPath = path.join(__dirname, "public/uploads/committee");
const committeeStorage = multer.diskStorage({
  destination: (req, file, cb) => {
    cb(null, uploadPath);
  },
  filename: (req, file, cb) => {
    const uniqueName = getUniqueFileName(uploadPath, file.originalname);
    cb(null, uniqueName);
  }
});
const uploadCommittee = multer({
  storage: committeeStorage
});

const storage = multer.diskStorage({
  destination: function (req, file, cb) {
    cb(null, 'public/images/');
  },
  filename: function (req, file, cb) {
    cb(null, Date.now() + '-' + file.originalname);
  }
});

const upload = multer({ storage });

const pdfStorage = multer.diskStorage({
  destination: (req, file, cb) => {
    cb(null, 'public/files');   // <-- IMPORTANT
  },
  filename: (req, file, cb) => {
    cb(null, file.originalname); // <-- NO TIMESTAMP
  }
});

const uploadPDF = multer({ storage: pdfStorage });

/* ================= SAFE HELPER (ADD THIS) ================= */
function safe(value) {
  return value === undefined || value === null ? '' : value;
}
/* ========================================================== */

/* ================= DIRECTORY HELPER ================= */
function ensureDirectoryExists(dirPath) {
  if (!fs.existsSync(dirPath)) {
    fs.mkdirSync(dirPath, { recursive: true });
  }
}
/* ===================================================== */

const recruitsUploadPath = path.join(__dirname, "public/uploads/recruits");
ensureDirectoryExists(recruitsUploadPath);
const recruitsStorage = multer.diskStorage({
  destination: (req, file, cb) => {
    cb(null, recruitsUploadPath);
  },
  filename: (req, file, cb) => {
    cb(null, Date.now() + "-" + file.originalname);
  }
});
const uploadRecruits = multer({ storage: recruitsStorage });

const digitalPaymentUploadPath = path.join(__dirname, "public/uploads/digital-payment");
ensureDirectoryExists(digitalPaymentUploadPath);
const digitalPaymentStorage = multer.diskStorage({
  destination: (req, file, cb) => {
    cb(null, digitalPaymentUploadPath);
  },
  filename: (req, file, cb) => {
    cb(null, Date.now() + "-" + file.originalname);
  }
});
const uploadDigitalPayment = multer({ storage: digitalPaymentStorage });

const dbConfig = {
  host: process.env.DB_HOST || 'localhost',
  user: process.env.DB_USER || 'demofilesco_thamirabharani',
  password: process.env.DB_PASSWORD || 'Demofilesco_thamirabharani@26',
  database: process.env.DB_NAME || 'demofilesco_thamirabharani',
  port: process.env.DB_PORT || 3306,
  waitForConnections: true,
  connectionLimit: 10,
  queueLimit: 0
};

app.use(async (req, res, next) => {
  try {
    const [menuCriteria] = await promisePool.execute(
      "SELECT * FROM qif_aqar23 ORDER BY id ASC"
    );

    res.locals.qif_menu = menuCriteria;  // <-- THIS makes it available in EJS
    next();
  } catch (error) {
    console.error("Error loading QIF menu:", error);
    res.locals.qif_menu = [];  // prevent crash
    next();
  }
});

const pool = mysql.createPool({
  host: "localhost",
  user: "demofilesco_thamirabharani",
  password: "Demofilesco_thamirabharani@26",
  database: "demofilesco_thamirabharani"
});
const promisePool = pool.promise();
app.use(async (req, res, next) => {
  try {
    const [rows] = await promisePool.execute(
      "SELECT pdf_file FROM aqar23_strategic_plan ORDER BY id DESC LIMIT 1"
    );
    res.locals.aqar23PDF = rows.length ? rows[0].pdf_file : null;
  } catch (e) {
    res.locals.aqar23PDF = null;
  }
  next();
});

promisePool.getConnection()
  .then(connection => {
    console.log('Connected to MySQL database successfully!');
    connection.release();
    initializeDatabase();
  })
  .catch(err => {
    console.error('Error connecting to MySQL database:', err.message);
    process.exit(1);
  });

async function initializeDatabase() {
  try {
    await promisePool.execute(`
      CREATE TABLE IF NOT EXISTS about_content (
        id INT PRIMARY KEY,
        subtitle VARCHAR(255),
        title VARCHAR(255),
        about_us_image VARCHAR(255),
        experience_title VARCHAR(255),
        experience_years VARCHAR(10),
        experience_subtitle VARCHAR(255),
        description TEXT,
        description_1 TEXT,
        phone VARCHAR(50),
        extra_content_1 TEXT,
        vision_title VARCHAR(255),
        vision_content TEXT,
        vision_image VARCHAR(255),
        quality_title VARCHAR(255),
        quality_content TEXT,
        updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP 
        ON UPDATE CURRENT_TIMESTAMP
      )
    `);

    await promisePool.execute(`
      CREATE TABLE IF NOT EXISTS vision_mission_quality (
        id INT AUTO_INCREMENT PRIMARY KEY,
        vision TEXT,
        mission TEXT,
        quality_policy TEXT,
        created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
        updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP
      )
    `);

    await promisePool.execute(`
      CREATE TABLE IF NOT EXISTS general_information (
        id INT AUTO_INCREMENT PRIMARY KEY,
        content TEXT,
        image VARCHAR(255),
        trust_head VARCHAR(255),
        trust_name VARCHAR(255),
        address TEXT,
        district VARCHAR(255),
        chairman_name VARCHAR(255),
        created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
        updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP
      )
    `);

    await promisePool.execute(`
      CREATE TABLE IF NOT EXISTS home_sliders (
        id INT AUTO_INCREMENT PRIMARY KEY,
        heading TEXT,
        subheading TEXT,
        cta_text VARCHAR(255),
        cta_link VARCHAR(255),
        slide_image VARCHAR(255),
        created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP
      )
    `);

    await promisePool.execute(`
      CREATE TABLE IF NOT EXISTS home_recentevent (
        id INT AUTO_INCREMENT PRIMARY KEY,
        event_title TEXT,
        event_date TIMESTAMP,
        event_description TEXT,
        created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP
      )
    `);

    await promisePool.execute(`
      CREATE TABLE IF NOT EXISTS home_banner (
        id INT AUTO_INCREMENT PRIMARY KEY,
        banner_image VARCHAR(255),
        created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
        updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP
      )
    `);

    await promisePool.execute(`
      CREATE TABLE IF NOT EXISTS chairman_content (
        id INT AUTO_INCREMENT PRIMARY KEY,
        name VARCHAR(255),
        designation VARCHAR(255),
        message TEXT,
        image VARCHAR(255),
        created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
        updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP
      )
    `);

    await promisePool.execute(`
      CREATE TABLE IF NOT EXISTS mandatory_disclosure (
        id INT AUTO_INCREMENT PRIMARY KEY,
        title VARCHAR(255),
        pdf_file VARCHAR(255),
        year VARCHAR(20),
        created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP
      )
    `);

    await promisePool.execute(`
      CREATE TABLE IF NOT EXISTS statutory_declaration (
        id INT AUTO_INCREMENT PRIMARY KEY,
        title VARCHAR(255),
        pdf_file VARCHAR(255),
        year VARCHAR(20),
        created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP
      )
    `);

    await promisePool.execute(`
      CREATE TABLE IF NOT EXISTS affiliation_recognition (
        id INT AUTO_INCREMENT PRIMARY KEY,
        title VARCHAR(255),
        pdf_file VARCHAR(255),
        year VARCHAR(20),
        created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP
      )
    `);

    await promisePool.execute(`
      CREATE TABLE IF NOT EXISTS aicte_approvals (
        id INT AUTO_INCREMENT PRIMARY KEY,
        title VARCHAR(255),
        pdf_file VARCHAR(255),
        year VARCHAR(20),
        created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP
      )
    `);

    await promisePool.execute(`
      CREATE TABLE IF NOT EXISTS hr_policy_main (
        id INT AUTO_INCREMENT PRIMARY KEY,
        college_name VARCHAR(255),
        college_address TEXT,
        main_title VARCHAR(255),
        sub_title VARCHAR(255),
        bottom_title VARCHAR(255),
        created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP
      )
    `);
    await promisePool.execute(`
      CREATE TABLE IF NOT EXISTS hr_policy_sections (
        id INT AUTO_INCREMENT PRIMARY KEY,
        section_type ENUM('about', 'middle_left', 'middle_right', 'bottom'),
        title VARCHAR(255),
        content TEXT,
        created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP
      )
    `);

    await promisePool.execute(`
      CREATE TABLE IF NOT EXISTS hr_policy_code (
        id INT AUTO_INCREMENT PRIMARY KEY,
        title VARCHAR(255),
        pdf_file VARCHAR(255),
        year VARCHAR(20),
        created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP
      )
    `);

    await promisePool.execute(`
      CREATE TABLE IF NOT EXISTS organogram (
        id INT AUTO_INCREMENT PRIMARY KEY,
        title VARCHAR(255),
        image_path VARCHAR(255),
        year VARCHAR(20),
        created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP
      )
    `);

    await promisePool.execute(`
      CREATE TABLE IF NOT EXISTS admission_info (
        id INT AUTO_INCREMENT PRIMARY KEY,
        title VARCHAR(255),
        content TEXT,
        created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP
      )
    `);
    await promisePool.execute(`
      CREATE TABLE IF NOT EXISTS admission_files (
        id INT AUTO_INCREMENT PRIMARY KEY,
        admission_id INT,
        title VARCHAR(255),
        pdf_file VARCHAR(255),
        year VARCHAR(20),
        created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
        FOREIGN KEY (admission_id) REFERENCES admission_info(id) ON DELETE CASCADE
      )
    `);

    await promisePool.execute(`
      CREATE TABLE IF NOT EXISTS academic_schedule (
        id INT AUTO_INCREMENT PRIMARY KEY,
        title VARCHAR(255),
        pdf_file VARCHAR(255),
        year VARCHAR(20),
        created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP
      )
    `);

    await promisePool.execute(`
      CREATE TABLE IF NOT EXISTS programs_ug (
        id INT AUTO_INCREMENT PRIMARY KEY,
        degree VARCHAR(255),
        slug VARCHAR(255),
        image_file VARCHAR(255),
        duration VARCHAR(50),
        intake VARCHAR(50),
        description TEXT,
        created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP
      )
    `);

    await promisePool.execute(`
      CREATE TABLE IF NOT EXISTS programs_pg (
        id INT AUTO_INCREMENT PRIMARY KEY,
        degree VARCHAR(255),
        slug VARCHAR(255),
        image_file VARCHAR(255),
        duration VARCHAR(50),
        intake VARCHAR(50),
        description TEXT,
        created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP
      )
    `);

    await promisePool.execute(`
      CREATE TABLE IF NOT EXISTS r_d_cell_content (
        id INT AUTO_INCREMENT PRIMARY KEY,
        vision TEXT,
        objectives TEXT,
        lab_details TEXT,
        thrust_areas TEXT,
        created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
        updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP
      )
    `);

    await promisePool.execute(`
      CREATE TABLE IF NOT EXISTS mou (
        id INT AUTO_INCREMENT PRIMARY KEY,
        category VARCHAR(100),   -- Industry / Educational Institution / etc
        name VARCHAR(255),
        year VARCHAR(10),
        duration VARCHAR(50),
        activities TEXT,
        participants VARCHAR(50),
        created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP
      )
    `);

    await promisePool.execute(`
      CREATE TABLE IF NOT EXISTS fdp (
        id INT AUTO_INCREMENT PRIMARY KEY,
        teacher_name VARCHAR(255),
        program_title TEXT,
        duration VARCHAR(100),
        photo VARCHAR(255),
        created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP
      )
    `);

    await promisePool.execute(`
      CREATE TABLE IF NOT EXISTS sponsored_conferences (
        id INT AUTO_INCREMENT PRIMARY KEY,
        year VARCHAR(20),
        teacher_name VARCHAR(255),
        conference_name TEXT,
        professional_body TEXT,
        amount_support VARCHAR(50),
        photo VARCHAR(255),
        created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP
      )
    `);

    await promisePool.execute(`
      CREATE TABLE IF NOT EXISTS coe_home (
        id INT AUTO_INCREMENT PRIMARY KEY,
        greeting_title VARCHAR(255),
        description TEXT,
        controllers JSON,
        office_members JSON,
        phone VARCHAR(50),
        landline VARCHAR(50),
        mail_id VARCHAR(255),
        created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP
      )
    `);

    await promisePool.execute(`
      CREATE TABLE IF NOT EXISTS timetable (
        id INT AUTO_INCREMENT PRIMARY KEY,
        section_title VARCHAR(255),
        sub_title VARCHAR(255),
        exam_items JSON,
        created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP
      )
    `);

    await promisePool.execute(`
      CREATE TABLE IF NOT EXISTS results (
        id INT AUTO_INCREMENT PRIMARY KEY,
        image_path VARCHAR(255),
        link_url VARCHAR(255),
        alt_text VARCHAR(255),
        created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP
      )
    `);

    await promisePool.execute(`
      CREATE TABLE IF NOT EXISTS revaluation (
        id INT AUTO_INCREMENT PRIMARY KEY,
        image_path VARCHAR(255),
        link_url VARCHAR(255),
        alt_text VARCHAR(255),
        created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP
      )
    `);

    await promisePool.execute(`
      CREATE TABLE IF NOT EXISTS forms (
        id INT AUTO_INCREMENT PRIMARY KEY,
        form_no INT,
        form_title VARCHAR(255),
        form_file VARCHAR(255),
        created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP
      )
    `);

    await promisePool.execute(`
      CREATE TABLE IF NOT EXISTS campus_facilities (
        id INT AUTO_INCREMENT PRIMARY KEY,
        section_title VARCHAR(255),
        content LONGTEXT,
        list_items JSON,
        created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP
      )
    `);

    await promisePool.execute(`
      CREATE TABLE IF NOT EXISTS library_images (
        id INT AUTO_INCREMENT PRIMARY KEY,
        image_path VARCHAR(255) NOT NULL
      );
    `);

    await promisePool.execute(`
      CREATE TABLE IF NOT EXISTS library_ug_books (
        id INT AUTO_INCREMENT PRIMARY KEY,
        department VARCHAR(255),
        no_of_titles INT,
        no_of_volumes INT
      );
    `);

    await promisePool.execute(`
      CREATE TABLE IF NOT EXISTS library_pg_books (
        id INT AUTO_INCREMENT PRIMARY KEY,
        department VARCHAR(255),
        no_of_titles INT,
        no_of_volumes INT
      );
    `);

    await promisePool.execute(`
      CREATE TABLE IF NOT EXISTS library_gratis (
        id INT AUTO_INCREMENT PRIMARY KEY,
        description VARCHAR(255),
        no_of_titles INT,
        no_of_volumes INT
      );
    `);

    await promisePool.execute(`
      CREATE TABLE IF NOT EXISTS library_journals (
        id INT AUTO_INCREMENT PRIMARY KEY,
        course_name VARCHAR(255),
        no_of_journals INT
      );
    `);

    await promisePool.execute(`
      CREATE TABLE IF NOT EXISTS library_journal_info (
        id INT AUTO_INCREMENT PRIMARY KEY,
        total_journals INT,
        database_name VARCHAR(255)
      );
    `);

    await promisePool.execute(`
      CREATE TABLE IF NOT EXISTS green_campus_policy (
        id INT AUTO_INCREMENT PRIMARY KEY,
        section_key VARCHAR(100),
        section_title VARCHAR(255),
        content LONGTEXT,
        order_index INT DEFAULT 0,
        created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP
      )
    `);

    await promisePool.execute(`
      CREATE TABLE IF NOT EXISTS regulations_syllabus (
        id INT AUTO_INCREMENT PRIMARY KEY,
        syllabus_year VARCHAR(20),
        pdf_title VARCHAR(255),
        pdf_file VARCHAR(255),
        created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP
      )
    `);

    await promisePool.execute(`
      CREATE TABLE IF NOT EXISTS qif_criteria (
        id INT AUTO_INCREMENT PRIMARY KEY,
        criterion_number VARCHAR(50),
        criterion_name VARCHAR(255),
        criterion_link VARCHAR(255),
        order_index INT DEFAULT 0,
        created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
        updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP
      )
    `);

    await promisePool.execute(`
      CREATE TABLE IF NOT EXISTS qif_aqar23 (
        id INT AUTO_INCREMENT PRIMARY KEY,
        criterion VARCHAR(50),
        name VARCHAR(255),
        page_link VARCHAR(255),
        created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
        updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP
      )
    `);

    await promisePool.execute(`
      CREATE TABLE IF NOT EXISTS qif_aqar23_criteria (
        id INT AUTO_INCREMENT PRIMARY KEY,
        criteria_number VARCHAR(50),
        criteria_name VARCHAR(255),
        download_link VARCHAR(255),
        created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
        updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP
      )
    `);

    const [bannerRows] = await promisePool.execute("SELECT COUNT(*) as count FROM home_banner");
    if (bannerRows[0].count === 0) {
      await promisePool.execute(`
        INSERT INTO home_banner (banner_image)
        VALUES (?)
      `, ['default-banner.jpg']); // replace with your actual image filename
      console.log('Default banner image inserted');
    }

    const [rows] = await promisePool.execute("SELECT id FROM about_content WHERE id = 1");
    if (rows.length === 0) {
      await promisePool.execute(`
        INSERT INTO about_content (id, subtitle, title)
        VALUES (1, 'About us', 'Thamirabharani Engineering College')
      `);
    }

    const [chairmanRows] = await promisePool.execute("SELECT COUNT(*) as count FROM chairman_content");
    if (chairmanRows[0].count === 0) {
      await promisePool.execute(`
        INSERT INTO chairman_content (name, designation, message, image)
        VALUES (?, ?, ?, ?)
      `, [
        'Mr. M. R. Paulraj',
        'Founder – Thamirabharani Engineering College',
        'As you know, Kalaiarasi Educational Trust was established to promote the noble cause of general, technical, vocational, and professional education in southern districts of Tamil Nadu. We believe that imparting quality education to this new generation is our greatest responsibility. Upholding our tradition of success, we have so far provided high-quality technical and managerial education to the young, empowering them to grow into a generation equipped with enhanced powers to be architects of modern India.',
        'chairman-desk.jpg'
      ]);
      console.log('Default chairman content inserted');
    }

    const [visionRows] = await promisePool.execute("SELECT COUNT(*) as count FROM vision_mission_quality");
    if (visionRows[0].count === 0) {
      await promisePool.execute(`
        INSERT INTO vision_mission_quality (vision, mission, quality_policy)
        VALUES (?, ?, ?)
      `, [
        'To emerge as one of the leading engineering colleges providing excellence in science, technology, and research.',
        'To impart value-based education through innovation and best practices for holistic development.',
        'We are committed to providing quality education that meets global standards through continuous improvement.'
      ]);
      console.log('Default vision-mission-quality content inserted');
    }

    const [genRows] = await promisePool.execute("SELECT COUNT(*) as count FROM general_information");
    if (genRows[0].count === 0) {
      await promisePool.execute(`
        INSERT INTO general_information (content, image, trust_head, trust_name, address, district, chairman_name)
        VALUES (?, ?, ?, ?, ?, ?, ?)
      `, [
        'Thamirabharani Engineering College is committed to providing quality education and holistic student development.',
        'campus.webp',
        'Chairman',
        'Kalaiarasi Educational Trust',
        '8/34, FIRST MAIN ROAD, SENTHIL NAGAR, ANNEX, PORUR, Chennai – 600116',
        'Chennai',
        'Mr. PAULRAJ M R'
      ]);
      console.log('Default general information inserted');
    }

    const [mdRows] = await promisePool.execute("SELECT COUNT(*) as count FROM mandatory_disclosure");
    if (mdRows[0].count === 0) {
      await promisePool.execute(`
        INSERT INTO mandatory_disclosure (title, pdf_file, year) VALUES
        ('Mandatory Disclosure 2024-2025', 'mandatory-disclosure-2024-2025.pdf', '2024-2025'),
        ('Mandatory Disclosure 2023-2024', 'mandatory-disclosure-2023-2024.pdf', '2023-2024'),
        ('Mandatory Disclosure 2022-2023', 'mandatory-disclosure-2022-2023.pdf', '2022-2023'),
        ('Mandatory Disclosure 2021-2022', 'mandatory-disclosure-2021-2022.pdf', '2021-2022')
      `);
      console.log('✅ Default data inserted into mandatory_disclosure');
    }

    const [sdRows] = await promisePool.execute("SELECT COUNT(*) as count FROM statutory_declaration");
    if (sdRows[0].count === 0) {
      await promisePool.execute(`
        INSERT INTO statutory_declaration (title, pdf_file) VALUES
        ('Statutory Declaration', 'statutory-declaration.pdf')
      `);
      console.log('✅ Default data inserted into statutory_declaration');
    }

    const [arRows] = await promisePool.execute("SELECT COUNT(*) as count FROM affiliation_recognition");
    if (arRows[0].count === 0) {
      await promisePool.execute(`
        INSERT INTO affiliation_recognition (title, pdf_file) VALUES
        ('Affiliation Recognition', 'affiliation-recognition.pdf')
      `);
      console.log('✅ Default data inserted into affiliation_recognition');
    }

    const [aaRows] = await promisePool.execute("SELECT COUNT(*) as count FROM aicte_approvals");
    if (aaRows[0].count === 0) {
      await promisePool.execute(`
        INSERT INTO aicte_approvals (title, pdf_file, year) VALUES
        ('AICTE Approval 2024-2025', 'aicte-approval-2024-2025.pdf', '2024-2025')
      `);
      console.log('✅ Default data inserted into aicte_approvals');
    }

    const [hpRows] = await promisePool.execute("SELECT COUNT(*) as count FROM hr_policy_code");
    if (hpRows[0].count === 0) {
      await promisePool.execute(`
        INSERT INTO hr_policy_code (title, pdf_file, year) VALUES
        ('HR Policy Code of Conduct 2024-2025', 'hr-policy-code-2024-2025.pdf', '2024-2025'),
        ('HR Policy Code of Conduct 2023-2024', 'hr-policy-code-2023-2024.pdf', '2023-2024')
      `);
      console.log('✅ Default data inserted into hr_policy_code');
    }

    const [ogRows] = await promisePool.execute("SELECT COUNT(*) as count FROM organogram");
    if (ogRows[0].count === 0) {
      await promisePool.execute(`
        INSERT INTO organogram (title, image_path, year)
        VALUES ('Institution Organogram', 'images/organogram.jpg', '2024-2025')
      `);
    }

    const [hrMainRows] = await promisePool.execute("SELECT COUNT(*) as count FROM hr_policy_main");
    if (hrMainRows[0].count === 0) {
      await promisePool.execute(`
        INSERT INTO hr_policy_main (college_name, college_address, main_title, sub_title, bottom_title)
        VALUES (
          'THAMIRABHARANI ENGINEERING COLLEGE',
          'Approved by AICTE, New Delhi & Affiliated to Anna University, Chennai<br>Chidambaranagar, Tirunelveli – Vagaikulam Road',
          'Governance of the Institution',
          'ROLES & RESPONSIBILITIES OF MANAGEMENT',
          'Governance of the Institution'
        )
      `);
      console.log('✅ Default data inserted into hr_policy_main');
    }
    const [hrSecRows] = await promisePool.execute("SELECT COUNT(*) as count FROM hr_policy_sections");
    if (hrSecRows[0].count === 0) {
      await promisePool.execute(`
        INSERT INTO hr_policy_sections (section_type, title, content) VALUES
        ('about', 'Name and Address of the Institution', 'Thamirabharani Engineering College, Tirunelveli – Vagaikulam Road'),
        ('about', 'Details of all the programs being offered by the Institute', 'B.E - Civil, Mechanical, CSE, EEE, ECE, AI & DS, MBA'),

        ('middle_left', 'Visionary', 'Visionary content goes here...'),
        ('middle_left', 'Regulations', 'Regulations and policies content...'),
        ('middle_left', 'Leadership', 'Leadership roles and structure...'),

        ('middle_right', 'Roles & Responsibilities of Librarian', 'Details of librarian roles...'),
        ('middle_right', 'Roles & Responsibilities of Administrative Officer', 'Administrative officer duties...'),
        ('middle_right', 'Roles & Responsibilities of Lab Technicians', 'Lab technicians code and duties...'),

        ('bottom', 'Scope', 'Scope of Governance within the institution...'),
        ('bottom', 'Authority', 'Authority and decision-making process...'),
        ('bottom', 'Monitoring', 'Monitoring and evaluation system details...')
      `);
      console.log('✅ Default data inserted into hr_policy_sections');
    }

    const [adRows] = await promisePool.execute("SELECT COUNT(*) AS count FROM admission_info");
    if (adRows[0].count === 0) {
      const [res] = await promisePool.execute(
        `INSERT INTO admission_info (title, content) VALUES (?, ?)`,
        [
          'Admission',
          'Admission to Thamirabharani Engineering College happens through counselling and merit-based selections. Please check the listed documents below for application forms, prospectus and admission schedule.'
        ]
      );
      const admissionId = res.insertId;
      await promisePool.execute(`
        INSERT INTO admission_files (admission_id, title, pdf_file, year) VALUES
        (?, 'Admission Prospectus 2024-25', 'admission-prospectus-2024-25.pdf', '2024-2025'),
        (?, 'Admission Form 2024', 'admission-form-2024.pdf', '2024')
      `, [admissionId, admissionId]);
      console.log('✅ Default admission data inserted');
    }

    const [schRows] = await promisePool.execute("SELECT COUNT(*) AS count FROM academic_schedule");
    if (schRows[0].count === 0) {
      await promisePool.execute(`
        INSERT INTO academic_schedule (title, pdf_file, year) VALUES
        ('Academic Schedule 2024-25', 'academic-schedule-2024-25.pdf', '2024-2025'),
        ('Academic Calendar 2023-24', 'academic-calendar-2023-24.pdf', '2023-2024')
      `);
      console.log('✅ Default academic_schedule data inserted');
    }

    const [ugRows] = await promisePool.execute("SELECT COUNT(*) AS count FROM programs_ug");
    if (ugRows[0].count === 0) {
      await promisePool.execute(`
        INSERT INTO programs_ug (degree, slug, image_file, duration, intake, description) VALUES
        ('B.E. Civil Engineering', 'civil-engineering', 'civil.webp', '4 Years', '60', 'Civil Engineering program with strong lab infrastructure'),
        ('B.E. Mechanical Engineering', 'mechanical-engineering', 'mechanical.webp', '4 Years', '120', 'Mechanical Engineering with advanced manufacturing labs'),
        ('B.E. Electrical and Electronics Engineering', 'electrical-electronics-engineering', 'eee.webp', '4 Years', '60', 'Core electrical and electronics concepts with industrial applications'),
        ('B.E. Electronics and Communication Engineering', 'electronics-communication-engineering', 'ece.webp', '4 Years', '120', 'Electronics and Communication with hands-on circuit design'),
        ('B.E. Computer Science and Engineering', 'computer-science-engineering', 'cse.webp', '4 Years', '120', 'Computer Science program covering software, AI, and IoT'),
        ('B.E. Computer Science and Engineering (Cyber Security)', 'cyber-security', 'cyber.webp', '4 Years', '60', 'Specialization in Cyber Security'),
        ('B.Tech. Artificial Intelligence and Data Science', 'ai-data-science', 'ai.webp', '4 Years', '60', 'AI and Data Science program with real-world projects'),
        ('B.Tech. Information Technology', 'information-technology', 'it.webp', '4 Years', '60', 'Information Technology program focused on full-stack development')
      `);
      console.log('✅ Default UG programs inserted');
    }

    const [pgRows] = await promisePool.execute("SELECT COUNT(*) AS count FROM programs_pg");
    if (pgRows[0].count === 0) {
      await promisePool.execute(`
        INSERT INTO programs_pg (degree, slug, image_file, duration, intake, description) VALUES
        ('MBA - Master of Business Administration', 'mba', 'mba.webp', '2 Years', '60', 'Business Administration with HR, Marketing, and Finance specializations')
      `);
      console.log('✅ Default PG program inserted');
    }

    const [rdPageRows] = await promisePool.execute("SELECT COUNT(*) AS count FROM r_d_cell_content");
    if (rdPageRows[0].count === 0) {
      await promisePool.execute(
        `INSERT INTO r_d_cell_content (vision, objectives, lab_details, thrust_areas)
         VALUES (?, ?, ?, ?)`,
        [
          `“To be a Centre of Excellence in Research & Development by promoting research in emerging frontier areas of Science and Engineering”`,

          `<ul>
            <li>To create and promote a culture of research among the faculty and students of TEC.</li>
            <li>To ensure quality, integrity and ethics in research.</li>
            <li>To encourage faculty and students to publish papers in reputed journals.</li>
            <li>To motivate participation in National/International conferences.</li>
            <li>To organize seminars, workshops, and training programs with financial assistance.</li>
            <li>To establish Research Centres within TEC and undertake consultancy activities.</li>
            <li>To get funded research projects from DRDO, DST, AICTE, UGC, etc.</li>
          </ul>`,

          `<ul>
            <li>Grid connected 1KWp solar PV system</li>
            <li>Micro Wind energy generator</li>
            <li>100W Fuel cell with hydrogen cylinder</li>
            <li>Solar PV Emulator</li>
            <li>DSP and Embedded Trainer modules</li>
            <li>PC with MATLAB software</li>
          </ul>`,

          `<ul>
            <li>Modelling and simulation of energy systems</li>
            <li>Condition monitoring and fault diagnosis in energy generation systems</li>
            <li>Real-time signal processing and data analysis</li>
          </ul>`
        ]
      );
      console.log("✅ Default R&D Cell content inserted.");
    }

    const [mouRows] = await promisePool.execute("SELECT COUNT(*) AS count FROM mou");
    if (mouRows[0].count === 0) {
      await promisePool.execute(`
        INSERT INTO mou (category, name, year, duration, activities, participants)
        VALUES 
        ('Industry', 'Bryan Softech Solution Pvt Ltd', '2021', '5 Years', 'Internship, Skill Development', 'Nil'),
        ('Industry', 'E-dot Technologies', '2021', '5 Years', 'Internship, Skill Development', 'Nil'),
        ('Industry', 'FACEAT', '2019', '5 Years', 'Internship, Skill Development', 'Nil'),
        ('Industry', 'GAYRA Constructions', '2019', '5 Years', 'Internship, Skill Development', 'Nil'),
        ('Industry', 'Mega Tech Scientific Instruments', '2021', '5 Years', 'Internship, Skill Development', 'Nil'),
        ('Industry', 'Om Namachivayam Builders', '2021', '5 Years', 'Internship, Research, Skill Development', 'Nil'),
        ('Industry', 'Prime Tech Instruments', '2021', '5 Years', 'Internship, Skill Development', 'Nil'),
        ('Industry', 'VEI Technologies', '2021', '5 Years', 'Internship, Project execution, Research guidance, FDP, Conference & Workshop, Special Lecturer, R&D, Training & Placement', 'Nil'),
        ('Industry', 'Velsine Power Solutions', '2021', '5 Years', 'Internship, Skill Development', 'Nil'),
        ('Industry', 'Alfaj Code Breeze', '2023', '5 Years', 'Internship, Addon Courses, Industry Training Program', 'Nil'),
        ('Industry', 'Iconix Software Solutions', '2019', '5 Years', 'Internship, Skill Development', '47'),
        ('Industry', 'Venus Home Appliances', '2022', '5 Years', 'Industrial visit, Research and development, Guest Lecturer', '6'),
        ('Industry', 'Seshsayee paper & Boards Ltd, Tirunelveli', '2022', '5 Years', 'Internship, Project work, Guest lecture, Industry visit', 'Nil'),
        ('Industry', 'WiseTech source Pvt Ltd', '2021', '5 Years', 'Workshops, Short term internships', 'Nil'),
        ('Industry', 'New Technology Institution', '2024', '5 Years', 'Industrial training, Internship, Research & Development, Guest Lecture, FDP', 'Nil'),
        ('Educational Institution', 'SNS College of Technology', '2024', '3 Years', 'Teacher training, Workshops, Guest Lecture', 'Nil'),
        ('Industry', 'Maxvy Technologies Pvt Ltd', '2024', '5 Years', 'Industrial Training, Internship, Research & Development, Guest Lecture, FDP', '17')
      `);
      console.log("✅ Default MoU data inserted.");
    }

    const [fdpRows] = await promisePool.execute("SELECT COUNT(*) AS count FROM fdp");
    if (fdpRows[0].count === 0) {
      await promisePool.execute(`
        INSERT INTO fdp (teacher_name, program_title, duration, photo)
        VALUES
        ('Mrs.R.Ambika', 'Technology based Entrepreneurship Development Programme on Biotechnology', '07.08.23 to 01.09.23', 'images/img-women.jpg'),
        ('Mrs.R.Ambika', 'Energy Efficient Buildings', '29.01.24 to 03.02.24', 'images/img-women.jpg'),
        ('Mr.S. Raja Muniasamy', 'Innovative teaching and learning pedagogy', '26.06.23 to 30.06.23', 'images/img-men.jpg'),
        ('Mr.S. Raja Muniasamy', 'Construction Safety', '28.06.24 to 03.07.24', 'images/img-men.jpg'),
        ('Mrs.S. Petchiammal', 'Internet of Things', '10.08.24', 'images/img-women.jpg'),
        ('Mrs.S. Petchiammal', 'Empowering educators to promote innovation and skillset', '23.05.24 to 28.05.24', 'images/img-women.jpg'),
        ('Mrs.R.Ambika', 'Recent Advances in the Power Sector Challenges, Solutions , and Opportunities', '31.07.23 to 05.08.23', 'images/img-women.jpg'),
        ('Mrs.R.Ambika', 'Concrete Building System Design', '24.07.23 to 28.07.23', 'images/img-women.jpg'),
        ('Dr.P.Revathy', 'AI Evolution: From Foundations to Generative AI', '13.05.24 to 17.05.24', 'images/img-women.jpg'),
        ('Mrs.S.Ponsundari', 'Harnessing Artificial Intelligence for Healthcare Excellence: Innovations, Insights and Impacts', '24.06.24 to 29.06.24', 'images/img-women.jpg'),
        ('Mrs.S.Ponsundari', 'Data Science', '26.02.24', 'images/img-women.jpg'),
        ('Mr.R.Jeeva', 'Devops and AWS', '04.07.23 to 08.07.23', 'images/img-men.jpg'),
        ('Mr.D.G.David', 'Intellectual Property Rights', '05.07.23 to 06.07.23', 'images/img-men.jpg'),
        ('Mr.D.G.David', 'Cloud Infrastructure (AWS)', '21.08.23 to 25.08.23', 'images/img-men.jpg'),
        ('Mr.D.G.David', 'Enterprise Design Thinking Practitioner', '22.02.24', 'images/img-men.jpg'),
        ('Mrs.S.Ponsundari', 'Cloud Infrastructure (AWS)', '21.08.23 to 25.08.23', 'images/img-women.jpg'),
        ('Mr.R.Ramar Kalangiyam', 'Modern VLSI Design and EDA Tools Hands-on', '02.11.23 to 08.11.23', 'images/img-men.jpg'),
        ('Mr.R.Ramar Kalangiyam', 'NPTEL E-Awareness workshop', '23.06.23', 'images/img-men.jpg'),
        ('Mr.R.Ramar Kalangiyam', '5G Networks and beyond', '22.08.23 to 26.08.23', 'images/img-men.jpg'),
        ('Mr.R.Ramar Kalangiyam', 'VLSI & Chip Design', '10.07.23 to 14.07.23', 'images/img-men.jpg'),
        ('Mr.R.Ramar Kalangiyam', 'VLSI Design - Trends & Practices', '26.06.23 to 28.06.23', 'images/img-men.jpg'),
        ('Dr.S.Selva Kumar', 'Recent Developments in Electrical and Electronics Engineering: A Research Prospective', '24.07.23 to 28.07.23', 'images/img-men.jpg'),
        ('Dr.S.Selva Kumar', 'Refresher Course in Engineering Physics', '17.07.23 to 28.07.23', 'images/img-men.jpg'),
        ('Dr.S.Selva Kumar', 'Electric Vehicle Technology', '09.10.23 to 13.10.23', 'images/img-men.jpg'),
        ('Dr.S.Selva Kumar', 'Advanced Materials for Energy Applications', '09.10.23 to 13.10.23', 'images/img-men.jpg'),
        ('Dr.S.Selva Kumar', 'Recent Advances, Trends and Challenges in Nanomaterials Characterization and Techniques (RCNMCT - 2023)', '25.09.23 to 29.09.23', 'images/img-men.jpg'),
        ('Dr.S.Selva Kumar', 'Environment Pollution: A threat to Life Below Water & World Environment Day Celebration Week', '05.06.23 to 11.06.23', 'images/img-men.jpg'),
        ('Dr.S.Selva Kumar', 'Environment Pollution: A threat to life in the Air', '15.08.23 to 21.08.23', 'images/img-men.jpg'),
        ('Dr.S.Selva Kumar', 'Technology based Entrepreneurship Development Programme on Biotechnology', '30.01.23 to 04.01.23', 'images/img-men.jpg')
      `);
      console.log("✅ Default FDP data inserted.");
    }

    const [confRows] = await promisePool.execute("SELECT COUNT(*) AS count FROM sponsored_conferences");
    if (confRows[0].count === 0) {
      await promisePool.execute(`
        INSERT INTO sponsored_conferences (year, teacher_name, conference_name, professional_body, amount_support, photo)
        VALUES
        ('2023-2024', 'Ms.S.Petchiammal', '8th International Conference on Engineering Technology and Science (ICETS''24)', '', '1500', 'images/img-women.jpg'),
        ('2023-2024', 'Mrs.R.Ambika', '8th International Conference on Engineering Technology and Science (ICETS''24)', '', '1500', 'images/img-women.jpg'),
        ('2023-2024', 'Ms.S.Petchiammal', '', 'IAENG International Association of Engineers', '300', 'images/img-women.jpg'),
        ('2023-2024', 'Dr.S.Selvakumar', 'AEF, AMR, KEM', '', '1500', 'images/img-men.jpg'),
        ('2023-2024', 'Dr.S.Selvakumar', 'American Journal of Applied Scientific Research', '', '500', 'images/img-men.jpg'),
        ('2023-2024', 'Dr.P.Revathy', 'Gradiva Review Journal', '', '2000', 'images/img-women.jpg'),
        ('2023-2024', 'Dr.P.Revathy', 'International Conference on Innovations and Technological Development in Electronics, Computers and Communication (ITDECC-2024)', '', '1500', 'images/img-women.jpg'),
        ('2023-2024', 'Mr.R.Jeeva', 'International Conference on Advancements in Science, Engineering and Management (ICASEM) 2024', '', '1500', 'images/img-men.jpg'),
        ('2023-2024', 'Dr.P.Revathy', 'International Journal of Intelligent Systems and Applications in Engineering', '', '2000', 'images/img-women.jpg'),
        ('2023-2024', 'Mr.R.Jeeva', 'International Journal of Engineering Research & Technology (IJERT)', '', '2000', 'images/img-men.jpg'),
        ('2023-2024', 'Mr.L.Shunmugaraj', 'NPTEL', '', '1000', 'images/img-men.jpg'),
        ('2023-2024', 'Mr.R.Ramar Kalangiyam', 'NPTEL', '', '1000', 'images/img-men.jpg'),
        ('2023-2024', 'Dr.S.Selvakumar', 'International Conference on Recent Advancements in Materials Science and Nuclear Energy Research (ICRAMNR - 2024)', '', '1500', 'images/img-men.jpg'),
        ('2023-2024', 'Dr.S.Selvakumar', 'Advances in Applied Sciences', '', '500', 'images/img-men.jpg'),
        ('2023-2024', 'Dr.S.Selvakumar', 'Makara Journal of Science', '', '2000', 'images/img-men.jpg'),
        ('2023-2024', 'Dr.S.Selvakumar', 'Journal of Electronic Materials', '', '2000', 'images/img-men.jpg')
      `);
      console.log("✅ Default Sponsored Conferences data inserted.");
    }

    const [coeHomeRows] = await promisePool.execute("SELECT COUNT(*) AS count FROM coe_home");
    if (coeHomeRows[0].count === 0) {
      await promisePool.execute(`
        INSERT INTO coe_home 
        (greeting_title, description, controllers, office_members, phone, landline, mail_id)
        VALUES (?, ?, ?, ?, ?, ?, ?)
      `, [
        'Greetings from the Controller of Examinations’ Office.',
        'An essential component of autonomy is the Controller of Examinations office. The COE Office is in charge of announcing the academic calendar, preparing questions for exams, administering tests, evaluating answer scripts, publishing the results, and providing grade reports. Students take end-of-semester exams in November and April of each academic year. Additionally, the Controller of Examination makes sure that the Internal assessment tests run smoothly.',
        JSON.stringify([
          { slno: 1, name: 'Mr. DAVID D G', designation: 'Controller of Examinations', contact: 'coe@tec-edu.in' },
          { slno: 2, name: 'Mrs. AMIRTHA JEEVA RUBY J', designation: 'Deputy Controller of Examinations', contact: 'dcoe@tec-edu.in' }
        ]),
        JSON.stringify([
          { slno: 1, name: 'Mrs. ROHINI N', designation: 'SYSTEM ANALYST' },
          { slno: 2, name: 'Mrs. DEEBA SUHANYA', designation: 'SYSTEM ANALYST' },
          { slno: 3, name: 'Mrs. MUTHU LAKSHMI U', designation: 'SYSTEM EXECUTIVE' },
          { slno: 4, name: 'Mr. SATHANKUMAR R', designation: 'OFFICE ASSISTANT' },
          { slno: 5, name: 'Ms. USHAMATHI P', designation: 'OFFICE ASSISTANT' }
        ]),
        '9361704669',
        '04622521369',
        'ocoe@tec-edu.in'
      ]);
      console.log('✅ Default COE Home data inserted');
    }

    const [resRows] = await promisePool.execute("SELECT COUNT(*) AS count FROM results");
    if (resRows[0].count === 0) {
      await promisePool.execute(`
        INSERT INTO results (image_path, link_url, alt_text)
        VALUES 
        ('images/ESE_RST_ND24_17.03.2025.webp', 'images/ESE_RST_ND24_17.03.2025.webp', 'End Semester Results'),
        ('images/ESE_RST_ND24_17.03.2025.webp', 'images/ESE_RST_ND24_17.03.2025.webp', 'End Semester Results 2')
      `);
      console.log('✅ Default Results data inserted');
    }

    const [revRows] = await promisePool.execute("SELECT COUNT(*) AS count FROM revaluation");
    if (revRows[0].count === 0) {
      await promisePool.execute(`
        INSERT INTO revaluation (image_path, link_url, alt_text)
        VALUES 
        ('images/ESC_CH_ND24_24.03.2025.webp', 'images/ESC_CH_ND24_24.03.2025.webp', 'Revaluation Notice 1'),
        ('images/ESE_RST_ND24_17.03.2025.webp', 'images/ESE_RST_ND24_17.03.2025.webp', 'Revaluation Notice 2')
      `);
      console.log('✅ Default Revaluation data inserted');
    }

    const [formsRows] = await promisePool.execute("SELECT COUNT(*) AS count FROM forms");
    if (formsRows[0].count === 0) {
      await promisePool.execute(`
        INSERT INTO forms (form_no, form_title, form_file) VALUES
        (1, 'Application for review', 'files/even-2020-21.pdf'),
        (2, 'Application form for Revaluation', 'files/even-2020-21.pdf'),
        (3, 'Application form for Photocopy', 'files/even-2020-21.pdf')
      `);
      console.log('✅ Default Forms data inserted');
    }

    const [facilitiesRows] = await promisePool.execute("SELECT COUNT(*) AS count FROM campus_facilities");
    if (facilitiesRows[0].count === 0) {
      await promisePool.execute(`
        INSERT INTO campus_facilities (section_title, content, list_items)
        VALUES (?, ?, ?)
      `, [
        'Learning by Doing',
        `
        <p>It is my privilege to warmly welcome you to this institute, which is dedicated to the cause of top-quality technical education.</p>
        <p>“Life is not a set of instructions, but is a series of experience and learning process”. This is where our role starts to make a difference and where learning is not just a series of instructions, but a fashion. To my mind an educational institute is not just about bricks, mortar and cement but about character, enriching minds, and the experience that lasts a lifetime.</p>
        <p>We offer B. E./B. Tech. Degree courses in seven branches of engineering:</p>
        <p>With this Mantra, our college seeks to impart Technical Education of high caliber to meet the growing needs of Engineers and Entrepreneurs in business and industry. The Thamirabharani Engineering College is sponsored by Kalaiarasi Educational Trust, a registered Charitable Trust. This college is mainly established for the betterment of the downtrodden people of Thatchanallur of Tirunelveli district.</p>
        <h3 class="wow fadeInUp">Library</h3>
        <p>The college has an automated library with More number of volumes of books on different disciplines, with separate section for the international and national periodicals, so as to provide effective learning resources to the students.</p>
        <h3 class="wow fadeInUp">Generator</h3>
        <p>Besides, a standby power generator 63 KVA is available in the campus to take care of the occasional power shut down due to maintenance.</p>
        <h3 class="wow fadeInUp">Facilities for the disabled students</h3>
        <p>The campus has ramp facilities to facilitate the differently abled to move by the wheel chair, without any difficulty. Special provision is facilitated for these students to write the university exam in the close proximity of the Exam Cell and scribe for the visually disabled students is also provided.</p>
        <h3 class="wow fadeInUp">Transport</h3>
        <p>The College provides accessible transport facility with buses plying to and fro connecting all remote corners of Tirunelveli, Tenkasi and Thoothukudi districts.</p>
        `,
        JSON.stringify([
          "Spacious rooms for instruction with proper ventilation are available to accommodate the students and they create a conductive environment for the most perfect learning activity.",
          "LCD Projector facilities are available in the class rooms.",
          "Separate hostel facilities are available for boys and girls. Tidy rooms and hygienic facilities make an ideal stay away from home ambiance.",
          "A mineral water plant is operated within the campus and has a well maintained RO facility. This water is used for cooking, drinking and is available throughout the campus."
        ])
      ]);
      console.log('✅ Default Campus Facilities data inserted');
    }

    const [dCount] = await promisePool.execute("SELECT COUNT(*) AS count FROM lab_departments");
    if (dCount[0].count === 0) {
      await promisePool.execute(
        "INSERT INTO lab_departments (department_name) VALUES (?), (?), (?)",
        ['CIVIL ENGINEERING LABORATORY','ELECTRONICS AND COMMUNICATION ENGINEERING LABORATORY','COMPUTER SCIENCE AND ENGINEERING LABORATORY']
      );
      console.log('Seeded lab_departments');
    }

    const [liCount] = await promisePool.execute("SELECT COUNT(*) AS count FROM library_images");
    if (liCount[0].count === 0) {
      await promisePool.execute("INSERT INTO library_images (image_path) VALUES (?), (?), (?)",
        ['images/library-1.jpg','images/library-2.jpg','images/library-3.jpg']);
      console.log('Seeded library_images');
    }

    const [ugCount] = await promisePool.execute("SELECT COUNT(*) AS count FROM library_ug_books");
    if (ugCount[0].count === 0) {
      await promisePool.execute(`
        INSERT INTO library_ug_books (department, no_of_titles, no_of_volumes) VALUES
        ('Civil', 488, 1882), ('ECE', 519, 2195), ('EEE', 477, 1972)
      `);
      console.log('Seeded library_ug_books');
    }

    const [pgCount] = await promisePool.execute("SELECT COUNT(*) AS count FROM library_pg_books");
    if (pgCount[0].count === 0) {
      await promisePool.execute("INSERT INTO library_pg_books (department, no_of_titles, no_of_volumes) VALUES (?, ?, ?)",
        ['MBA', 100, 500]);
      console.log('Seeded library_pg_books');
    }

    const [grCount] = await promisePool.execute("SELECT COUNT(*) AS count FROM library_gratis");
    if (grCount[0].count === 0) {
      await promisePool.execute("INSERT INTO library_gratis (description, no_of_titles, no_of_volumes) VALUES (?, ?, ?)",
        ['Competitive Exam Materials', 80, 143]);
      console.log('Seeded library_gratis');
    }

    const [journCount] = await promisePool.execute("SELECT COUNT(*) AS count FROM library_journals");
    if (journCount[0].count === 0) {
      await promisePool.execute(`
        INSERT INTO library_journals (course_name, no_of_journals) VALUES
        ('Civil Engineering', 10), ('Computer Science', 12)
      `);
      console.log('Seeded library_journals');
    }

    const [infoCount] = await promisePool.execute("SELECT COUNT(*) AS count FROM library_journal_info");
    if (infoCount[0].count === 0) {
      await promisePool.execute("INSERT INTO library_journal_info (total_journals, database_name) VALUES (?, ?)",
        [22, 'DELNET, IEEE, Springer']);
      console.log('Seeded library_journal_info');
    }

    // Mechanical Engineering (same structure as civil)
    await promisePool.execute(`
      CREATE TABLE IF NOT EXISTS mechanical_engineering (
        id INT PRIMARY KEY,
        section_key VARCHAR(100) UNIQUE,
        title VARCHAR(255),
        content LONGTEXT
      )
    `);
    await promisePool.execute(`
      CREATE TABLE IF NOT EXISTS mechanical_gallery (
        id INT AUTO_INCREMENT PRIMARY KEY,
        image_path VARCHAR(500)
      )
    `);
    await promisePool.execute(`
      CREATE TABLE IF NOT EXISTS mechanical_faculty_profile (
        id INT PRIMARY KEY,
        teacher_image VARCHAR(255),
        teacher_name VARCHAR(255),
        teacher_department VARCHAR(255)
      )
    `);

    // EEE (same structure as civil; table name "eee" only, not eee_engineering)
    await promisePool.execute(`
      CREATE TABLE IF NOT EXISTS eee (
        id INT PRIMARY KEY,
        section_key VARCHAR(100) UNIQUE,
        title VARCHAR(255),
        content LONGTEXT
      )
    `);
    await promisePool.execute(`
      CREATE TABLE IF NOT EXISTS eee_gallery (
        id INT AUTO_INCREMENT PRIMARY KEY,
        image_path VARCHAR(500)
      )
    `);
    await promisePool.execute(`
      CREATE TABLE IF NOT EXISTS eee_faculty_profile (
        id INT PRIMARY KEY,
        teacher_image VARCHAR(255),
        teacher_name VARCHAR(255),
        teacher_department VARCHAR(255)
      )
    `);
    console.log('EEE tables created: eee, eee_gallery, eee_faculty_profile');

    // EANDC (E and C Engineering - same structure as civil)
    await promisePool.execute(`
      CREATE TABLE IF NOT EXISTS eandc (
        id INT PRIMARY KEY,
        section_key VARCHAR(100) UNIQUE,
        title VARCHAR(255),
        content LONGTEXT
      )
    `);
    await promisePool.execute(`
      CREATE TABLE IF NOT EXISTS eandc_gallery (
        id INT AUTO_INCREMENT PRIMARY KEY,
        image_path VARCHAR(500)
      )
    `);
    await promisePool.execute(`
      CREATE TABLE IF NOT EXISTS eandc_faculty_profile (
        id INT PRIMARY KEY,
        teacher_image VARCHAR(255),
        teacher_name VARCHAR(255),
        teacher_department VARCHAR(255)
      )
    `);
    console.log('EANDC tables created: eandc, eandc_gallery, eandc_faculty_profile');

    // CSE (same structure as civil)
    await promisePool.execute(`
      CREATE TABLE IF NOT EXISTS cse (
        id INT PRIMARY KEY,
        section_key VARCHAR(100) UNIQUE,
        title VARCHAR(255),
        content LONGTEXT
      )
    `);
    await promisePool.execute(`
      CREATE TABLE IF NOT EXISTS cse_gallery (
        id INT AUTO_INCREMENT PRIMARY KEY,
        image_path VARCHAR(500)
      )
    `);
    await promisePool.execute(`
      CREATE TABLE IF NOT EXISTS cse_faculty_profile (
        id INT PRIMARY KEY,
        teacher_image VARCHAR(255),
        teacher_name VARCHAR(255),
        teacher_department VARCHAR(255)
      )
    `);
    console.log('CSE tables created: cse, cse_gallery, cse_faculty_profile');

    // CSE Cyber Security (same structure as civil)
    await promisePool.execute(`
      CREATE TABLE IF NOT EXISTS cse_cyber (
        id INT PRIMARY KEY,
        section_key VARCHAR(100) UNIQUE,
        title VARCHAR(255),
        content LONGTEXT
      )
    `);
    await promisePool.execute(`
      CREATE TABLE IF NOT EXISTS cse_cyber_gallery (
        id INT AUTO_INCREMENT PRIMARY KEY,
        image_path VARCHAR(500)
      )
    `);
    await promisePool.execute(`
      CREATE TABLE IF NOT EXISTS cse_cyber_faculty_profile (
        id INT PRIMARY KEY,
        teacher_image VARCHAR(255),
        teacher_name VARCHAR(255),
        teacher_department VARCHAR(255)
      )
    `);
    console.log('CSE Cyber Security tables created: cse_cyber, cse_cyber_gallery, cse_cyber_faculty_profile');

    // B.Tech AI & Data Science (same structure as civil)
    await promisePool.execute(`
      CREATE TABLE IF NOT EXISTS btech_ai (
        id INT PRIMARY KEY,
        section_key VARCHAR(100) UNIQUE,
        title VARCHAR(255),
        content LONGTEXT
      )
    `);
    await promisePool.execute(`
      CREATE TABLE IF NOT EXISTS btech_ai_gallery (
        id INT AUTO_INCREMENT PRIMARY KEY,
        image_path VARCHAR(500)
      )
    `);
    await promisePool.execute(`
      CREATE TABLE IF NOT EXISTS btech_ai_faculty_profile (
        id INT PRIMARY KEY,
        teacher_image VARCHAR(255),
        teacher_name VARCHAR(255),
        teacher_department VARCHAR(255)
      )
    `);
    console.log('B.Tech AI tables created: btech_ai, btech_ai_gallery, btech_ai_faculty_profile');

    // B.Tech IT (same structure as civil)
    await promisePool.execute(`
      CREATE TABLE IF NOT EXISTS btech_it (
        id INT PRIMARY KEY,
        section_key VARCHAR(100) UNIQUE,
        title VARCHAR(255),
        content LONGTEXT
      )
    `);
    await promisePool.execute(`
      CREATE TABLE IF NOT EXISTS btech_it_gallery (
        id INT AUTO_INCREMENT PRIMARY KEY,
        image_path VARCHAR(500)
      )
    `);
    await promisePool.execute(`
      CREATE TABLE IF NOT EXISTS btech_it_faculty_profile (
        id INT PRIMARY KEY,
        teacher_image VARCHAR(255),
        teacher_name VARCHAR(255),
        teacher_department VARCHAR(255)
      )
    `);
    console.log('B.Tech IT tables created: btech_it, btech_it_gallery, btech_it_faculty_profile');

    // MBA (same structure as civil)
    await promisePool.execute(`
      CREATE TABLE IF NOT EXISTS mba (
        id INT PRIMARY KEY,
        section_key VARCHAR(100) UNIQUE,
        title VARCHAR(255),
        content LONGTEXT
      )
    `);
    await promisePool.execute(`
      CREATE TABLE IF NOT EXISTS mba_gallery (
        id INT AUTO_INCREMENT PRIMARY KEY,
        image_path VARCHAR(500)
      )
    `);
    await promisePool.execute(`
      CREATE TABLE IF NOT EXISTS mba_faculty_profile (
        id INT PRIMARY KEY,
        teacher_image VARCHAR(255),
        teacher_name VARCHAR(255),
        teacher_department VARCHAR(255)
      )
    `);
    console.log('MBA tables created: mba, mba_gallery, mba_faculty_profile');

    console.log('Database initialization completed successfully!');
  } catch (error) {
    console.error('Error initializing database:', error);
  }
}

// =================== SESSION ===================
app.use(session({
  secret: 'thamirabharani_secret_key',
  resave: false,
  saveUninitialized: true
}));

const db = mysql.createConnection({
  host: process.env.DB_HOST || 'localhost',
  user: process.env.DB_USER || 'demofilesco_thamirabharani',
  password: process.env.DB_PASSWORD || 'Demofilesco_thamirabharani',
  database: process.env.DB_NAME || 'demofilesco_thamirabharani'
});

db.connect(err => {
  if (err) {
    console.error('❌ Database connection failed:', err);
  } else {
    console.log('✅ MySQL Connected to thamirabharani');
  }
});

app.set('view engine', 'ejs');
app.set('views', path.join(__dirname, 'views'));

app.use(express.static(path.join(__dirname, 'public')));
app.use(bodyParser.urlencoded({ extended: true }));
app.use(bodyParser.json());

app.get('/admin/login', (req, res) => {
  res.render('admin/login', { error: null });
});

app.post('/admin/login', (req, res) => {
  const { email, password } = req.body;

  const sql = 'SELECT * FROM admins WHERE email = ? AND password = ?';
  db.query(sql, [email, password], (err, result) => {
    if (err) throw err;

    if (result.length > 0) {
      req.session.admin = result[0];
      res.redirect('/admin/dashboard');
    } else {
      res.render('admin/login', { error: 'Invalid Email or Password' });
    }
  });
});

app.get('/admin/dashboard', (req, res) => {
  if (!req.session.admin) return res.redirect('/admin/login');
  res.render('admin/dashboard', { title: 'Dashboard', admin: req.session.admin });
});

app.get('/admin/profile', (req, res) => {
  if (!req.session.admin) return res.redirect('/admin/login');
  res.render('admin/profile', { admin: req.session.admin, message: null });
});

app.get('/admin/logout', (req, res) => {
  req.session.destroy();
  res.redirect('/admin/login');
});

app.get('/admin', (req, res) => {
  res.redirect('/admin/login');
});

app.post('/admin/profile/update', (req, res) => {
  if (!req.session.admin) return res.redirect('/admin/login');

  const { name, email, password } = req.body;
  const adminId = req.session.admin.id;

  let sql, values;

  if (password && password.trim() !== '') {
    // Update including password
    sql = 'UPDATE admins SET name = ?, email = ?, password = ? WHERE id = ?';
    values = [name, email, password, adminId];
  } else {
    // Update without changing password
    sql = 'UPDATE admins SET name = ?, email = ? WHERE id = ?';
    values = [name, email, adminId];
  }

  db.query(sql, values, (err) => {
    if (err) throw err;

    // Update session data
    req.session.admin.name = name;
    req.session.admin.email = email;
    if (password && password.trim() !== '') {
      req.session.admin.password = password;
    }

    res.render('admin/profile', { admin: req.session.admin, message: 'Profile updated successfully!' });
  });
});

app.get('/', async (req, res) => {
  try {
    const [banners] = await promisePool.execute('SELECT * FROM home_banner ORDER BY id ASC');
    const [recentevent] = await promisePool.execute('SELECT * FROM home_recentevent ORDER BY event_date DESC');
    const [recentannouncement] = await promisePool.execute('SELECT * FROM home_recentannouncement ORDER BY announce_date DESC');
    const [promobanner] = await promisePool.execute('SELECT * FROM home_promo_banner ORDER BY id DESC LIMIT 1');
    const [campussection] = await promisePool.execute('SELECT * FROM home_campus_section ORDER BY id ASC');
    const [courseoffered] = await promisePool.execute('SELECT * FROM home_course_offered ORDER BY id ASC');
    const [rows] = await promisePool.execute("SELECT * FROM home_placement_section ORDER BY id DESC LIMIT 1");
    const placementSection = rows.length ? rows[0] : null;
    const [recruitLogos] = await promisePool.execute("SELECT * FROM recruits_logos ORDER BY id DESC");
    const [counters] = await promisePool.execute('SELECT * FROM home_counters ORDER BY id ASC');
    const [sports] = await promisePool.execute('SELECT * FROM home_sports ORDER BY id ASC');

    const [whyusTabs] = await promisePool.execute("SELECT * FROM home_whyus_tabs ORDER BY id ASC");
    const [whyusBoxes] = await promisePool.execute("SELECT * FROM home_whyus_boxes ORDER BY display_order ASC");

    const [lifeattec] = await promisePool.execute('SELECT * FROM home_lifeattec ORDER BY id DESC LIMIT 1');
    const [testimonials] = await promisePool.execute('SELECT * FROM home_testimonials ORDER BY id DESC');

    res.render('index', {
      sliders: banners,
      recentevent,
      recentannouncement,
      promobanner: promobanner?.[0] || null,
      campussection,
      courseoffered,
      placementSection,
      recruitLogos,
      counters,
      sports,
      whyusTabs,
      whyusBoxes,
      lifeattec: lifeattec?.[0] || null,
      testimonials
    });

  } catch (error) {
    console.error("🔥 Error loading home page:", error);
    res.status(500).send("Error loading home page");
  }
});

app.get('/admin/home-banner', async (req, res) => {
  const [banners] = await promisePool.execute('SELECT * FROM home_banner ORDER BY id DESC');
  res.render('admin/home-banner', { banners });
});
app.post('/admin/home-banner/add', upload.single('banner_image'), async (req, res) => {
  const { banner_title, banner_description, banner_button_text, banner_button_link } = req.body;
  const image = req.file ? req.file.filename : null;
  await promisePool.execute(
    'INSERT INTO home_banner (banner_image, banner_title, banner_description, banner_button_text, banner_button_link) VALUES (?, ?, ?, ?, ?)',
    [image, banner_title, banner_description, banner_button_text, banner_button_link]
  );
  res.redirect('/admin/home-banner');
});
app.get('/admin/home-banner/edit/:id', async (req, res) => {
  const [rows] = await promisePool.execute('SELECT * FROM home_banner WHERE id = ?', [req.params.id]);
  res.render('admin/home-banner-edit', { banner: rows[0] });
});
app.post('/admin/home-banner/update/:id', upload.single('banner_image'), async (req, res) => {
  const { banner_title, banner_description, banner_button_text, banner_button_link, old_image } = req.body;
  const image = req.file ? req.file.filename : old_image;
  await promisePool.execute(
    `UPDATE home_banner 
     SET banner_image=?, banner_title=?, banner_description=?, banner_button_text=?, banner_button_link=? 
     WHERE id=?`,
    [image, banner_title, banner_description, banner_button_text, banner_button_link, req.params.id]
  );
  res.redirect('/admin/home-banner');
});
app.get('/admin/home-banner/delete/:id', async (req, res) => {
  await promisePool.execute('DELETE FROM home_banner WHERE id=?', [req.params.id]);
  res.redirect('/admin/home-banner');
});

app.get('/admin/home-recentevent', async (req, res) => {
  const [events] = await promisePool.execute('SELECT * FROM home_recentevent ORDER BY event_date DESC');
  res.render('admin/home-recentevent', { events });
});
app.post('/admin/home-recentevent/add', upload.single('event_image'), async (req, res) => {
  const { event_title, event_description, event_date } = req.body;
  const image = req.file ? req.file.filename : null;
  await promisePool.execute(
    'INSERT INTO home_recentevent (event_title, event_description, event_date, event_image) VALUES (?, ?, ?, ?)',
    [event_title, event_description, event_date, image]
  );
  res.redirect('/admin/home-recentevent');
});
app.get('/admin/home-recentevent/edit/:id', async (req, res) => {
  const [rows] = await promisePool.execute('SELECT * FROM home_recentevent WHERE id=?', [req.params.id]);
  res.render('admin/home-recentevent-edit', { event: rows[0] });
});
app.post('/admin/home-recentevent/update/:id', upload.single('event_image'), async (req, res) => {
  const { event_title, event_description, event_date, old_image } = req.body;
  const image = req.file ? req.file.filename : old_image;
  await promisePool.execute(
    'UPDATE home_recentevent SET event_title=?, event_description=?, event_date=?, event_image=? WHERE id=?',
    [event_title, event_description, event_date, image, req.params.id]
  );
  res.redirect('/admin/home-recentevent');
});
app.get('/admin/home-recentevent/delete/:id', async (req, res) => {
  await promisePool.execute('DELETE FROM home_recentevent WHERE id=?', [req.params.id]);
  res.redirect('/admin/home-recentevent');
});

app.get('/admin/home-recentannouncement', async (req, res) => {
  try {
    const [announcements] = await promisePool.execute(
      'SELECT * FROM home_recentannouncement ORDER BY announce_date DESC'
    );
    res.render('admin/home-recentannouncement', { announcements });
  } catch (err) {
    console.error(err);
    res.render('admin/home-recentannouncement', { announcements: [] });
  }
});
app.post('/admin/home-recentannouncement/add', async (req, res) => {
  const { announce_title, announce_description, announce_date, announce_location } = req.body;
  await promisePool.execute(
    'INSERT INTO home_recentannouncement (announce_title, announce_description, announce_date, announce_location) VALUES (?, ?, ?, ?)',
    [announce_title, announce_description, announce_date, announce_location]
  );
  res.redirect('/admin/home-recentannouncement');
});
app.get('/admin/home-recentannouncement/edit/:id', async (req, res) => {
  const [rows] = await promisePool.execute('SELECT * FROM home_recentannouncement WHERE id=?', [req.params.id]);
  if (rows.length === 0) {
    return res.send('Announcement not found');
  }
  res.render('admin/home-recentannouncement-edit', { announce: rows[0] });
});
app.post('/admin/home-recentannouncement/update/:id', async (req, res) => {
  const { announce_title, announce_description, announce_date, announce_location } = req.body;
  await promisePool.execute(
    'UPDATE home_recentannouncement SET announce_title=?, announce_description=?, announce_date=?, announce_location=? WHERE id=?',
    [announce_title, announce_description, announce_date, announce_location, req.params.id]
  );
  res.redirect('/admin/home-recentannouncement');
});
app.get('/admin/home-recentannouncement/delete/:id', async (req, res) => {
  await promisePool.execute('DELETE FROM home_recentannouncement WHERE id=?', [req.params.id]);
  res.redirect('/admin/home-recentannouncement');
});

app.get('/admin/home-promobanner', async (req, res) => {
  const [banners] = await promisePool.execute('SELECT * FROM home_promo_banner ORDER BY id DESC');
  res.render('admin/home-promobanner', { banners });
});
app.post('/admin/home-promobanner/add', upload.single('banner_image'), async (req, res) => {
  const { banner_link } = req.body;
  const image = req.file ? req.file.filename : null;
  await promisePool.execute('INSERT INTO home_promo_banner (banner_image, banner_link) VALUES (?, ?)', [image, banner_link]);
  res.redirect('/admin/home-promobanner');
});
app.get('/admin/home-promobanner/edit/:id', async (req, res) => {
  const [rows] = await promisePool.execute('SELECT * FROM home_promo_banner WHERE id=?', [req.params.id]);
  res.render('admin/home-promobanner-edit', { banner: rows[0] });
});
app.post('/admin/home-promobanner/update/:id', upload.single('banner_image'), async (req, res) => {
  const { banner_link, old_image } = req.body;
  const image = req.file ? req.file.filename : old_image;
  await promisePool.execute('UPDATE home_promo_banner SET banner_image=?, banner_link=? WHERE id=?', [image, banner_link, req.params.id]);
  res.redirect('/admin/home-promobanner');
});
app.get('/admin/home-promobanner/delete/:id', async (req, res) => {
  await promisePool.execute('DELETE FROM home_promo_banner WHERE id=?', [req.params.id]);
  res.redirect('/admin/home-promobanner');
});

app.get('/admin/home-campussection', async (req, res) => {
  const [campus] = await promisePool.execute('SELECT * FROM home_campus_section ORDER BY id ASC');
  res.render('admin/home-campussection', { campus });
});
app.post('/admin/home-campussection/add', upload.single('campus_image'), async (req, res) => {
  const { campus_title, campus_link } = req.body;
  const image = req.file ? req.file.filename : null;
  await promisePool.execute(
    'INSERT INTO home_campus_section (campus_title, campus_image, campus_link) VALUES (?, ?, ?)',
    [campus_title, image, campus_link]
  );
  res.redirect('/admin/home-campussection');
});
app.get('/admin/home-campussection/edit/:id', async (req, res) => {
  const [rows] = await promisePool.execute('SELECT * FROM home_campus_section WHERE id=?', [req.params.id]);
  res.render('admin/home-campussection-edit', { campus: rows[0] });
});
app.post('/admin/home-campussection/update/:id', upload.single('campus_image'), async (req, res) => {
  const { campus_title, campus_link, old_image } = req.body;
  const image = req.file ? req.file.filename : old_image;
  await promisePool.execute(
    'UPDATE home_campus_section SET campus_title=?, campus_image=?, campus_link=? WHERE id=?',
    [campus_title, image, campus_link, req.params.id]
  );
  res.redirect('/admin/home-campussection');
});
app.get('/admin/home-campussection/delete/:id', async (req, res) => {
  await promisePool.execute('DELETE FROM home_campus_section WHERE id=?', [req.params.id]);
  res.redirect('/admin/home-campussection');
});

app.get('/admin/home-courseoffered', async (req, res) => {
  const [courses] = await promisePool.execute('SELECT * FROM home_course_offered ORDER BY id DESC');
  res.render('admin/home-courseoffered', { courses });
});
app.post('/admin/home-courseoffered/add', async (req, res) => {
  const { course_category, course_name } = req.body;
  await promisePool.execute('INSERT INTO home_course_offered (course_category, course_name) VALUES (?, ?)', [course_category, course_name]);
  res.redirect('/admin/home-courseoffered');
});
app.get('/admin/home-courseoffered/edit/:id', async (req, res) => {
  const [rows] = await promisePool.execute('SELECT * FROM home_course_offered WHERE id=?', [req.params.id]);
  res.render('admin/home-courseoffered-edit', { course: rows[0] });
});
app.post('/admin/home-courseoffered/update/:id', async (req, res) => {
  const { course_category, course_name } = req.body;
  await promisePool.execute('UPDATE home_course_offered SET course_category=?, course_name=? WHERE id=?', [course_category, course_name, req.params.id]);
  res.redirect('/admin/home-courseoffered');
});
app.get('/admin/home-courseoffered/delete/:id', async (req, res) => {
  await promisePool.execute('DELETE FROM home_course_offered WHERE id=?', [req.params.id]);
  res.redirect('/admin/home-courseoffered');
});

app.get('/admin/home-placement-section', async (req, res) => {
  const [rows] = await promisePool.execute('SELECT * FROM home_placement_section ORDER BY id DESC');
  res.render('admin/home-placement-section', { sections: rows });
});
app.post('/admin/home-placement-section/add', upload.single('bg_image'), async (req, res) => {
  const { title, content } = req.body;
  const bg_image = req.file ? req.file.filename : null;

  await promisePool.execute(
    'INSERT INTO home_placement_section (title, content, bg_image) VALUES (?, ?, ?)',
    [title, content, bg_image]
  );

  res.redirect('/admin/home-placement-section');
});
app.get('/admin/home-placement-section/edit/:id', async (req, res) => {
  const [rows] = await promisePool.execute('SELECT * FROM home_placement_section WHERE id=?', [req.params.id]);
  res.render('admin/home-placement-section-edit', { section: rows[0] });
});
app.post('/admin/home-placement-section/update/:id', upload.single('bg_image'), async (req, res) => {
  const { title, content, old_image } = req.body;
  const bg_image = req.file ? req.file.filename : old_image;

  await promisePool.execute(
    'UPDATE home_placement_section SET title=?, content=?, bg_image=? WHERE id=?',
    [title, content, bg_image, req.params.id]
  );

  res.redirect('/admin/home-placement-section');
});
app.get('/admin/home-placement-section/delete/:id', async (req, res) => {
  await promisePool.execute('DELETE FROM home_placement_section WHERE id=?', [req.params.id]);
  res.redirect('/admin/home-placement-section');
});

app.get('/admin/home-counter', async (req, res) => {
  const [counters] = await promisePool.execute('SELECT * FROM home_counter ORDER BY id ASC');
  res.render('admin/home-counter', { counters });
});
app.post('/admin/home-counter/add', async (req, res) => {
  const { number, label } = req.body;
  await promisePool.execute('INSERT INTO home_counter (number, label) VALUES (?, ?)', [number, label]);
  res.redirect('/admin/home-counter');
});
app.get('/admin/home-counter/edit/:id', async (req, res) => {
  const [rows] = await promisePool.execute('SELECT * FROM home_counter WHERE id=?', [req.params.id]);
  res.render('admin/home-counter-edit', { counter: rows[0] });
});
app.post('/admin/home-counter/update/:id', async (req, res) => {
  const { number, label } = req.body;
  await promisePool.execute('UPDATE home_counter SET number=?, label=? WHERE id=?', [number, label, req.params.id]);
  res.redirect('/admin/home-counter');
});
app.get('/admin/home-counter/delete/:id', async (req, res) => {
  await promisePool.execute('DELETE FROM home_counter WHERE id=?', [req.params.id]);
  res.redirect('/admin/home-counter');
});

app.get('/admin/home-sports', async (req, res) => {
  const [sports] = await promisePool.execute('SELECT * FROM home_sports ORDER BY id DESC');
  res.render('admin/home-sports', { sports });
});
app.post('/admin/home-sports/add', upload.single('sport_image'), async (req, res) => {
  const { sport_title, sport_description } = req.body;
  const sport_image = req.file ? req.file.filename : null;
  await promisePool.execute('INSERT INTO home_sports (sport_title, sport_description, sport_image) VALUES (?, ?, ?)', [sport_title, sport_description, sport_image]);
  res.redirect('/admin/home-sports');
});
app.get('/admin/home-sports/edit/:id', async (req, res) => {
  const [rows] = await promisePool.execute('SELECT * FROM home_sports WHERE id=?', [req.params.id]);
  res.render('admin/home-sports-edit', { sport: rows[0] });
});
app.post('/admin/home-sports/update/:id', upload.single('sport_image'), async (req, res) => {
  const { sport_title, sport_description, old_image } = req.body;
  const sport_image = req.file ? req.file.filename : old_image;
  await promisePool.execute('UPDATE home_sports SET sport_title=?, sport_description=?, sport_image=? WHERE id=?', [sport_title, sport_description, sport_image, req.params.id]);
  res.redirect('/admin/home-sports');
});
app.get('/admin/home-sports/delete/:id', async (req, res) => {
  await promisePool.execute('DELETE FROM home_sports WHERE id=?', [req.params.id]);
  res.redirect('/admin/home-sports');
});

app.get('/admin/home-why-us-left', async (req, res) => {
  const [tabs] = await promisePool.execute("SELECT * FROM home_whyus_tabs ORDER BY id ASC");
  res.render('admin/home-why-us-left', { tabs });
});
app.get('/admin/home-why-us-left/edit/:id', async (req, res) => {
  const [rows] = await promisePool.execute("SELECT * FROM home_whyus_tabs WHERE id=?", [req.params.id]);
  res.render('admin/home-why-us-left-edit', { tab: rows[0] });
});
app.post('/admin/home-why-us-left/update/:id', async (req, res) => {
  const { tab_title, tab_content } = req.body;

  await promisePool.execute(
    "UPDATE home_whyus_tabs SET tab_title=?, tab_content=? WHERE id=?",
    [tab_title, tab_content, req.params.id]
  );

  res.redirect('/admin/home-why-us-left');
});
app.get('/admin/home-why-us-right', async (req, res) => {
  const [boxes] = await promisePool.execute("SELECT * FROM home_whyus_boxes ORDER BY display_order ASC");
  res.render('admin/home-why-us-right', { boxes });
});
app.get('/admin/home-why-us-right/edit/:id', async (req, res) => {
  const [rows] = await promisePool.execute("SELECT * FROM home_whyus_boxes WHERE id=?", [req.params.id]);
  res.render('admin/home-why-us-right-edit', { box: rows[0] });
});
app.post('/admin/home-why-us-right/update/:id', upload.single('box_image'), async (req, res) => {
  const { box_title, display_order, old_image } = req.body;
  const box_image = req.file ? req.file.filename : old_image;

  await promisePool.execute(
    "UPDATE home_whyus_boxes SET box_title=?, box_image=?, display_order=? WHERE id=?",
    [box_title, box_image, display_order, req.params.id]
  );

  res.redirect('/admin/home-why-us-right');
});

app.get('/admin/home-lifeattec', async (req, res) => {
  const [items] = await promisePool.execute('SELECT * FROM home_lifeattec ORDER BY id DESC');
  res.render('admin/home-lifeattec', { items });
});
app.post('/admin/home-lifeattec/add', upload.single('image'), async (req, res) => {
  const { title, subtitle, content } = req.body;
  const image = req.file ? req.file.filename : null;
  await promisePool.execute('INSERT INTO home_lifeattec (title, subtitle, content, image) VALUES (?, ?, ?, ?)', [title, subtitle, content, image]);
  res.redirect('/admin/home-lifeattec');
});
app.get('/admin/home-lifeattec/edit/:id', async (req, res) => {
  const [rows] = await promisePool.execute('SELECT * FROM home_lifeattec WHERE id=?', [req.params.id]);
  res.render('admin/home-lifeattec-edit', { item: rows[0] });
});
app.post('/admin/home-lifeattec/update/:id', upload.single('image'), async (req, res) => {
  const { title, subtitle, content, old_image } = req.body;
  const image = req.file ? req.file.filename : old_image;
  await promisePool.execute('UPDATE home_lifeattec SET title=?, subtitle=?, content=?, image=? WHERE id=?', [title, subtitle, content, image, req.params.id]);
  res.redirect('/admin/home-lifeattec');
});
app.get('/admin/home-lifeattec/delete/:id', async (req, res) => {
  await promisePool.execute('DELETE FROM home_lifeattec WHERE id=?', [req.params.id]);
  res.redirect('/admin/home-lifeattec');
});

app.get('/admin/home-testimonials', async (req, res) => {
  const [list] = await promisePool.execute('SELECT * FROM home_testimonials ORDER BY id DESC');
  res.render('admin/home-testimonials', { list });
});
app.post('/admin/home-testimonials/add', async (req, res) => {
  const { author, content, rating, author_title } = req.body;
  await promisePool.execute('INSERT INTO home_testimonials (author, content, rating, author_title) VALUES (?, ?, ?, ?)', [author, content, rating || 5, author_title]);
  res.redirect('/admin/home-testimonials');
});
app.get('/admin/home-testimonials/edit/:id', async (req, res) => {
  const [rows] = await promisePool.execute('SELECT * FROM home_testimonials WHERE id=?', [req.params.id]);
  res.render('admin/home-testimonials-edit', { item: rows[0] });
});
app.post('/admin/home-testimonials/update/:id', async (req, res) => {
  const { author, content, rating, author_title } = req.body;
  await promisePool.execute('UPDATE home_testimonials SET author=?, content=?, rating=?, author_title=? WHERE id=?', [author, content, rating || 5, author_title, req.params.id]);
  res.redirect('/admin/home-testimonials');
});
app.get('/admin/home-testimonials/delete/:id', async (req, res) => {
  await promisePool.execute('DELETE FROM home_testimonials WHERE id=?', [req.params.id]);
  res.redirect('/admin/home-testimonials');
});

app.get('/about', async (req, res) => {
  const [rows] = await promisePool.execute(
    "SELECT * FROM about_content WHERE id = 1"
  );
  res.render('about', { aboutData: rows[0] || {} });
});
app.get('/admin/about', async (req, res) => {
  const [rows] = await promisePool.execute(
    "SELECT * FROM about_content WHERE id = 1"
  );
  res.render('admin/about', { aboutData: rows[0] || {} });
});
app.get('/admin/about-edit', async (req, res) => {
  const [rows] = await promisePool.execute(
    "SELECT * FROM about_content WHERE id = 1"
  );
  res.render('admin/about-edit', { aboutData: rows[0] || {} });
});
app.post('/admin/about/update',
  upload.fields([
    { name: 'about_us_image', maxCount: 1 },
    { name: 'vision_image', maxCount: 1 }
  ]),
  async (req, res) => {

    // helper → undefined => null
    const v = (val) => (val === undefined || val === '' ? null : val);

    const {
      subtitle, title,
      experience_title, experience_years, experience_subtitle,
      description, description_1,
      phone, extra_content_1,
      vision_title, vision_content,
      quality_title, quality_content,
      old_about_us_image,
      old_vision_image
    } = req.body;

    const aboutImage = req.files?.about_us_image
      ? req.files.about_us_image[0].filename
      : v(old_about_us_image);

    const visionImage = req.files?.vision_image
      ? req.files.vision_image[0].filename
      : v(old_vision_image);

    await promisePool.execute(
      `
      UPDATE about_content SET
        subtitle = ?,
        title = ?,
        about_us_image = ?,
        experience_title = ?,
        experience_years = ?,
        experience_subtitle = ?,
        description = ?,
        description_1 = ?,
        phone = ?,
        extra_content_1 = ?,
        vision_title = ?,
        vision_content = ?,
        vision_image = ?,
        quality_title = ?,
        quality_content = ?
      WHERE id = 1
      `,
      [
        v(subtitle),
        v(title),
        aboutImage,
        v(experience_title),
        v(experience_years),
        v(experience_subtitle),
        v(description),
        v(description_1),
        v(phone),
        v(extra_content_1),
        v(vision_title),
        v(vision_content),
        visionImage,
        v(quality_title),
        v(quality_content)
      ]
    );

    res.redirect('/admin/about');
  }
);

const principalUpload = multer({
  storage: multer.diskStorage({
    destination: (req, file, cb) => {
      if (file.fieldname === 'principal_image') {
        cb(null, 'public/images');
      } else if (file.fieldname === 'principal_profile') {
        cb(null, 'public/files');
      }
    },
    filename: (req, file, cb) => {
      if (file.fieldname === 'principal_profile') {
        cb(null, 'Principal-Profile.pdf');
      } else {
        cb(null, Date.now() + '-' + file.originalname);
      }
    }
  })
});
app.get('/principal-desk', async (req, res) => {
  const [rows] = await promisePool.execute(
    "SELECT * FROM principal_desk ORDER BY id DESC LIMIT 1"
  );
  res.render('principal-desk', { data: rows[0] || {} });
});
app.get('/admin/principal-desk', async (req, res) => {
  const [rows] = await promisePool.execute(
    "SELECT * FROM principal_desk ORDER BY id DESC LIMIT 1"
  );
  res.render('admin/principal-desk', { data: rows[0] || {} });
});
app.get(['/admin/principal-desk-edit', '/admin/principal-desk-edit/:id'], async (req, res) => {
  let data = {};
  if (req.params.id) {
    const [rows] = await promisePool.execute(
      "SELECT * FROM principal_desk WHERE id=?",
      [req.params.id]
    );
    data = rows[0];
  }
  res.render('admin/principal-desk-edit', { data });
});
app.post('/admin/principal-desk-update',
  principalUpload.any(),
  async (req, res) => {

    const {
      id,
      principal_name,
      principal_position,
      principal_content,
      principal_list_left,
      principal_list_right,
      principal_content_bottom,
      old_image,
      old_profile
    } = req.body;

    let imageFile = old_image;
    let pdfFile = old_profile;

    if (req.files && req.files.length > 0) {
      req.files.forEach(file => {
        if (file.fieldname === 'principal_image') {
          imageFile = file.filename;
        }
        if (file.fieldname === 'principal_profile') {
          pdfFile = file.filename;
        }
      });
    }

    if (id) {
      await promisePool.execute(
        `UPDATE principal_desk SET
          principal_name=?,
          principal_position=?,
          principal_image=?,
          principal_profile=?,
          principal_content=?,
          principal_list_left=?,
          principal_list_right=?,
          principal_content_bottom=?
         WHERE id=?`,
        [
          principal_name,
          principal_position,
          imageFile,
          pdfFile,
          principal_content,
          principal_list_left,
          principal_list_right,
          principal_content_bottom,
          id
        ]
      );
    } else {
      await promisePool.execute(
        `INSERT INTO principal_desk
        (principal_name, principal_position, principal_image, principal_profile, principal_content, principal_list_left, principal_list_right, principal_content_bottom)
        VALUES (?,?,?,?,?,?,?,?)`,
        [
          principal_name,
          principal_position,
          imageFile,
          pdfFile,
          principal_content,
          principal_list_left,
          principal_list_right,
          principal_content_bottom
        ]
      );
    }

    res.redirect('/admin/principal-desk');
  }
);

app.get('/chairman-desk', async (req, res) => {
  try {
    const [rows] = await promisePool.execute(
      'SELECT * FROM chairman_content ORDER BY id DESC LIMIT 1'
    );
    const data = rows[0] || {};
    res.render('chairman-desk', {
      title: 'Chairman Desk',
      data,
    });
  } catch (error) {
    console.error('Error loading chairman data:', error);
    res.render('chairman-desk', {
      title: 'Chairman Desk',
      data: {},
    });
  }
});
app.get('/admin/chairman-desk', async (req, res) => {
  const [rows] = await promisePool.execute('SELECT * FROM chairman_content LIMIT 1');
  res.render('admin/chairman-desk', { title: 'Chairman Desk', data: rows[0] || {} });
});
app.get('/admin/chairman-desk-edit', async (req, res) => {
  const [rows] = await promisePool.execute('SELECT * FROM chairman_content LIMIT 1');
  res.render('admin/chairman-desk-edit', { title: 'Edit Chairman Desk', data: rows[0] || {} });
});
app.post('/admin/chairman-desk-update', upload.single('image'), async (req, res) => {
  const { name, designation, message, oldImage } = req.body;
  const image = req.file ? req.file.filename : oldImage;
  await promisePool.execute('DELETE FROM chairman_content');
  await promisePool.execute(
    'INSERT INTO chairman_content (name, designation, message, image) VALUES (?, ?, ?, ?)',
    [name, designation, message, image]
  );
  res.redirect('/admin/chairman-desk');
});
app.post('/admin/chairman-desk-delete', async (req, res) => {
  await promisePool.execute('DELETE FROM chairman_content');
  res.redirect('/admin/chairman-desk');
});

app.get('/vision-mission-quality-policy', async (req, res) => {
  try {
    const [rows] = await promisePool.execute(
      'SELECT * FROM vision_mission_quality ORDER BY id DESC LIMIT 1'
    );
    const data = rows[0] || {};
    res.render('vision-mission-quality-policy', {
      title: 'Vision, Mission & Quality Policy',
      data,
    });
  } catch (error) {
    console.error('Error loading Vision/Mission/Quality data:', error);
    res.render('vision-mission-quality-policy', {
      title: 'Vision, Mission & Quality Policy',
      data: {},
    });
  }
});
app.get('/admin/vision-mission-quality-policy', async (req, res) => {
  try {
    const [rows] = await promisePool.execute('SELECT * FROM vision_mission_quality LIMIT 1');
    res.render('admin/vision-mission-quality-policy', { title: 'Vision, Mission & Quality Policy', data: rows[0] || {} });
  } catch (err) {
    console.error('Error loading vision mission quality policy:', err);
    res.render('admin/vision-mission-quality-policy', { title: 'Vision, Mission & Quality Policy', data: {} });
  }
});
app.get('/admin/vision-mission-quality-policy', async (req, res) => {
  try {
    const [rows] = await promisePool.execute('SELECT * FROM vision_mission_quality LIMIT 1');
    res.render('admin/vision-mission-quality-policy', {
      title: 'Vision, Mission & Quality Policy',
      data: rows[0] || {}
    });
  } catch (err) {
    console.error('Error loading vision page:', err);
    res.render('admin/vision-mission-quality-policy', {
      title: 'Vision, Mission & Quality Policy',
      data: {}
    });
  }
});
app.get('/admin/vision-mission-quality-policy-edit', async (req, res) => {
  try {
    const [rows] = await promisePool.execute('SELECT * FROM vision_mission_quality LIMIT 1');
    res.render('admin/vision-mission-quality-policy-edit', {
      title: 'Edit Vision, Mission & Quality Policy',
      data: rows[0] || {}
    });
  } catch (err) {
    console.error('Error loading edit page:', err);
    res.render('admin/vision-mission-quality-policy-edit', { title: 'Edit Vision, Mission & Quality Policy', data: {} });
  }
});
app.post('/admin/vision-mission-quality-policy-update', upload.single('image'), async (req, res) => {
  try {
    const { vision, mission, quality_policy, oldImage } = req.body;
    const image = req.file ? req.file.filename : oldImage;

    await promisePool.execute('DELETE FROM vision_mission_quality');
    await promisePool.execute(
      'INSERT INTO vision_mission_quality (vision, mission, quality_policy, image) VALUES (?, ?, ?, ?)',
      [vision, mission, quality_policy, image]
    );

    res.redirect('/admin/vision-mission-quality-policy');
  } catch (err) {
    console.error('Error updating vision page:', err);
    res.status(500).send('Error updating vision page');
  }
});
app.post('/admin/vision-mission-quality-policy-delete', async (req, res) => {
  await promisePool.execute('DELETE FROM vision_mission_quality');
  res.redirect('/admin/vision-mission-quality-policy');
});

app.get('/general-information', async (req, res) => {
  try {
    const [rows] = await promisePool.execute('SELECT * FROM general_information ORDER BY id DESC LIMIT 1');
    const data = rows[0] || {};
    res.render('general-information', { title: 'General Information', data });
  } catch (error) {
    console.error('Error loading general information:', error);
    res.render('general-information', { title: 'General Information', data: {} });
  }
});
app.get('/admin/general-information', async (req, res) => {
  const [rows] = await promisePool.execute('SELECT * FROM general_information LIMIT 1');
  res.render('admin/general-information', { title: 'General Information', data: rows[0] || {} });
});
app.get('/admin/general-information-edit', async (req, res) => {
  const [rows] = await promisePool.execute('SELECT * FROM general_information LIMIT 1');
  res.render('admin/general-information-edit', { title: 'Edit General Information', data: rows[0] || {} });
});
app.post('/admin/general-information-update', upload.single('image'), async (req, res) => {
  const { content, trust_head, trust_name, address, district, chairman_name, oldImage } = req.body;
  const image = req.file ? req.file.filename : oldImage;
  await promisePool.execute('DELETE FROM general_information');
  await promisePool.execute(
    'INSERT INTO general_information (content, image, trust_head, trust_name, address, district, chairman_name) VALUES (?, ?, ?, ?, ?, ?, ?)',
    [content, image, trust_head, trust_name, address, district, chairman_name]
  );
  res.redirect('/admin/general-information');
});
app.post('/admin/general-information-delete', async (req, res) => {
  await promisePool.execute('DELETE FROM general_information');
  res.redirect('/admin/general-information');
});

app.get('/mandatory-disclosure', async (req, res) => {
  try {
    const [pdfList] = await promisePool.execute("SELECT * FROM mandatory_disclosure ORDER BY year DESC");
    res.render('mandatory-disclosure', { pdfList });
  } catch (error) {
    console.error('Error fetching mandatory disclosures:', error);
    res.render('mandatory-disclosure', { pdfList: [] });
  }
});
app.get('/admin/mandatory-disclosure', async (req, res) => {
  const [rows] = await promisePool.execute('SELECT * FROM mandatory_disclosure ORDER BY id DESC');
  res.render('admin/mandatory-disclosure', { title: 'Mandatory Disclosure', data: rows });
});
app.get(['/admin/mandatory-disclosure-edit', '/admin/mandatory-disclosure-edit/:id'], async (req, res) => {
  const { id } = req.params;
  let data = {};
  if (id) {
    const [rows] = await promisePool.execute('SELECT * FROM mandatory_disclosure WHERE id = ?', [id]);
    data = rows[0] || {};
  }
  res.render('admin/mandatory-disclosure-edit', { title: 'Edit Mandatory Disclosure', data });
});
app.post('/admin/mandatory-disclosure-update',
  uploadPDF.single('pdf_file'),
  async (req, res) => {

    const { id, title, year, oldFile } = req.body;

    let pdfFile = oldFile;

    if (req.file) {
      pdfFile = req.file.originalname; // <-- SAME NAME AS FILE
    }

    if (id) {
      await promisePool.execute(
        `UPDATE mandatory_disclosure 
         SET title=?, year=?, pdf_file=? 
         WHERE id=?`,
        [title, year, pdfFile, id]
      );
    } else {
      await promisePool.execute(
        `INSERT INTO mandatory_disclosure (title, year, pdf_file)
         VALUES (?,?,?)`,
        [title, year, pdfFile]
      );
    }

    res.redirect('/admin/mandatory-disclosure');
  }
);
app.get('/admin/mandatory-disclosure-delete/:id', async (req, res) => {
  const { id } = req.params;
  await promisePool.execute('DELETE FROM mandatory_disclosure WHERE id = ?', [id]);
  res.redirect('/admin/mandatory-disclosure');
});

app.get('/statutory-declaration', async (req, res) => {
  try {
    const [pdfList] = await promisePool.execute(
      "SELECT * FROM statutory_declaration ORDER BY id DESC"
    );
    res.render('statutory-declaration', { pdfList });
  } catch (e) {
    res.render('statutory-declaration', { pdfList: [] });
  }
});
app.get('/admin/statutory-declaration', async (req, res) => {
  const [rows] = await promisePool.execute(
    'SELECT * FROM statutory_declaration ORDER BY id DESC'
  );

  res.render('admin/statutory-declaration', {
    title: 'Statutory Declaration',
    data: rows
  });
});
app.get(['/admin/statutory-declaration-edit', '/admin/statutory-declaration-edit/:id'],
  async (req, res) => {

    const { id } = req.params;
    let data = {};

    if (id) {
      const [rows] = await promisePool.execute(
        'SELECT * FROM statutory_declaration WHERE id=?',
        [id]
      );
      data = rows[0] || {};
    }

    res.render('admin/statutory-declaration-edit', {
      title: 'Statutory Declaration',
      data
    });
  }
);
app.post('/admin/statutory-declaration-update',
  uploadPDF.single('pdf_file'),
  async (req, res) => {

    const { id, title, oldFile } = req.body;
    const pdfFile = req.file ? req.file.originalname : oldFile;

    if (id) {
      await promisePool.execute(
        'UPDATE statutory_declaration SET title=?, pdf_file=? WHERE id=?',
        [title, pdfFile, id]
      );
    } else {
      await promisePool.execute(
        'INSERT INTO statutory_declaration (title, pdf_file) VALUES (?,?)',
        [title, pdfFile]
      );
    }

    res.redirect('/admin/statutory-declaration');
  }
);
app.get('/admin/statutory-declaration-delete/:id', async (req, res) => {
  await promisePool.execute(
    'DELETE FROM statutory_declaration WHERE id=?',
    [req.params.id]
  );
  res.redirect('/admin/statutory-declaration');
});

app.get('/affiliation-recognition', async (req, res) => {
  try {
    const [rows] = await promisePool.execute(
      "SELECT * FROM affiliation_recognition ORDER BY id DESC"
    );
    res.render('affiliation-recognition', { pdfList: rows });
  } catch (error) {
    console.error('Error fetching affiliation recognition files:', error);
    res.render('affiliation-recognition', { pdfList: [] });
  }
});
app.get('/admin/affiliation-recognition', async (req, res) => {
  const [rows] = await promisePool.execute(
    'SELECT * FROM affiliation_recognition ORDER BY id DESC'
  );

  res.render('admin/affiliation-recognition', {
    title: 'Affiliation Recognition',
    data: rows
  });
});
app.get(['/admin/affiliation-recognition-edit', '/admin/affiliation-recognition-edit/:id'],
  async (req, res) => {

    const { id } = req.params;
    let data = {};

    if (id) {
      const [rows] = await promisePool.execute(
        'SELECT * FROM affiliation_recognition WHERE id=?',
        [id]
      );
      data = rows[0] || {};
    }

    res.render('admin/affiliation-recognition-edit', {
      title: 'Affiliation Recognition',
      data
    });
  }
);
app.post('/admin/affiliation-recognition-update',
  uploadPDF.single('pdf_file'),
  async (req, res) => {

    const { id, title, oldFile } = req.body;
    const pdfFile = req.file ? req.file.originalname : oldFile;

    if (id) {
      await promisePool.execute(
        'UPDATE affiliation_recognition SET title=?, pdf_file=? WHERE id=?',
        [title, pdfFile, id]
      );
    } else {
      await promisePool.execute(
        'INSERT INTO affiliation_recognition (title, pdf_file) VALUES (?,?)',
        [title, pdfFile]
      );
    }

    res.redirect('/admin/affiliation-recognition');
  }
);
app.get('/admin/affiliation-recognition-delete/:id', async (req, res) => {
  await promisePool.execute(
    'DELETE FROM affiliation_recognition WHERE id=?',
    [req.params.id]
  );
  res.redirect('/admin/affiliation-recognition');
});

app.get('/aicte-approvals', async (req, res) => {
  try {
    const [rows] = await promisePool.execute(
      "SELECT * FROM aicte_approvals ORDER BY year DESC"
    );
    res.render('aicte-approvals', { pdfList: rows });
  } catch (error) {
    console.error('Error fetching AICTE approvals:', error);
    res.render('aicte-approvals', { pdfList: [] });
  }
});
app.get('/admin/aicte-approvals', async (req, res) => {
  const [rows] = await promisePool.execute(
    'SELECT * FROM aicte_approvals ORDER BY year DESC'
  );
  res.render('admin/aicte-approvals', {
    title: 'AICTE Approvals',
    data: rows
  });
});
app.get(['/admin/aicte-approvals-edit', '/admin/aicte-approvals-edit/:id'],
  async (req, res) => {

    const { id } = req.params;
    let data = {};

    if (id) {
      const [rows] = await promisePool.execute(
        'SELECT * FROM aicte_approvals WHERE id=?',
        [id]
      );
      data = rows[0] || {};
    }

    res.render('admin/aicte-approvals-edit', {
      title: 'Edit AICTE Approvals',
      data
    });
  }
);
app.post('/admin/aicte-approvals-update',
  uploadPDF.single('pdf_file'),
  async (req, res) => {

    const { id, title, year, oldFile } = req.body;
    const pdfFile = req.file ? req.file.originalname : oldFile;

    if (id) {
      await promisePool.execute(
        'UPDATE aicte_approvals SET title=?, year=?, pdf_file=? WHERE id=?',
        [title, year, pdfFile, id]
      );
    } else {
      await promisePool.execute(
        'INSERT INTO aicte_approvals (title, year, pdf_file) VALUES (?,?,?)',
        [title, year, pdfFile]
      );
    }

    res.redirect('/admin/aicte-approvals');
  }
);
app.get('/admin/aicte-approvals-delete/:id', async (req, res) => {
  await promisePool.execute(
    'DELETE FROM aicte_approvals WHERE id=?',
    [req.params.id]
  );
  res.redirect('/admin/aicte-approvals');
});

app.get('/hr-policy', async (req, res) => {
  try {
    // main page values (single latest row)
    const [mainRow] = await promisePool.execute("SELECT * FROM hr_policy_main ORDER BY id DESC LIMIT 1");
    const mainData = mainRow[0] || {};

    // about accordion
    const [aboutRows] = await promisePool.execute("SELECT * FROM hr_policy_main_about ORDER BY id ASC");

    // governance top (single row)
    const [topRow] = await promisePool.execute("SELECT * FROM hr_policy_main_governance_top ORDER BY id DESC LIMIT 1");
    const topData = topRow[0] || {};

    // left/right/bottom governance items
    const [leftRows] = await promisePool.execute("SELECT * FROM hr_policy_main_governance_left ORDER BY id ASC");
    const [rightRows] = await promisePool.execute("SELECT * FROM hr_policy_main_governance_right ORDER BY id ASC");
    const [bottomRows] = await promisePool.execute("SELECT * FROM hr_policy_main_governance_bottom ORDER BY id ASC");

    // contact
    const [contactRow] = await promisePool.execute("SELECT * FROM hr_policy_main_contact ORDER BY id DESC LIMIT 1");
    const contactData = contactRow[0] || {};

    res.render('hr-policy', {
      mainData: Object.assign({}, mainData, topData),
      aboutList: aboutRows,
      hrLeft: leftRows,
      hrRight: rightRows,
      bottomList: bottomRows,
      contactData
    });

  } catch (err) {
    console.error("HR Policy Error:", err);
    res.render('hr-policy', { mainData: {}, aboutList: [], hrLeft: [], hrRight: [], bottomList: [], contactData: {} });
  }
});
app.get('/admin/hr-policy', (req, res) => {
  res.render('admin/hr-policy-dashboard');
});
app.get('/admin/hr-policy-main', async (req, res) => {
  const [rows] = await promisePool.execute("SELECT * FROM hr_policy_main ORDER BY id DESC");
  res.render('admin/hr-policy-main', { rows });
});
app.post('/admin/hr-policy-main/add', async (req, res) => {
  const { college_name, college_address, main_title, sub_title, bottom_title } = req.body;
  await promisePool.execute("INSERT INTO hr_policy_main (college_name, college_address, main_title, sub_title, bottom_title) VALUES (?, ?, ?, ?, ?)",
    [college_name, college_address, main_title, sub_title, bottom_title]);
  res.redirect('/admin/hr-policy-main');
});
app.get('/admin/hr-policy-main-edit/:id', async (req, res) => {
  const [row] = await promisePool.execute("SELECT * FROM hr_policy_main WHERE id=?", [req.params.id]);
  res.render('admin/hr-policy-main-edit', { row: row[0] });
});
app.post('/admin/hr-policy-main/update/:id', async (req, res) => {
  const { college_name, college_address, main_title, sub_title, bottom_title } = req.body;
  await promisePool.execute("UPDATE hr_policy_main SET college_name=?, college_address=?, main_title=?, sub_title=?, bottom_title=? WHERE id=?",
    [college_name, college_address, main_title, sub_title, bottom_title, req.params.id]);
  res.redirect('/admin/hr-policy-main');
});
app.get('/admin/hr-policy-main/delete/:id', async (req, res) => {
  await promisePool.execute("DELETE FROM hr_policy_main WHERE id=?", [req.params.id]);
  res.redirect('/admin/hr-policy-main');
});
app.get('/admin/hr-policy-about', async (req, res) => {
  const [rows] = await promisePool.execute("SELECT * FROM hr_policy_main_about ORDER BY id ASC");
  res.render('admin/hr-policy-about', { rows });
});
app.post('/admin/hr-policy-about/add', async (req, res) => {
  const { title, content } = req.body;
  await promisePool.execute("INSERT INTO hr_policy_main_about (title, content) VALUES (?, ?)", [title, content]);
  res.redirect('/admin/hr-policy-about');
});
app.get('/admin/hr-policy-about-edit/:id', async (req, res) => {
  const [row] = await promisePool.execute("SELECT * FROM hr_policy_main_about WHERE id=?", [req.params.id]);
  res.render('admin/hr-policy-about-edit', { row: row[0] });
});
app.post('/admin/hr-policy-about/update/:id', async (req, res) => {
  const { title, content } = req.body;
  await promisePool.execute("UPDATE hr_policy_main_about SET title=?, content=? WHERE id=?", [title, content, req.params.id]);
  res.redirect('/admin/hr-policy-about');
});
app.get('/admin/hr-policy-about/delete/:id', async (req, res) => {
  await promisePool.execute("DELETE FROM hr_policy_main_about WHERE id=?", [req.params.id]);
  res.redirect('/admin/hr-policy-about');
});
app.get('/admin/hr-policy-top', async (req, res) => {
  const [rows] = await promisePool.execute("SELECT * FROM hr_policy_main_governance_top ORDER BY id DESC");
  res.render('admin/hr-policy-top', { rows });
});
app.post('/admin/hr-policy-top/add', async (req, res) => {
  const { main_title, sub_title } = req.body;
  await promisePool.execute("INSERT INTO hr_policy_main_governance_top (main_title, sub_title) VALUES (?, ?)", [main_title, sub_title]);
  res.redirect('/admin/hr-policy-top');
});
app.get('/admin/hr-policy-top-edit/:id', async (req, res) => {
  const [row] = await promisePool.execute("SELECT * FROM hr_policy_main_governance_top WHERE id=?", [req.params.id]);
  res.render('admin/hr-policy-top-edit', { row: row[0] });
});
app.post('/admin/hr-policy-top/update/:id', async (req, res) => {
  const { main_title, sub_title } = req.body;
  await promisePool.execute("UPDATE hr_policy_main_governance_top SET main_title=?, sub_title=? WHERE id=?", [main_title, sub_title, req.params.id]);
  res.redirect('/admin/hr-policy-top');
});
app.get('/admin/hr-policy-top/delete/:id', async (req, res) => {
  await promisePool.execute("DELETE FROM hr_policy_main_governance_top WHERE id=?", [req.params.id]);
  res.redirect('/admin/hr-policy-top');
});
app.get('/admin/hr-policy-left', async (req, res) => {
  const [rows] = await promisePool.execute("SELECT * FROM hr_policy_main_governance_left ORDER BY id ASC");
  res.render('admin/hr-policy-left', { rows });
});
app.post('/admin/hr-policy-left/add', async (req, res) => {
  const { title, content } = req.body;
  await promisePool.execute("INSERT INTO hr_policy_main_governance_left (title, content) VALUES (?, ?)", [title, content]);
  res.redirect('/admin/hr-policy-left');
});
app.get('/admin/hr-policy-left-edit/:id', async (req, res) => {
  const [row] = await promisePool.execute("SELECT * FROM hr_policy_main_governance_left WHERE id=?", [req.params.id]);
  res.render('admin/hr-policy-left-edit', { row: row[0] });
});
app.post('/admin/hr-policy-left/update/:id', async (req, res) => {
  await promisePool.execute("UPDATE hr_policy_main_governance_left SET title=?, content=? WHERE id=?", [req.body.title, req.body.content, req.params.id]);
  res.redirect('/admin/hr-policy-left');
});
app.get('/admin/hr-policy-left/delete/:id', async (req, res) => {
  await promisePool.execute("DELETE FROM hr_policy_main_governance_left WHERE id=?", [req.params.id]);
  res.redirect('/admin/hr-policy-left');
});
app.get('/admin/hr-policy-right', async (req, res) => {
  const [rows] = await promisePool.execute("SELECT * FROM hr_policy_main_governance_right ORDER BY id ASC");
  res.render('admin/hr-policy-right', { rows });
});
app.post('/admin/hr-policy-right/add', async (req, res) => {
  const { title, content } = req.body;
  await promisePool.execute("INSERT INTO hr_policy_main_governance_right (title, content) VALUES (?, ?)", [title, content]);
  res.redirect('/admin/hr-policy-right');
});
app.get('/admin/hr-policy-right-edit/:id', async (req, res) => {
  const [row] = await promisePool.execute("SELECT * FROM hr_policy_main_governance_right WHERE id=?", [req.params.id]);
  res.render('admin/hr-policy-right-edit', { row: row[0] });
});
app.post('/admin/hr-policy-right/update/:id', async (req, res) => {
  await promisePool.execute("UPDATE hr_policy_main_governance_right SET title=?, content=? WHERE id=?", [req.body.title, req.body.content, req.params.id]);
  res.redirect('/admin/hr-policy-right');
});
app.get('/admin/hr-policy-right/delete/:id', async (req, res) => {
  await promisePool.execute("DELETE FROM hr_policy_main_governance_right WHERE id=?", [req.params.id]);
  res.redirect('/admin/hr-policy-right');
});
app.get('/admin/hr-policy-bottom', async (req, res) => {
  const [rows] = await promisePool.execute("SELECT * FROM hr_policy_main_governance_bottom ORDER BY id ASC");
  res.render('admin/hr-policy-bottom', { rows });
});
app.post('/admin/hr-policy-bottom/add', async (req, res) => {
  const { title, content } = req.body;
  await promisePool.execute("INSERT INTO hr_policy_main_governance_bottom (title, content) VALUES (?, ?)", [title, content]);
  res.redirect('/admin/hr-policy-bottom');
});
app.get('/admin/hr-policy-bottom-edit/:id', async (req, res) => {
  const [row] = await promisePool.execute("SELECT * FROM hr_policy_main_governance_bottom WHERE id=?", [req.params.id]);
  res.render('admin/hr-policy-bottom-edit', { row: row[0] });
});
app.post('/admin/hr-policy-bottom/update/:id', async (req, res) => {
  await promisePool.execute("UPDATE hr_policy_main_governance_bottom SET title=?, content=? WHERE id=?", [req.body.title, req.body.content, req.params.id]);
  res.redirect('/admin/hr-policy-bottom');
});
app.get('/admin/hr-policy-bottom/delete/:id', async (req, res) => {
  await promisePool.execute("DELETE FROM hr_policy_main_governance_bottom WHERE id=?", [req.params.id]);
  res.redirect('/admin/hr-policy-bottom');
});
app.get('/admin/hr-policy-contact', async (req, res) => {
  const [rows] = await promisePool.execute("SELECT * FROM hr_policy_main_contact ORDER BY id DESC");
  res.render('admin/hr-policy-contact', { rows });
});
app.post('/admin/hr-policy-contact/add', async (req, res) => {
  const { content } = req.body;
  await promisePool.execute("INSERT INTO hr_policy_main_contact (content) VALUES (?)", [content]);
  res.redirect('/admin/hr-policy-contact');
});
app.get('/admin/hr-policy-contact-edit/:id', async (req, res) => {
  const [row] = await promisePool.execute("SELECT * FROM hr_policy_main_contact WHERE id=?", [req.params.id]);
  res.render('admin/hr-policy-contact-edit', { row: row[0] });
});
app.post('/admin/hr-policy-contact/update/:id', async (req, res) => {
  await promisePool.execute("UPDATE hr_policy_main_contact SET content=? WHERE id=?", [req.body.content, req.params.id]);
  res.redirect('/admin/hr-policy-contact');
});
app.get('/admin/hr-policy-contact/delete/:id', async (req, res) => {
  await promisePool.execute("DELETE FROM hr_policy_main_contact WHERE id=?", [req.params.id]);
  res.redirect('/admin/hr-policy-contact');
});

app.get('/hr-policy-code-of-conduct', async (req, res) => {
  try {
    const [rows] = await promisePool.execute(
      "SELECT * FROM hr_policy_code ORDER BY id DESC"
    );
    res.render('hr-policy-code-of-conduct', { pdfList: rows });
  } catch (error) {
    console.error('Error fetching HR policy code files:', error);
    res.render('hr-policy-code-of-conduct', { pdfList: [] });
  }
});
app.get('/admin/hr-policy-code-of-conduct', async (req, res) => {
  const [rows] = await promisePool.execute(
    'SELECT * FROM hr_policy_code ORDER BY id DESC'
  );
  res.render('admin/hr-policy-code-of-conduct', {
    title: 'HR Policy Code of Conduct',
    data: rows
  });
});
app.get(['/admin/hr-policy-code-of-conduct-edit', '/admin/hr-policy-code-of-conduct-edit/:id'],
  async (req, res) => {
    const { id } = req.params;
    let data = {};

    if (id) {
      const [rows] = await promisePool.execute(
        'SELECT * FROM hr_policy_code WHERE id = ?',
        [id]
      );
      data = rows[0] || {};
    }

    res.render('admin/hr-policy-code-of-conduct-edit', {
      title: 'HR Policy Code of Conduct',
      data
    });
  }
);
app.post('/admin/hr-policy-code-of-conduct-update',
  uploadPDF.single('pdf_file'),
  async (req, res) => {
    const { id, title, oldFile } = req.body;
    const pdf_file = req.file ? req.file.filename : oldFile;

    if (id) {
      await promisePool.execute(
        'UPDATE hr_policy_code SET title=?, pdf_file=? WHERE id=?',
        [title, pdf_file, id]
      );
    } else {
      await promisePool.execute(
        'INSERT INTO hr_policy_code (title, pdf_file) VALUES (?, ?)',
        [title, pdf_file]
      );
    }

    res.redirect('/admin/hr-policy-code-of-conduct');
  }
);
app.get('/admin/hr-policy-code-of-conduct-delete/:id', async (req, res) => {
  const { id } = req.params;
  await promisePool.execute(
    'DELETE FROM hr_policy_code WHERE id = ?',
    [id]
  );
  res.redirect('/admin/hr-policy-code-of-conduct');
});

app.get('/organogram', async (req, res) => {
  const [rows] = await promisePool.execute(
    "SELECT * FROM organogram ORDER BY id DESC"
  );
  res.render('organogram', { imageList: rows });
});
app.get('/admin/organogram', async (req, res) => {
  const [rows] = await promisePool.execute(
    "SELECT * FROM organogram ORDER BY id DESC"
  );
  res.render('admin/organogram', { data: rows });
});
app.get(['/admin/organogram-edit','/admin/organogram-edit/:id'], async (req,res)=>{
  let data = {};
  if (req.params.id) {
    const [rows] = await promisePool.execute(
      "SELECT * FROM organogram WHERE id=?",
      [req.params.id]
    );
    data = rows[0];
  }
  res.render('admin/organogram-edit', { data });
});
app.post('/admin/organogram-update',
  uploadImage.single('image'),
  async (req, res) => {

    const { id, title, year, oldImage } = req.body;

    const image_path = req.file
      ? 'images/' + req.file.filename
      : oldImage;

    if (id) {
      await promisePool.execute(
        "UPDATE organogram SET title=?, image_path=?, year=? WHERE id=?",
        [title, image_path, year, id]
      );
    } else {
      await promisePool.execute(
        "INSERT INTO organogram (title,image_path,year) VALUES (?,?,?)",
        [title, image_path, year]
      );
    }

    res.redirect('/admin/organogram');
});
app.get('/admin/organogram-delete/:id', async (req,res)=>{
  await promisePool.execute(
    "DELETE FROM organogram WHERE id=?",
    [req.params.id]
  );
  res.redirect('/admin/organogram');
});

app.get('/admission', async (req, res) => {
  try {
    const [rows] = await promisePool.execute('SELECT * FROM admission_info ORDER BY id DESC LIMIT 1');
    const admissionData = rows[0] || {};
    res.render('admission', { title: 'Admission', admissionData });
  } catch (error) {
    console.error('Error loading admission page:', error);
    res.render('admission', { title: 'Admission', admissionData: {} });
  }
});
app.get('/admin/admission', async (req, res) => {
  const [rows] = await promisePool.execute('SELECT * FROM admission_info ORDER BY id DESC LIMIT 1');
  const data = rows[0] || {};
  res.render('admin/admission', { title: 'Admin | Admission Page', data });
});
app.get('/admin/admission-edit/:id', async (req, res) => {
  const { id } = req.params;
  const [rows] = await promisePool.execute('SELECT * FROM admission_info WHERE id = ?', [id]);
  const data = rows[0] || {};
  res.render('admin/admission-edit', { title: 'Edit Admission Page', data });
});
app.get('/admin/admission-edit', async (req, res) => {
  const [rows] = await promisePool.execute('SELECT * FROM admission_info ORDER BY id DESC LIMIT 1');
  const data = rows[0] || {};
  res.render('admin/admission-edit', { title: 'Edit Admission Page', data });
});
app.post('/admin/admission-update', async (req, res) => {
  try {
    const {
      id, title, link_text, link_url,
      title_2, title_2_content, title_3, title_3_content,
      title_4, title_4_content, title_5, title_5_content,
      title_6, title_6_content, title_7, title_7_content,
      title_8, title_8_content
    } = req.body;

    const fields = [
      title, link_text, link_url,
      title_2, title_2_content,
      title_3, title_3_content,
      title_4, title_4_content,
      title_5, title_5_content,
      title_6, title_6_content,
      title_7, title_7_content,
      title_8, title_8_content
    ];

    if (id && id !== '') {
      await promisePool.execute(`
        UPDATE admission_info SET
          title=?, link_text=?, link_url=?,
          title_2=?, title_2_content=?,
          title_3=?, title_3_content=?,
          title_4=?, title_4_content=?,
          title_5=?, title_5_content=?,
          title_6=?, title_6_content=?,
          title_7=?, title_7_content=?,
          title_8=?, title_8_content=?
        WHERE id=?`, [...fields, id]);
    } else {
      await promisePool.execute(`
        INSERT INTO admission_info (
          title, link_text, link_url,
          title_2, title_2_content,
          title_3, title_3_content,
          title_4, title_4_content,
          title_5, title_5_content,
          title_6, title_6_content,
          title_7, title_7_content,
          title_8, title_8_content
        ) VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?)`, fields);
    }

    res.redirect('/admin/admission');
  } catch (err) {
    console.error('Error updating admission:', err);
    res.redirect('/admin/admission');
  }
});
app.get('/admin/admission-delete/:id', async (req, res) => {
  const { id } = req.params;
  await promisePool.execute('DELETE FROM admission_info WHERE id=?', [id]);
  res.redirect('/admin/admission');
});

app.get('/academic-schedule', async (req, res) => {
  const [rows] = await promisePool.execute(
    'SELECT * FROM academic_schedule ORDER BY year DESC, id ASC'
  );

  const grouped = {};
  rows.forEach(item => {
    if (!grouped[item.year]) grouped[item.year] = [];
    grouped[item.year].push(item);
  });

  res.render('academic-schedule', {
    title: 'Academic Schedule',
    groupedData: grouped
  });
});
app.get('/admin/academic-schedule', async (req, res) => {
  const [rows] = await promisePool.execute(
    'SELECT * FROM academic_schedule ORDER BY year DESC, id ASC'
  );

  res.render('admin/academic-schedule', {
    title: 'Academic Schedule',
    data: rows
  });
});
app.get(['/admin/academic-schedule-edit', '/admin/academic-schedule-edit/:id'],
  async (req, res) => {

    let data = {};
    if (req.params.id) {
      const [rows] = await promisePool.execute(
        'SELECT * FROM academic_schedule WHERE id = ?',
        [req.params.id]
      );
      data = rows[0] || {};
    }

    res.render('admin/academic-schedule-edit', {
      title: 'Academic Schedule',
      data
    });
  }
);
app.post('/admin/academic-schedule-update',
  uploadPDF.single('pdf_file'),
  async (req, res) => {

    const { id, year, semester, oldFile } = req.body;
    const pdf_file = req.file ? req.file.filename : oldFile;

    if (id && id !== '') {
      // UPDATE
      await promisePool.execute(
        'UPDATE academic_schedule SET year=?, semester=?, pdf_file=? WHERE id=?',
        [year, semester, pdf_file, id]
      );
    } else {
      // INSERT
      await promisePool.execute(
        'INSERT INTO academic_schedule (year, semester, pdf_file) VALUES (?, ?, ?)',
        [year, semester, pdf_file]
      );
    }

    res.redirect('/admin/academic-schedule');
  }
);
app.get('/admin/academic-schedule-delete/:id', async (req, res) => {
  const { id } = req.params;

  await promisePool.execute(
    'DELETE FROM academic_schedule WHERE id = ?',
    [id]
  );

  res.redirect('/admin/academic-schedule');
});

const syllabusStorage = multer.diskStorage({
  destination: function (req, file, cb) {
    cb(null, path.join(__dirname, "public/pdf"));
  },
  filename: function (req, file, cb) {
    cb(null, Date.now() + "-" + file.originalname);
  }
});
const uploadSyllabus = multer({ storage: syllabusStorage });
app.get('/regulations-syllabus', async (req, res) => {
  const [rows] = await promisePool.execute(
    "SELECT * FROM regulations_syllabus ORDER BY syllabus_year DESC, id ASC"
  );
  res.render("regulations-syllabus", { data: rows });
});
app.get('/admin/regulations-syllabus-list', async (req, res) => {
  const [rows] = await promisePool.execute(
    "SELECT * FROM regulations_syllabus ORDER BY id DESC"
  );
  res.render("admin/regulations-syllabus-list", { data: rows });
});
app.get('/admin/regulations-syllabus', (req, res) => {
  res.render("admin/regulations-syllabus");
});
app.post('/admin/regulations-syllabus',
  uploadSyllabus.single("pdf_file"),
  async (req, res) => {

    const { syllabus_year, pdf_title } = req.body;
    const pdf_file = req.file.filename;

    await promisePool.execute(
      "INSERT INTO regulations_syllabus (syllabus_year, pdf_title, pdf_file) VALUES (?, ?, ?)",
      [syllabus_year, pdf_title, pdf_file]
    );

    res.redirect('/admin/regulations-syllabus-list');
  }
);
app.get('/admin/regulations-syllabus-edit/:id', async (req, res) => {
  const [rows] = await promisePool.execute(
    "SELECT * FROM regulations_syllabus WHERE id=?",
    [req.params.id]
  );

  res.render("admin/regulations-syllabus-edit", { data: rows[0] });
});
app.post('/admin/regulations-syllabus-edit/:id',
  uploadSyllabus.single("pdf_file"),
  async (req, res) => {

    const id = req.params.id;
    const { syllabus_year, pdf_title } = req.body;

    if (req.file) {
      await promisePool.execute(
        "UPDATE regulations_syllabus SET syllabus_year=?, pdf_title=?, pdf_file=? WHERE id=?",
        [syllabus_year, pdf_title, req.file.filename, id]
      );
    } else {
      await promisePool.execute(
        "UPDATE regulations_syllabus SET syllabus_year=?, pdf_title=? WHERE id=?",
        [syllabus_year, pdf_title, id]
      );
    }

    res.redirect('/admin/regulations-syllabus-list');
  }
);
app.get('/admin/regulations-syllabus-delete/:id', async (req, res) => {
  await promisePool.execute(
    "DELETE FROM regulations_syllabus WHERE id=?",
    [req.params.id]
  );
  res.redirect('/admin/regulations-syllabus-list');
});

app.get('/r-and-d-cell', async (req, res) => {
  try {
    const [rows] = await promisePool.execute('SELECT * FROM r_d_cell_content ORDER BY id DESC LIMIT 1');
    const data = rows[0] || {};
    res.render('r-and-d-cell', { data });
  } catch (err) {
    console.error('Error fetching R&D Cell:', err);
    res.status(500).send('Server error');
  }
});
app.get('/admin/r-and-d-cell', async (req, res) => {
  try {
    const [rows] = await promisePool.execute('SELECT * FROM r_d_cell_content ORDER BY id DESC LIMIT 1');
    res.render('admin/r-and-d-cell', { data: rows[0] || {} });
  } catch (err) {
    console.error('Error admin r-and-d-cell:', err);
    res.render('admin/r-and-d-cell', { data: {} });
  }
});
app.get('/admin/r-and-d-cell-edit/new', (req, res) => {
  res.render('admin/r-and-d-cell-edit', { data: {} });
});
app.get('/admin/r-and-d-cell-edit/:id', async (req, res) => {
  try {
    const id = req.params.id;
    const [rows] = await promisePool.execute('SELECT * FROM r_d_cell_content WHERE id = ?', [id]);
    res.render('admin/r-and-d-cell-edit', { data: rows[0] || {} });
  } catch (err) {
    console.error('Error loading r-and-d-cell edit:', err);
    res.render('admin/r-and-d-cell-edit', { data: {} });
  }
});
app.post('/admin/r-and-d-cell-update', async (req, res) => {
  try {
    const { id, vision, objectives, lab_details, thrust_areas } = req.body;

    if (id) {
      await promisePool.execute(
        'UPDATE r_d_cell_content SET vision=?, objectives=?, lab_details=?, thrust_areas=? WHERE id=?',
        [vision || '', objectives || '', lab_details || '', thrust_areas || '', id]
      );
    } else {
      await promisePool.execute(
        'INSERT INTO r_d_cell_content (vision, objectives, lab_details, thrust_areas) VALUES (?, ?, ?, ?)',
        [vision || '', objectives || '', lab_details || '', thrust_areas || '']
      );
    }

    res.redirect('/admin/r-and-d-cell');
  } catch (err) {
    console.error('Error updating r-and-d-cell:', err);
    res.status(500).send('Server error');
  }
});

app.get('/mou', async (req, res) => {
  try {
    const [rows] = await promisePool.execute('SELECT * FROM mou ORDER BY id DESC');
    res.render('mou', { data: rows || [] });
  } catch (err) {
    console.error('Error fetching MoU list:', err);
    res.status(500).send('Server error');
  }
});
app.get('/admin/mou', async (req, res) => {
  try {
    const [rows] = await promisePool.execute('SELECT * FROM mou ORDER BY id DESC');
    res.render('admin/mou', { data: rows || [] });
  } catch (err) {
    console.error('Error admin mou:', err);
    res.render('admin/mou', { data: [] });
  }
});
app.get('/admin/mou-edit/new', (req, res) => {
  res.render('admin/mou-edit', { data: {} });
});
app.get('/admin/mou-edit/:id', async (req, res) => {
  try {
    const id = req.params.id;
    const [rows] = await promisePool.execute('SELECT * FROM mou WHERE id = ?', [id]);
    res.render('admin/mou-edit', { data: rows[0] || {} });
  } catch (err) {
    console.error('Error loading mou edit:', err);
    res.render('admin/mou-edit', { data: {} });
  }
});
app.post('/admin/mou-update', async (req, res) => {
  try {
    const { id, category, name, year, duration, activities, participants } = req.body;
    if (id) {
      await promisePool.execute(
        'UPDATE mou SET category=?, name=?, year=?, duration=?, activities=?, participants=? WHERE id=?',
        [safe(category), safe(name), safe(year), safe(duration), safe(activities), safe(participants), id]
      );
    } else {
      await promisePool.execute(
        'INSERT INTO mou (category, name, year, duration, activities, participants) VALUES (?, ?, ?, ?, ?, ?)',
        [safe(category), safe(name), safe(year), safe(duration), safe(activities), safe(participants)]
      );
    }
    res.redirect('/admin/mou');
  } catch (err) {
    console.error('Error updating mou:', err);
    res.status(500).send('Server error');
  }
});
app.get('/admin/mou-delete/:id', async (req, res) => {
  try {
    await promisePool.execute('DELETE FROM mou WHERE id = ?', [req.params.id]);
    res.redirect('/admin/mou');
  } catch (err) {
    console.error('Error deleting mou:', err);
    res.status(500).send('Server error');
  }
});

app.get('/fdp', async (req, res) => {
  try {
    const [rows] = await promisePool.execute('SELECT * FROM fdp ORDER BY id DESC');
    res.render('fdp', { data: rows || [] });
  } catch (err) {
    console.error('Error fetching FDP list:', err);
    res.status(500).send('Server error');
  }
});
app.get('/admin/fdp', async (req, res) => {
  try {
    const [rows] = await promisePool.execute('SELECT * FROM fdp ORDER BY id DESC');
    res.render('admin/fdp', { title: 'Admin | FDP', data: rows });
  } catch (err) {
    console.error('Error loading FDP list:', err);
    res.render('admin/fdp', { title: 'Admin | FDP', data: [] });
  }
});
app.get(['/admin/fdp-edit', '/admin/fdp-edit/:id'], async (req, res) => {
  const { id } = req.params;
  let data = {};
  if (id) {
    const [rows] = await promisePool.execute('SELECT * FROM fdp WHERE id = ?', [id]);
    data = rows[0] || {};
  }
  res.render('admin/fdp-edit', { title: 'Edit FDP', data });
});
app.post('/admin/fdp-update', upload.single('photo'), async (req, res) => {
  try {
    const { id, teacher_name, program_title, duration, oldPhoto } = req.body;
    const photo = req.file ? `images/${req.file.filename}` : oldPhoto;

    if (id && id !== '') {
      // ✅ UPDATE EXISTING
      await promisePool.execute(
        `UPDATE fdp 
         SET teacher_name=?, program_title=?, duration=?, photo=? 
         WHERE id=?`,
        [teacher_name, program_title, duration, photo, id]
      );
    } else {
      // ✅ INSERT NEW
      await promisePool.execute(
        `INSERT INTO fdp (teacher_name, program_title, duration, photo) 
         VALUES (?, ?, ?, ?)`,
        [teacher_name, program_title, duration, photo]
      );
    }
    console.log('✅ FDP data updated successfully');
    res.redirect('/admin/fdp');
  } catch (err) {
    console.error('❌ Error updating FDP:', err);
    res.status(500).send('Server error');
  }
});
app.get('/admin/fdp-delete/:id', async (req, res) => {
  const { id } = req.params;
  await promisePool.execute('DELETE FROM fdp WHERE id = ?', [id]);
  res.redirect('/admin/fdp');
});

app.get('/sponsored-conferences', async (req, res) => {
  try {
    const [rows] = await promisePool.execute('SELECT * FROM sponsored_conferences ORDER BY id DESC');
    res.render('sponsored-conferences', { data: rows || [] });
  } catch (err) {
    console.error('Error fetching sponsored conferences:', err);
    res.status(500).send('Server error');
  }
});
app.get('/admin/sponsored-conferences', async (req, res) => {
  try {
    const [rows] = await promisePool.execute('SELECT * FROM sponsored_conferences ORDER BY id DESC');
    res.render('admin/sponsored-conferences', { data: rows || [] });
  } catch (err) {
    console.error('Error admin sponsored conferences:', err);
    res.render('admin/sponsored-conferences', { data: [] });
  }
});
app.get('/admin/sponsored-conferences-edit/new', (req, res) => {
  res.render('admin/sponsored-conferences-edit', { data: {} });
});
app.get('/admin/sponsored-conferences-edit/:id', async (req, res) => {
  try {
    const id = req.params.id;
    const [rows] = await promisePool.execute('SELECT * FROM sponsored_conferences WHERE id = ?', [id]);
    res.render('admin/sponsored-conferences-edit', { data: rows[0] || {} });
  } catch (err) {
    console.error('Error loading sponsored conferences edit:', err);
    res.render('admin/sponsored-conferences-edit', { data: {} });
  }
});
app.post('/admin/sponsored-conferences-update', upload.single('photo'), async (req, res) => {
  try {
    const { id, year, teacher_name, conference_name, professional_body, amount_support, old_photo } = req.body;
    const photo = req.file ? req.file.filename : old_photo || null;

    if (id) {
      await promisePool.execute(
        'UPDATE sponsored_conferences SET year=?, teacher_name=?, conference_name=?, professional_body=?, amount_support=?, photo=? WHERE id=?',
        [safe(year), safe(teacher_name), safe(conference_name), safe(professional_body), safe(amount_support), safe(photo), id]
      );
    } else {
      await promisePool.execute(
        'INSERT INTO sponsored_conferences (year, teacher_name, conference_name, professional_body, amount_support, photo) VALUES (?, ?, ?, ?, ?, ?)',
        [safe(year), safe(teacher_name), safe(conference_name), safe(professional_body), safe(amount_support), safe(photo)]
      );
    }
    res.redirect('/admin/sponsored-conferences');
  } catch (err) {
    console.error('Error updating sponsored conferences:', err);
    res.status(500).send('Server error');
  }
});
app.get('/admin/sponsored-conferences-delete/:id', async (req, res) => {
  try {
    await promisePool.execute('DELETE FROM sponsored_conferences WHERE id = ?', [req.params.id]);
    res.redirect('/admin/sponsored-conferences');
  } catch (err) {
    console.error('Error deleting sponsored conference:', err);
    res.status(500).send('Server error');
  }
});

app.get('/research-activities', async (req, res) => {
  try {
    const [db] = await promisePool.execute("SELECT DATABASE() AS db");
    console.log("FRONTEND DB:", db[0].db);

    const [phdHolders] = await promisePool.execute(
      "SELECT * FROM research_activities_phd_holders"
    );

    console.log("PHD HOLDERS COUNT:", phdHolders.length);

    const [phdRegistration] = await promisePool.execute(
      "SELECT * FROM research_activities_phd_registration"
    );

    const [supervisors] = await promisePool.execute(
      "SELECT * FROM research_activities_supervisors"
    );

    res.render('research-activities', {
      phdHolders,
      phdRegistration,
      supervisors
    });

  } catch (error) {
    console.error(error);
    res.send("Error");
  }
});
app.get('/admin/research-activities', async (req, res) => {
  try {
    const [phdHolders] = await promisePool.execute('SELECT * FROM research_activities_phd_holders ORDER BY id ASC');
    const [phdRegistration] = await promisePool.execute('SELECT * FROM research_activities_phd_registration ORDER BY id ASC');
    const [supervisors] = await promisePool.execute('SELECT * FROM research_activities_supervisors ORDER BY id ASC');
    res.render('admin/research-activities', { phdHolders, phdRegistration, supervisors });
  } catch (error) {
    console.error('Error loading research activities:', error);
    res.status(500).send('Error loading research activities');
  }
});
app.post('/admin/research-activities/phd-holders/add', async (req, res) => {
  try {
    const { department, faculty_name, designation, research_area, year_of_completion, university } = req.body;
    await promisePool.execute(
      'INSERT INTO research_activities_phd_holders (department, faculty_name, designation, research_area, year_of_completion, university) VALUES (?, ?, ?, ?, ?, ?)',
      [department, faculty_name, designation, research_area, year_of_completion, university]
    );
    res.redirect('/admin/research-activities');
  } catch (error) {
    console.error('Error adding Ph.D holder:', error);
    res.status(500).send('Error adding Ph.D holder');
  }
});
app.get('/admin/research-activities/phd-holders/edit/:id', async (req, res) => {
  try {
    if (req.params.id === 'new') {
      res.render('admin/research-activities-phd-holders-edit', { item: {} });
    } else {
      const [rows] = await promisePool.execute('SELECT * FROM research_activities_phd_holders WHERE id = ?', [req.params.id]);
      res.render('admin/research-activities-phd-holders-edit', { item: rows[0] || {} });
    }
  } catch (error) {
    console.error('Error loading Ph.D holder edit:', error);
    res.status(500).send('Error loading Ph.D holder edit');
  }
});
app.post('/admin/research-activities/phd-holders/update/:id', async (req, res) => {
  try {
    const { department, faculty_name, designation, research_area, year_of_completion, university } = req.body;
    await promisePool.execute(
      'UPDATE research_activities_phd_holders SET department = ?, faculty_name = ?, designation = ?, research_area = ?, year_of_completion = ?, university = ? WHERE id = ?',
      [department, faculty_name, designation, research_area, year_of_completion, university, req.params.id]
    );
    res.redirect('/admin/research-activities');
  } catch (error) {
    console.error('Error updating Ph.D holder:', error);
    res.status(500).send('Error updating Ph.D holder');
  }
});
app.get('/admin/research-activities/phd-holders/delete/:id', async (req, res) => {
  try {
    await promisePool.execute('DELETE FROM research_activities_phd_holders WHERE id = ?', [req.params.id]);
    res.redirect('/admin/research-activities');
  } catch (error) {
    console.error('Error deleting Ph.D holder:', error);
    res.status(500).send('Error deleting Ph.D holder');
  }
});
app.post('/admin/research-activities/phd-registration/add', async (req, res) => {
  try {
    const { department, faculty_name, research_title, university, registration_date, status } = req.body;
    await promisePool.execute(
      'INSERT INTO research_activities_phd_registration (department, faculty_name, research_title, university, registration_date, status) VALUES (?, ?, ?, ?, ?, ?)',
      [department, faculty_name, research_title, university, registration_date, status]
    );
    res.redirect('/admin/research-activities');
  } catch (error) {
    console.error('Error adding Ph.D registration:', error);
    res.status(500).send('Error adding Ph.D registration');
  }
});
app.get('/admin/research-activities/phd-registration/edit/:id', async (req, res) => {
  try {
    if (req.params.id === 'new') {
      res.render('admin/research-activities-phd-registration-edit', { item: {} });
    } else {
      const [rows] = await promisePool.execute('SELECT * FROM research_activities_phd_registration WHERE id = ?', [req.params.id]);
      res.render('admin/research-activities-phd-registration-edit', { item: rows[0] || {} });
    }
  } catch (error) {
    console.error('Error loading Ph.D registration edit:', error);
    res.status(500).send('Error loading Ph.D registration edit');
  }
});
app.post('/admin/research-activities/phd-registration/update/:id', async (req, res) => {
  try {
    const { department, faculty_name, research_title, university, registration_date, status } = req.body;
    await promisePool.execute(
      'UPDATE research_activities_phd_registration SET department = ?, faculty_name = ?, research_title = ?, university = ?, registration_date = ?, status = ? WHERE id = ?',
      [department, faculty_name, research_title, university, registration_date, status, req.params.id]
    );
    res.redirect('/admin/research-activities');
  } catch (error) {
    console.error('Error updating Ph.D registration:', error);
    res.status(500).send('Error updating Ph.D registration');
  }
});
app.get('/admin/research-activities/phd-registration/delete/:id', async (req, res) => {
  try {
    await promisePool.execute('DELETE FROM research_activities_phd_registration WHERE id = ?', [req.params.id]);
    res.redirect('/admin/research-activities');
  } catch (error) {
    console.error('Error deleting Ph.D registration:', error);
    res.status(500).send('Error deleting Ph.D registration');
  }
});
app.post('/admin/research-activities/supervisors/add', async (req, res) => {
  try {
    const { department, faculty_name, university, supervisor_ref_no } = req.body;
    await promisePool.execute(
      'INSERT INTO research_activities_supervisors (department, faculty_name, university, supervisor_ref_no) VALUES (?, ?, ?, ?)',
      [department, faculty_name, university, supervisor_ref_no]
    );
    res.redirect('/admin/research-activities');
  } catch (error) {
    console.error('Error adding supervisor:', error);
    res.status(500).send('Error adding supervisor');
  }
});
app.get('/admin/research-activities/supervisors/edit/:id', async (req, res) => {
  try {
    if (req.params.id === 'new') {
      res.render('admin/research-activities-supervisors-edit', { item: {} });
    } else {
      const [rows] = await promisePool.execute('SELECT * FROM research_activities_supervisors WHERE id = ?', [req.params.id]);
      res.render('admin/research-activities-supervisors-edit', { item: rows[0] || {} });
    }
  } catch (error) {
    console.error('Error loading supervisor edit:', error);
    res.status(500).send('Error loading supervisor edit');
  }
});
app.post('/admin/research-activities/supervisors/update/:id', async (req, res) => {
  try {
    const { department, faculty_name, university, supervisor_ref_no } = req.body;
    await promisePool.execute(
      'UPDATE research_activities_supervisors SET department = ?, faculty_name = ?, university = ?, supervisor_ref_no = ? WHERE id = ?',
      [department, faculty_name, university, supervisor_ref_no, req.params.id]
    );
    res.redirect('/admin/research-activities');
  } catch (error) {
    console.error('Error updating supervisor:', error);
    res.status(500).send('Error updating supervisor');
  }
});
app.get('/admin/research-activities/supervisors/delete/:id', async (req, res) => {
  try {
    await promisePool.execute('DELETE FROM research_activities_supervisors WHERE id = ?', [req.params.id]);
    res.redirect('/admin/research-activities');
  } catch (error) {
    console.error('Error deleting supervisor:', error);
    res.status(500).send('Error deleting supervisor');
  }
});

app.get('/publications', async (req, res) => {
  try {
    const [journals] = await promisePool.execute(
      "SELECT * FROM publications_journals ORDER BY year DESC, id DESC"
    );

    const [conferences] = await promisePool.execute(
      "SELECT * FROM publications_conferences ORDER BY year DESC, id DESC"
    );

    res.render('publications', {
      journals,
      conferences
    });
  } catch (error) {
    console.error('Error loading publications frontend:', error);
    res.render('publications', {
      journals: [],
      conferences: []
    });
  }
});
app.get('/admin/publications', async (req, res) => {
  try {
    const [journals] = await promisePool.execute('SELECT * FROM publications_journals ORDER BY year DESC, id DESC');
    const [conferences] = await promisePool.execute('SELECT * FROM publications_conferences ORDER BY year DESC, id DESC');
    res.render('admin/publications', { journals, conferences });
  } catch (error) {
    console.error('Error loading publications:', error);
    res.status(500).send('Error loading publications');
  }
});
app.post('/admin/publications/journals/add', async (req, res) => {
  try {
    const { department, authors, title, journal_name, publisher, year, volume_issue, page_numbers, indexed } = req.body;
    await promisePool.execute(
      'INSERT INTO publications_journals (department, authors, title, journal_name, publisher, year, volume_issue, page_numbers, indexed) VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?)',
      [department, authors, title, journal_name, publisher, year, volume_issue, page_numbers, indexed]
    );
    res.redirect('/admin/publications');
  } catch (error) {
    console.error('Error adding journal publication:', error);
    res.status(500).send('Error adding journal publication');
  }
});
app.get('/admin/publications/journals/edit/:id', async (req, res) => {
  try {
    if (req.params.id === 'new') {
      res.render('admin/publications-journals-edit', { item: {} });
    } else {
      const [rows] = await promisePool.execute('SELECT * FROM publications_journals WHERE id = ?', [req.params.id]);
      res.render('admin/publications-journals-edit', { item: rows[0] || {} });
    }
  } catch (error) {
    console.error('Error loading journal edit:', error);
    res.status(500).send('Error loading journal edit');
  }
});
app.post('/admin/publications/journals/update/:id', async (req, res) => {
  try {
    const { department, authors, title, journal_name, publisher, year, volume_issue, page_numbers, indexed } = req.body;
    await promisePool.execute(
      'UPDATE publications_journals SET department = ?, authors = ?, title = ?, journal_name = ?, publisher = ?, year = ?, volume_issue = ?, page_numbers = ?, indexed = ? WHERE id = ?',
      [department, authors, title, journal_name, publisher, year, volume_issue, page_numbers, indexed, req.params.id]
    );
    res.redirect('/admin/publications');
  } catch (error) {
    console.error('Error updating journal publication:', error);
    res.status(500).send('Error updating journal publication');
  }
});
app.get('/admin/publications/journals/delete/:id', async (req, res) => {
  try {
    await promisePool.execute('DELETE FROM publications_journals WHERE id = ?', [req.params.id]);
    res.redirect('/admin/publications');
  } catch (error) {
    console.error('Error deleting journal publication:', error);
    res.status(500).send('Error deleting journal publication');
  }
});
app.post('/admin/publications/conferences/add', async (req, res) => {
  try {
    const { department, authors, title, conference_name, publisher, year, volume_issue, page_numbers, indexed } = req.body;
    await promisePool.execute(
      'INSERT INTO publications_conferences (department, authors, title, conference_name, publisher, year, volume_issue, page_numbers, indexed) VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?)',
      [department, authors, title, conference_name, publisher, year, volume_issue, page_numbers, indexed]
    );
    res.redirect('/admin/publications');
  } catch (error) {
    console.error('Error adding conference publication:', error);
    res.status(500).send('Error adding conference publication');
  }
});
app.get('/admin/publications/conferences/edit/:id', async (req, res) => {
  try {
    if (req.params.id === 'new') {
      res.render('admin/publications-conferences-edit', { item: {} });
    } else {
      const [rows] = await promisePool.execute('SELECT * FROM publications_conferences WHERE id = ?', [req.params.id]);
      res.render('admin/publications-conferences-edit', { item: rows[0] || {} });
    }
  } catch (error) {
    console.error('Error loading conference edit:', error);
    res.status(500).send('Error loading conference edit');
  }
});
app.post('/admin/publications/conferences/update/:id', async (req, res) => {
  try {
    const { department, authors, title, conference_name, publisher, year, volume_issue, page_numbers, indexed } = req.body;
    await promisePool.execute(
      'UPDATE publications_conferences SET department = ?, authors = ?, title = ?, conference_name = ?, publisher = ?, year = ?, volume_issue = ?, page_numbers = ?, indexed = ? WHERE id = ?',
      [department, authors, title, conference_name, publisher, year, volume_issue, page_numbers, indexed, req.params.id]
    );
    res.redirect('/admin/publications');
  } catch (error) {
    console.error('Error updating conference publication:', error);
    res.status(500).send('Error updating conference publication');
  }
});
app.get('/admin/publications/conferences/delete/:id', async (req, res) => {
  try {
    await promisePool.execute('DELETE FROM publications_conferences WHERE id = ?', [req.params.id]);
    res.redirect('/admin/publications');
  } catch (error) {
    console.error('Error deleting conference publication:', error);
    res.status(500).send('Error deleting conference publication');
  }
});

app.get('/patent', async (req, res) => {
  const [rows] = await promisePool.execute("SELECT * FROM patents ORDER BY id DESC");
  res.render('patent', { patents: rows });
});
app.get('/admin/patent', async (req, res) => {
  try {
    const [patents] = await promisePool.execute('SELECT * FROM patents ORDER BY id DESC');
    res.render('admin/patent', { patents });
  } catch (error) {
    console.error('Error loading patents:', error);
    res.status(500).send('Error loading patents');
  }
});
app.post('/admin/patent/add', async (req, res) => {
  try {
    const { department, faculty_name, patent_title, application_no, status } = req.body;
    await promisePool.execute(
      'INSERT INTO patents (department, faculty_name, patent_title, application_no, status) VALUES (?, ?, ?, ?, ?)',
      [department, faculty_name, patent_title, application_no, status]
    );
    res.redirect('/admin/patent');
  } catch (error) {
    console.error('Error adding patent:', error);
    res.status(500).send('Error adding patent');
  }
});
app.get('/admin/patent/edit/:id', async (req, res) => {
  try {
    if (req.params.id === 'new') {
      res.render('admin/patent-edit', { item: {} });
    } else {
      const [rows] = await promisePool.execute('SELECT * FROM patents WHERE id = ?', [req.params.id]);
      res.render('admin/patent-edit', { item: rows[0] || {} });
    }
  } catch (error) {
    console.error('Error loading patent edit:', error);
    res.status(500).send('Error loading patent edit');
  }
});
app.post('/admin/patent/update/:id', async (req, res) => {
  try {
    const { department, faculty_name, patent_title, application_no, status } = req.body;
    await promisePool.execute(
      'UPDATE patents SET department = ?, faculty_name = ?, patent_title = ?, application_no = ?, status = ? WHERE id = ?',
      [department, faculty_name, patent_title, application_no, status, req.params.id]
    );
    res.redirect('/admin/patent');
  } catch (error) {
    console.error('Error updating patent:', error);
    res.status(500).send('Error updating patent');
  }
});
app.get('/admin/patent/delete/:id', async (req, res) => {
  try {
    await promisePool.execute('DELETE FROM patents WHERE id = ?', [req.params.id]);
    res.redirect('/admin/patent');
  } catch (error) {
    console.error('Error deleting patent:', error);
    res.status(500).send('Error deleting patent');
  }
});

app.get('/consultancy', async (req, res) => {
  try {
    const [rows] = await promisePool.execute(
      "SELECT * FROM consultancy_work ORDER BY start_date DESC, id DESC"
    );

    res.render('consultancy', { consultancies: rows });
  } catch (error) {
    console.error('Error loading consultancy frontend:', error);
    res.render('consultancy', { consultancies: [] });
  }
});
app.get('/admin/consultancy', async (req, res) => {
  try {
    const [consultancies] = await promisePool.execute('SELECT * FROM consultancy_work ORDER BY start_date DESC, id DESC');
    res.render('admin/consultancy', { consultancies });
  } catch (error) {
    console.error('Error loading consultancy:', error);
    res.status(500).send('Error loading consultancy');
  }
});
app.post('/admin/consultancy/add', async (req, res) => {
  try {
    const { department, client_name, title, start_date, end_date, cost } = req.body;
    await promisePool.execute(
      'INSERT INTO consultancy_work (department, client_name, title, start_date, end_date, cost) VALUES (?, ?, ?, ?, ?, ?)',
      [department, client_name, title, start_date || null, end_date || null, cost || null]
    );
    res.redirect('/admin/consultancy');
  } catch (error) {
    console.error('Error adding consultancy:', error);
    res.status(500).send('Error adding consultancy');
  }
});
app.get('/admin/consultancy/edit/:id', async (req, res) => {
  try {
    if (req.params.id === 'new') {
      res.render('admin/consultancy-edit', { item: {} });
    } else {
      const [rows] = await promisePool.execute('SELECT * FROM consultancy_work WHERE id = ?', [req.params.id]);
      res.render('admin/consultancy-edit', { item: rows[0] || {} });
    }
  } catch (error) {
    console.error('Error loading consultancy edit:', error);
    res.status(500).send('Error loading consultancy edit');
  }
});
app.post('/admin/consultancy/update/:id', async (req, res) => {
  try {
    const { department, client_name, title, start_date, end_date, cost } = req.body;
    await promisePool.execute(
      'UPDATE consultancy_work SET department = ?, client_name = ?, title = ?, start_date = ?, end_date = ?, cost = ? WHERE id = ?',
      [department, client_name, title, start_date || null, end_date || null, cost || null, req.params.id]
    );
    res.redirect('/admin/consultancy');
  } catch (error) {
    console.error('Error updating consultancy:', error);
    res.status(500).send('Error updating consultancy');
  }
});
app.get('/admin/consultancy/delete/:id', async (req, res) => {
  try {
    await promisePool.execute('DELETE FROM consultancy_work WHERE id = ?', [req.params.id]);
    res.redirect('/admin/consultancy');
  } catch (error) {
    console.error('Error deleting consultancy:', error);
    res.status(500).send('Error deleting consultancy');
  }
});

const cbpStorage = multer.diskStorage({
    destination: (req, file, cb) => {
        cb(null, "public/pdf/");
    },
    filename: (req, file, cb) => {
        cb(null, Date.now() + "-" + file.originalname);
    }
});
const cbpUpload = multer({ storage: cbpStorage });
app.get("/capacity-building-programs", (req, res) => {
    db.query("SELECT * FROM capacity_building_programs ORDER BY id DESC", (err, rows) => {
        if (err) return res.send("Error loading page");
        res.render("capacity-building-programs", { data: rows });
    });
});
app.get("/admin/capacity-building-programs-list", (req, res) => {
    db.query("SELECT * FROM capacity_building_programs ORDER BY id DESC", (err, rows) => {
        if (err) return res.send("Error loading admin list");
        res.render("admin/capacity-building-programs-list", { data: rows });
    });
});
app.get("/admin/capacity-building-programs", (req, res) => {
    res.render("admin/capacity-building-programs");
});
app.post("/admin/capacity-building-programs",
    cbpUpload.single("pdf_link"),
    (req, res) => {

        const { pdf_title } = req.body;
        const pdf_link = req.file.filename;

        db.query(
            "INSERT INTO capacity_building_programs (pdf_title, pdf_link) VALUES (?, ?)",
            [pdf_title, pdf_link],
            () => res.redirect("/admin/capacity-building-programs-list")
        );
    }
);
app.get("/admin/capacity-building-programs-edit/:id", (req, res) => {
    db.query("SELECT * FROM capacity_building_programs WHERE id=?", [req.params.id], (err, rows) => {
        if (err) return res.send("Error loading edit page");
        res.render("admin/capacity-building-programs-edit", { data: rows[0] });
    });
});
app.post("/admin/capacity-building-programs-edit/:id",
    cbpUpload.single("pdf_link"),
    (req, res) => {

        const { pdf_title, oldPdf } = req.body;
        const pdf_link = req.file ? req.file.filename : oldPdf;

        db.query(
            "UPDATE capacity_building_programs SET pdf_title=?, pdf_link=? WHERE id=?",
            [pdf_title, pdf_link, req.params.id],
            () => res.redirect("/admin/capacity-building-programs-list")
        );
    }
);
app.get("/admin/capacity-building-programs-delete/:id", (req, res) => {
    db.query("DELETE FROM capacity_building_programs WHERE id=?", [req.params.id], () => {
        res.redirect("/admin/capacity-building-programs-list");
    });
});

app.get('/coe/home', async (req, res) => {
  try {
    const [home] = await promisePool.execute(
      'SELECT * FROM coehome ORDER BY id DESC LIMIT 1'
    );

    let controllers = [];
    let office = [];

    if (home.length > 0) {
      const [c] = await promisePool.execute(
        'SELECT * FROM coehome_controllers WHERE home_id = ?',
        [home[0].id]
      );
      const [o] = await promisePool.execute(
        'SELECT * FROM coehome_office WHERE home_id = ?',
        [home[0].id]
      );

      controllers = c;
      office = o;
    }

    res.render('ui/home', {
      home: home[0] || {},
      controllers,
      office,
    });
  } catch (err) {
    console.error('Error loading COE Home page:', err);
    res.status(500).send('Server error');
  }
});
app.get('/home', async (req, res) => {
  try {
    // get latest coe home record
    const [homeRows] = await promisePool.execute('SELECT * FROM coehome ORDER BY id DESC LIMIT 1');
    const home = homeRows.length ? homeRows[0] : {};

    // load controllers & office members for that home_id (use 0 if no home)
    const homeId = home.id || 0;
    const [controllers] = await promisePool.execute(
      'SELECT * FROM coehome_controllers WHERE home_id = ? ORDER BY controllers_si_id ASC',
      [homeId]
    );
    const [office] = await promisePool.execute(
      'SELECT * FROM coehome_office WHERE home_id = ? ORDER BY office_members_si_id ASC',
      [homeId]
    );

    // render views/home.ejs (make sure home.ejs exists in views/ or change name accordingly)
    res.render('home', { home, controllers, office });
  } catch (err) {
    console.error('Error loading /home page:', err);
    res.status(500).send('Server error');
  }
});
app.get('/admin/home', async (req, res) => {
  try {
    const [rows] = await promisePool.execute(
      'SELECT * FROM coehome ORDER BY id DESC'
    );
    res.render('admin/home', { data: rows || [] });
  } catch (err) {
    console.error('Error loading admin home list:', err);
    res.render('admin/home', { data: [] });
  }
});
app.get(['/admin/home-edit', '/admin/home-edit/:id'], async (req, res) => {
  try {
    const id = req.params.id;
    let home = {},
      controllers = [],
      office = [];

    if (id) {
      const [h] = await promisePool.execute('SELECT * FROM coehome WHERE id = ?', [id]);
      const [c] = await promisePool.execute('SELECT * FROM coehome_controllers WHERE home_id = ?', [id]);
      const [o] = await promisePool.execute('SELECT * FROM coehome_office WHERE home_id = ?', [id]);
      home = h[0] || {};
      controllers = c;
      office = o;
    }

    res.render('admin/home-edit', { home, controllers, office });
  } catch (err) {
    console.error('Error loading home-edit page:', err);
    res.render('admin/home-edit', { home: {}, controllers: [], office: [] });
  }
});
app.post('/admin/home-update', async (req, res) => {
  try {
    const { id, title, content, contact_phone, contact_landline, contact_mail_id } = req.body;

    let homeId = id;

    if (id) {
      await promisePool.execute(
        'UPDATE coehome SET title=?, content=?, contact_phone=?, contact_landline=?, contact_mail_id=? WHERE id=?',
        [title, content, contact_phone, contact_landline, contact_mail_id, id]
      );
    } else {
      const [result] = await promisePool.execute(
        'INSERT INTO coehome (title, content, contact_phone, contact_landline, contact_mail_id) VALUES (?, ?, ?, ?, ?)',
        [title, content, contact_phone, contact_landline, contact_mail_id]
      );
      homeId = result.insertId;
    }

    await promisePool.execute('DELETE FROM coehome_controllers WHERE home_id = ?', [homeId]);
    await promisePool.execute('DELETE FROM coehome_office WHERE home_id = ?', [homeId]);

    // Save controllers
    const controllers = JSON.parse(req.body.controllers_json || '[]');
    for (const c of controllers) {
      await promisePool.execute(
        `INSERT INTO coehome_controllers (home_id, controllers_si_id, controllers_name, controllers_designation, controllers_contact)
         VALUES (?, ?, ?, ?, ?)`,
        [
          homeId,
          c.controllers_si_id,
          c.controllers_name,
          c.controllers_designation,
          c.controllers_contact,
        ]
      );
    }

    // Save office members
    const office = JSON.parse(req.body.office_json || '[]');
    for (const o of office) {
      await promisePool.execute(
        `INSERT INTO coehome_office (home_id, office_members_si_id, office_members_name, office_members_designation)
         VALUES (?, ?, ?, ?)`,
        [
          homeId,
          o.office_members_si_id,
          o.office_members_name,
          o.office_members_designation,
        ]
      );
    }

    res.redirect('/admin/home');
  } catch (err) {
    console.error('Error updating coehome:', err);
    res.status(500).send('Server error');
  }
});
app.get('/admin/home-delete/:id', async (req, res) => {
  try {
    await promisePool.execute('DELETE FROM coehome WHERE id = ?', [req.params.id]);
    res.redirect('/admin/home');
  } catch (err) {
    console.error('Error deleting coehome record:', err);
    res.status(500).send('Server error');
  }
});

app.get('/timetable', async (req, res) => {
  try {
    const [rows] = await promisePool.execute(
      "SELECT * FROM timetable ORDER BY id ASC"
    );
    res.render('timetable', { data: rows });
  } catch (err) {
    console.error(err);
    res.render('timetable', { data: [] });
  }
});
app.get('/admin/timetable', async (req, res) => {
  const [rows] = await promisePool.execute(
    "SELECT * FROM timetable ORDER BY id DESC"
  );
  res.render('admin/timetable', { data: rows });
});
app.get('/admin/timetable-edit', (req, res) => {
  res.render('admin/timetable-edit', {
    data: {},
    isEdit: false
  });
});
app.get('/admin/timetable-edit/:id', async (req, res) => {
  const [rows] = await promisePool.execute(
    "SELECT * FROM timetable WHERE id = ?",
    [req.params.id]
  );

  res.render('admin/timetable-edit', {
    data: rows[0] || {},
    isEdit: true
  });
});
app.post('/admin/timetable-update',
  uploadPDF.single('pdf_file'), // ✅ CORRECT MULTER
  async (req, res) => {

    const { id, main_title, pdf_title, oldFile } = req.body;

    let pdf_link = oldFile;

    if (req.file) {
      pdf_link = req.file.originalname;
    }

    if (id) {
      await promisePool.execute(
        "UPDATE timetable SET main_title=?, pdf_title=?, pdf_link=? WHERE id=?",
        [main_title, pdf_title, pdf_link, id]
      );
    } else {
      await promisePool.execute(
        "INSERT INTO timetable (main_title, pdf_title, pdf_link) VALUES (?,?,?)",
        [main_title, pdf_title, pdf_link]
      );
    }

    res.redirect('/admin/timetable');
  }
);
app.get('/admin/timetable-delete/:id', async (req, res) => {
  await promisePool.execute(
    "DELETE FROM timetable WHERE id=?",
    [req.params.id]
  );
  res.redirect('/admin/timetable');
});

app.get('/results', async (req, res) => {
  try {
    const [results] = await promisePool.execute("SELECT * FROM results ORDER BY id DESC");
    res.render('results', {
      title: "Results",
      results: results 
    });
  } catch (error) {
    console.error("Error loading results page:", error);
    res.render("results", { title: "Results", results: [] });
  }
});
app.get('/admin/results', async (req, res) => {
  try {
    const [rows] = await promisePool.execute('SELECT * FROM results ORDER BY id DESC');
    res.render('admin/results', { data: rows || [] });
  } catch (err) {
    console.error('Error admin results:', err);
    res.render('admin/results', { data: [] });
  }
});
app.get('/admin/results-edit/new', (req, res) => {
  res.render('admin/results-edit', { data: {} });
});
app.get('/admin/results-edit/:id', async (req, res) => {
  try {
    const [rows] = await promisePool.execute('SELECT * FROM results WHERE id = ?', [req.params.id]);
    res.render('admin/results-edit', { data: rows[0] || {} });
  } catch (err) {
    console.error('Error loading results edit:', err);
    res.render('admin/results-edit', { data: {} });
  }
});
app.post('/admin/results-update', async (req, res) => {
  try {
    const { id, image_path, link_url, alt_text } = req.body;
    if (id) {
      await promisePool.execute(
        'UPDATE results SET image_path=?, link_url=?, alt_text=? WHERE id=?',
        [image_path, link_url, alt_text, id]
      );
    } else {
      await promisePool.execute(
        'INSERT INTO results (image_path, link_url, alt_text) VALUES (?, ?, ?)',
        [image_path, link_url, alt_text]
      );
    }
    res.redirect('/admin/results');
  } catch (err) {
    console.error('Error updating results:', err);
    res.status(500).send('Server error');
  }
});
app.get('/admin/results-delete/:id', async (req, res) => {
  try {
    await promisePool.execute('DELETE FROM results WHERE id = ?', [req.params.id]);
    res.redirect('/admin/results');
  } catch (err) {
    console.error('Error deleting results:', err);
    res.status(500).send('Server error');
  }
});

app.get('/revaluation', async (req, res) => {
  try {
    const [revaluationData] = await promisePool.execute(
      "SELECT * FROM revaluation ORDER BY id DESC"
    );
    res.render('revaluation', {
      title: "Revaluation",
      revaluationData: revaluationData   // 🔥 MUST SEND THIS
    });
  } catch (error) {
    console.error("Error loading revaluation:", error);
    res.render('revaluation', { title: "Revaluation", revaluationData: [] });
  }
});
app.get('/admin/revaluation', async (req, res) => {
  try {
    const [rows] = await promisePool.execute('SELECT * FROM revaluation ORDER BY id DESC');
    res.render('admin/revaluation', { data: rows || [] });
  } catch (err) {
    console.error('Error admin revaluation:', err);
    res.render('admin/revaluation', { data: [] });
  }
});
app.get('/admin/revaluation-edit/new', (req, res) => {
  res.render('admin/revaluation-edit', { data: {} });
});
app.get('/admin/revaluation-edit/:id', async (req, res) => {
  try {
    const [rows] = await promisePool.execute('SELECT * FROM revaluation WHERE id = ?', [req.params.id]);
    res.render('admin/revaluation-edit', { data: rows[0] || {} });
  } catch (err) {
    console.error('Error loading revaluation edit:', err);
    res.render('admin/revaluation-edit', { data: {} });
  }
});
app.post('/admin/revaluation-update', async (req, res) => {
  try {
    const { id, image_path, link_url, alt_text } = req.body;
    if (id) {
      await promisePool.execute(
        'UPDATE revaluation SET image_path=?, link_url=?, alt_text=? WHERE id=?',
        [image_path, link_url, alt_text, id]
      );
    } else {
      await promisePool.execute(
        'INSERT INTO revaluation (image_path, link_url, alt_text) VALUES (?, ?, ?)',
        [image_path, link_url, alt_text]
      );
    }
    res.redirect('/admin/revaluation');
  } catch (err) {
    console.error('Error updating revaluation:', err);
    res.status(500).send('Server error');
  }
});
app.get('/admin/revaluation-delete/:id', async (req, res) => {
  try {
    await promisePool.execute('DELETE FROM revaluation WHERE id = ?', [req.params.id]);
    res.redirect('/admin/revaluation');
  } catch (err) {
    console.error('Error deleting revaluation:', err);
    res.status(500).send('Server error');
  }
});

app.get('/forms', async (req, res) => {
  try {
    const [forms] = await promisePool.execute(
      "SELECT * FROM forms ORDER BY id ASC"
    );
    res.render('forms', {
      title: "Forms",
      forms: forms   // 🔥 MUST SEND THIS
    });
  } catch (error) {
    console.error("Error loading forms:", error);
    res.render('forms', { title: "Forms", forms: [] });
  }
});
app.get('/admin/forms', async (req, res) => {
  try {
    const [rows] = await promisePool.execute('SELECT * FROM forms ORDER BY id DESC');
    res.render('admin/forms', { data: rows || [] });
  } catch (err) {
    console.error('Error admin forms:', err);
    res.render('admin/forms', { data: [] });
  }
});
app.get('/admin/forms-edit/new', (req, res) => {
  res.render('admin/forms-edit', { data: {} });
});
app.get('/admin/forms-edit/:id', async (req, res) => {
  try {
    const [rows] = await promisePool.execute('SELECT * FROM forms WHERE id = ?', [req.params.id]);
    res.render('admin/forms-edit', { data: rows[0] || {} });
  } catch (err) {
    console.error('Error loading forms edit:', err);
    res.render('admin/forms-edit', { data: {} });
  }
});
app.post('/admin/forms-update', async (req, res) => {
  try {
    const { id, form_no, form_title, form_file } = req.body;
    if (id) {
      await promisePool.execute(
        'UPDATE forms SET form_no=?, form_title=?, form_file=? WHERE id=?',
        [form_no, form_title, form_file, id]
      );
    } else {
      await promisePool.execute(
        'INSERT INTO forms (form_no, form_title, form_file) VALUES (?, ?, ?)',
        [form_no, form_title, form_file]
      );
    }
    res.redirect('/admin/forms');
  } catch (err) {
    console.error('Error updating forms:', err);
    res.status(500).send('Server error');
  }
});
app.get('/admin/forms-delete/:id', async (req, res) => {
  try {
    await promisePool.execute('DELETE FROM forms WHERE id = ?', [req.params.id]);
    res.redirect('/admin/forms');
  } catch (err) {
    console.error('Error deleting forms:', err);
    res.status(500).send('Server error');
  }
});

app.get('/campus-facilities', async (req, res) => {
  try {
    const [content] = await promisePool.execute(
      "SELECT * FROM campus_facilities_content ORDER BY id DESC LIMIT 1"
    );
    const [listItems] = await promisePool.execute(
      "SELECT * FROM campus_facilities_list ORDER BY id ASC"
    );

    res.render('campus-facilities', {
      content: content[0] || {},
      list: listItems || []
    });
  } catch (err) {
    console.error(err);
    res.render('campus-facilities', { content: {}, list: [] });
  }
});
app.get('/admin/campus-facilities', async (req, res) => {
  try {
    const [contentRows] = await promisePool.execute(
      "SELECT * FROM campus_facilities_content ORDER BY id DESC"
    );

    const [listRows] = await promisePool.execute(
      "SELECT * FROM campus_facilities_list ORDER BY id ASC"
    );

    res.render('admin/campus-facilities', {
      content: contentRows,
      list: listRows
    });

  } catch (err) {
    console.error(err);
    res.render('admin/campus-facilities', { content: [], list: [] });
  }
});
app.get('/admin/campus-facilities-edit/new', (req, res) => {
  res.render('admin/campus-facilities-edit', { data: {} });
});
app.get('/admin/campus-facilities-edit/:id', async (req, res) => {
  const [rows] = await promisePool.execute(
    "SELECT * FROM campus_facilities_content WHERE id=?",
    [req.params.id]
  );
  res.render('admin/campus-facilities-edit', { data: rows[0] || {} });
});
app.post('/admin/campus-facilities-update', async (req, res) => {
  const { id, title, content } = req.body;

  if (id) {
    await promisePool.execute(
      "UPDATE campus_facilities_content SET title=?, content=? WHERE id=?",
      [title, content, id]
    );
  } else {
    await promisePool.execute(
      "INSERT INTO campus_facilities_content (title, content) VALUES (?, ?)",
      [title, content]
    );
  }
  res.redirect('/admin/campus-facilities');
});
app.get('/admin/campus-facilities-list', async (req, res) => {
  const [rows] = await promisePool.execute(
    "SELECT * FROM campus_facilities_list ORDER BY id ASC"
  );
  res.render('admin/campus-facilities-list', { data: rows });
});
app.get('/admin/campus-facilities-list-edit/new', (req, res) => {
  res.render('admin/campus-facilities-list-edit', { data: {} });
});
app.get('/admin/campus-facilities-list-edit/:id', async (req, res) => {
  const [rows] = await promisePool.execute(
    "SELECT * FROM campus_facilities_list WHERE id = ?",
    [req.params.id]
  );
  res.render('admin/campus-facilities-list-edit', { data: rows[0] || {} });
});
app.post('/admin/campus-facilities-list-update', async (req, res) => {
  const { id, title, content } = req.body;

  if (id) {
    await promisePool.execute(
      "UPDATE campus_facilities_list SET title=?, content=? WHERE id=?",
      [title, content, id]
    );
  } else {
    await promisePool.execute(
      "INSERT INTO campus_facilities_list (title, content) VALUES (?, ?)",
      [title, content]
    );
  }

  res.redirect('/admin/campus-facilities-list');
});
app.get('/admin/campus-facilities-list-delete/:id', async (req, res) => {
  await promisePool.execute(
    "DELETE FROM campus_facilities_list WHERE id = ?",
    [req.params.id]
  );
  res.redirect('/admin/campus-facilities-list');
});

app.get("/laboratory", async (req, res) => {
  try {
    // MAIN PAGE INFO
    const [pageInfo] = await promisePool.execute(
      "SELECT * FROM laboratory_page_info ORDER BY id DESC LIMIT 1"
    );

    // GALLERY IMAGES
    const [gallery] = await promisePool.execute(
      "SELECT * FROM laboratory_gallery ORDER BY id DESC"
    );

    // LAB SECTIONS
    const [sections] = await promisePool.execute(
      "SELECT * FROM laboratory_sections ORDER BY id ASC"
    );

    res.render("laboratory", {
      pageInfo: pageInfo[0] || {},
      gallery,
      sections
    });

  } catch (err) {
    console.error("Laboratory Page Error:", err);
    res.render("laboratory", {
      pageInfo: {},
      gallery: [],
      sections: []
    });
  }
});
app.get('/admin/laboratory', async (req, res) => {
  try {
    // MAIN PAGE INFO
    const [pageInfo] = await promisePool.execute(
      "SELECT * FROM laboratory_page_info ORDER BY id DESC"
    );

    // SECTIONS
    const [sections] = await promisePool.execute(
      "SELECT * FROM laboratory_sections ORDER BY id ASC"
    );

    // GALLERY
    const [gallery] = await promisePool.execute(
      "SELECT * FROM laboratory_gallery ORDER BY id DESC"
    );

    res.render("admin/laboratory", {
      pageInfo,
      sections,
      gallery
    });

  } catch (err) {
    console.error(err);
    res.render("admin/laboratory", {
      pageInfo: [],
      sections: [],
      gallery: []
    });
  }
});
app.get('/admin/laboratory-edit/new', (req, res) => {
  res.render('admin/laboratory-edit', { data: {} });
});
app.get('/admin/laboratory-edit/:id', async (req, res) => {
  const [rows] = await promisePool.execute(
    "SELECT * FROM laboratory_page_info WHERE id = ?",
    [req.params.id]
  );
  res.render('admin/laboratory-edit', { data: rows[0] || {} });
});
app.post('/admin/laboratory-update', async (req, res) => {
  const { id, title, subtitle, content } = req.body;

  if (id) {
    await promisePool.execute(
      "UPDATE laboratory_page_info SET title=?, subtitle=?, content=? WHERE id=?",
      [title, subtitle, content, id]
    );
  } else {
    await promisePool.execute(
      "INSERT INTO laboratory_page_info (title, subtitle, content) VALUES (?, ?, ?)",
      [title, subtitle, content]
    );
  }

  res.redirect('/admin/laboratory');
});
app.get('/admin/laboratory-sections', async (req, res) => {
  const [rows] = await promisePool.execute(
    "SELECT * FROM laboratory_sections ORDER BY id ASC"
  );

  res.render("admin/laboratory-sections", { sections: rows });
});
app.get('/admin/laboratory-section-edit/new', (req, res) => {
  res.render("admin/laboratory-section-edit", { data: {} });
});
app.get('/admin/laboratory-section-edit/:id', async (req, res) => {
  const [rows] = await promisePool.execute(
    "SELECT * FROM laboratory_sections WHERE id=?",
    [req.params.id]
  );
  res.render("admin/laboratory-section-edit", { data: rows[0] || {} });
});
app.post('/admin/laboratory-section-update', async (req, res) => {
  const { id, department, labs } = req.body;

  if (id) {
    await promisePool.execute(
      "UPDATE laboratory_sections SET department=?, labs=? WHERE id=?",
      [department, labs, id]
    );
  } else {
    await promisePool.execute(
      "INSERT INTO laboratory_sections (department, labs) VALUES (?, ?)",
      [department, labs]
    );
  }

  res.redirect("/admin/laboratory-sections");
});
app.get('/admin/laboratory-section-delete/:id', async (req, res) => {
  await promisePool.execute(
    "DELETE FROM laboratory_sections WHERE id=?",
    [req.params.id]
  );
  res.redirect("/admin/laboratory-sections");
});
app.get("/admin/laboratory-gallery-upload", (req, res) => {
  res.render("admin/laboratory-gallery-upload");
});
app.post(
  "/admin/laboratory-gallery-upload",
  upload.single("image_file"),
  async (req, res) => {
    await promisePool.execute(
      "INSERT INTO laboratory_gallery (image_path) VALUES (?)",
      ["/images/" + req.file.filename]
    );
    res.redirect("/admin/laboratory");
  }
);
app.get("/admin/laboratory-gallery-delete/:id", async (req, res) => {
  await promisePool.execute(
    "DELETE FROM laboratory_gallery WHERE id=?",
    [req.params.id]
  );
  res.redirect("/admin/laboratory");
});

app.get('/library', async (req, res) => {

  const [page] = await promisePool.execute("SELECT * FROM library_page LIMIT 1");
  const libraryInfo = page[0] || {};

  const [libraryImages] = await promisePool.execute("SELECT * FROM library_images ORDER BY id DESC");

  const [ugBooks] = await promisePool.execute("SELECT * FROM library_ug_books ORDER BY id ASC");
  const [pgBooks] = await promisePool.execute("SELECT * FROM library_pg_books ORDER BY id ASC");
  const [gratis] = await promisePool.execute("SELECT * FROM library_gratis ORDER BY id ASC");
  const [journals] = await promisePool.execute("SELECT * FROM library_journals ORDER BY id ASC");

  const [journalInfoData] = await promisePool.execute("SELECT * FROM library_journal_info LIMIT 1");
  const journalInfo = journalInfoData[0] || {};

  const totalBooks =
    ugBooks.reduce((t, b) => t + b.no_of_volumes, 0) +
    pgBooks.reduce((t, b) => t + b.no_of_volumes, 0) +
    gratis.reduce((t, g) => t + g.no_of_volumes, 0);

  res.render("library", {
    libraryInfo,
    libraryImages,
    ugBooks,
    pgBooks,
    gratis,
    totalBooks,
    journals,
    journalInfo
  });
});
app.get('/admin/library', async (req, res) => {
  const [rows] = await promisePool.execute("SELECT * FROM library_page");
  res.render('admin/library', { data: rows });
});
app.get('/admin/library-edit/new', (req, res) =>
  res.render('admin/library-edit', { data: {} })
);
app.get('/admin/library-edit/:id', async (req, res) => {
  const [rows] = await promisePool.execute(
    "SELECT * FROM library_page WHERE id=?",
    [req.params.id]
  );
  res.render('admin/library-edit', { data: rows[0] });
});
app.post('/admin/library-update', async (req, res) => {
  const { id, title, intro_text, features, description, timing } = req.body;

  const featuresJson = JSON.stringify(
    features.split('\n').map(f => f.trim()).filter(Boolean)
  );

  if (id) {
    await promisePool.execute(
      "UPDATE library_page SET title=?, intro_text=?, features=?, description=?, timing=? WHERE id=?",
      [title, intro_text, featuresJson, description, timing, id]
    );
  } else {
    await promisePool.execute(
      "INSERT INTO library_page (title,intro_text,features,description,timing) VALUES (?,?,?,?,?)",
      [title, intro_text, featuresJson, description, timing]
    );
  }

  res.redirect('/admin/library');
});
app.get('/admin/library-ug-books', async (req, res) => {
  const [rows] = await promisePool.execute("SELECT * FROM library_ug_books ORDER BY id ASC");
  res.render('admin/library-ug-books', { data: rows });
});
app.get('/admin/library-ug-books-edit/new', (req, res) =>
  res.render('admin/library-ug-books-edit', { data: {} })
);
app.get('/admin/library-ug-books-edit/:id', async (req, res) => {
  const [rows] = await promisePool.execute(
    "SELECT * FROM library_ug_books WHERE id=?",
    [req.params.id]
  );
  res.render('admin/library-ug-books-edit', { data: rows[0] });
});
app.post('/admin/library-ug-books-update', async (req, res) => {
  const { id, department, no_of_titles, no_of_volumes } = req.body;

  if (id) {
    await promisePool.execute(
      "UPDATE library_ug_books SET department=?, no_of_titles=?, no_of_volumes=? WHERE id=?",
      [department, no_of_titles, no_of_volumes, id]
    );
  } else {
    await promisePool.execute(
      "INSERT INTO library_ug_books (department,no_of_titles,no_of_volumes) VALUES (?,?,?)",
      [department, no_of_titles, no_of_volumes]
    );
  }

  res.redirect('/admin/library-ug-books');
});
app.get('/admin/library-ug-books-delete/:id', async (req, res) => {
  await promisePool.execute("DELETE FROM library_ug_books WHERE id=?", [req.params.id]);
  res.redirect('/admin/library-ug-books');
});
app.get('/admin/library-pg-books', async (req, res) => {
  const [rows] = await promisePool.execute("SELECT * FROM library_pg_books ORDER BY id ASC");
  res.render('admin/library-pg-books', { data: rows });
});
app.get('/admin/library-pg-books-edit/new', (req, res) =>
  res.render('admin/library-pg-books-edit', { data: {} })
);
app.get('/admin/library-pg-books-edit/:id', async (req, res) => {
  const [rows] = await promisePool.execute(
    "SELECT * FROM library_pg_books WHERE id=?",
    [req.params.id]
  );
  res.render('admin/library-pg-books-edit', { data: rows[0] });
});
app.post('/admin/library-pg-books-update', async (req, res) => {
  const { id, department, no_of_titles, no_of_volumes } = req.body;

  if (id) {
    await promisePool.execute(
      "UPDATE library_pg_books SET department=?, no_of_titles=?, no_of_volumes=? WHERE id=?",
      [department, no_of_titles, no_of_volumes, id]
    );
  } else {
    await promisePool.execute(
      "INSERT INTO library_pg_books (department,no_of_titles,no_of_volumes) VALUES (?,?,?)",
      [department, no_of_titles, no_of_volumes]
    );
  }

  res.redirect('/admin/library-pg-books');
});
app.get('/admin/library-pg-books-delete/:id', async (req, res) => {
  await promisePool.execute("DELETE FROM library_pg_books WHERE id=?", [req.params.id]);
  res.redirect('/admin/library-pg-books');
});
app.get('/admin/library-gratis', async (req, res) => {
  const [rows] = await promisePool.execute("SELECT * FROM library_gratis ORDER BY id ASC");
  res.render('admin/library-gratis', { data: rows });
});
app.get('/admin/library-gratis-edit/new', (req, res) =>
  res.render('admin/library-gratis-edit', { data: {} })
);
app.get('/admin/library-gratis-edit/:id', async (req, res) => {
  const [rows] = await promisePool.execute(
    "SELECT * FROM library_gratis WHERE id=?",
    [req.params.id]
  );
  res.render('admin/library-gratis-edit', { data: rows[0] });
});
app.post('/admin/library-gratis-update', async (req, res) => {
  const { id, description, no_of_titles, no_of_volumes } = req.body;

  if (id) {
    await promisePool.execute(
      "UPDATE library_gratis SET description=?, no_of_titles=?, no_of_volumes=? WHERE id=?",
      [description, no_of_titles, no_of_volumes, id]
    );
  } else {
    await promisePool.execute(
      "INSERT INTO library_gratis (description,no_of_titles,no_of_volumes) VALUES (?,?,?)",
      [description, no_of_titles, no_of_volumes]
    );
  }

  res.redirect('/admin/library-gratis');
});
app.get('/admin/library-gratis-delete/:id', async (req, res) => {
  await promisePool.execute("DELETE FROM library_gratis WHERE id=?", [req.params.id]);
  res.redirect('/admin/library-gratis');
});
app.get('/admin/library-images', async (req, res) => {
  const [rows] = await promisePool.execute("SELECT * FROM library_images ORDER BY id DESC");
  res.render('admin/library-images', { data: rows });
});
app.get('/admin/library-images-upload', (req, res) =>
  res.render('admin/library-images-upload', {})
);
app.post('/admin/library-images-upload', upload.single('image_file'), async (req, res) => {
  const imagePath = "/images/" + req.file.filename;
  await promisePool.execute("INSERT INTO library_images (image_path) VALUES (?)", [imagePath]);
  res.redirect('/admin/library-images');
});
app.get('/admin/library-images-delete/:id', async (req, res) => {
  await promisePool.execute("DELETE FROM library_images WHERE id=?", [req.params.id]);
  res.redirect('/admin/library-images');
});

app.get('/admin/library-journals', async (req, res) => {
  const [rows] = await promisePool.execute("SELECT * FROM library_journals ORDER BY id ASC");
  res.render('admin/library-journals', { data: rows });
});
app.get('/admin/library-journals-edit/new', (req, res) =>
  res.render('admin/library-journals-edit', { data: {} })
);
app.get('/admin/library-journals-edit/:id', async (req, res) => {
  const [rows] = await promisePool.execute(
    "SELECT * FROM library_journals WHERE id=?",
    [req.params.id]
  );
  res.render('admin/library-journals-edit', { data: rows[0] });
});
app.post('/admin/library-journals-update', async (req, res) => {
  const { id, course_name, no_of_journals } = req.body;

  if (id) {
    await promisePool.execute(
      "UPDATE library_journals SET course_name=?, no_of_journals=? WHERE id=?",
      [course_name, no_of_journals, id]
    );
  } else {
    await promisePool.execute(
      "INSERT INTO library_journals (course_name,no_of_journals) VALUES (?,?)",
      [course_name, no_of_journals]
    );
  }

  res.redirect('/admin/library-journals');
});
app.get('/admin/library-journals-delete/:id', async (req, res) => {
  await promisePool.execute("DELETE FROM library_journals WHERE id=?", [req.params.id]);
  res.redirect('/admin/library-journals');
});
app.get('/admin/library-journal-info', async (req, res) => {
  const [rows] = await promisePool.execute("SELECT * FROM library_journal_info LIMIT 1");
  res.render('admin/library-journal-info', { data: rows[0] || {} });
});
app.get('/admin/library-journal-info-edit/new', (req, res) =>
  res.render('admin/library-journal-info-edit', { data: {} })
);
app.get('/admin/library-journal-info-edit/:id', async (req, res) => {
  const [rows] = await promisePool.execute(
    "SELECT * FROM library_journal_info WHERE id=?",
    [req.params.id]
  );
  res.render('admin/library-journal-info-edit', { data: rows[0] });
});
app.post('/admin/library-journal-info-update', async (req, res) => {
  const { id, total_journals, database_name } = req.body;

  if (id) {
    await promisePool.execute(
      "UPDATE library_journal_info SET total_journals=?, database_name=? WHERE id=?",
      [total_journals, database_name, id]
    );
  } else {
    await promisePool.execute(
      "INSERT INTO library_journal_info (total_journals,database_name) VALUES (?,?)",
      [total_journals, database_name]
    );
  }

  res.redirect('/admin/library-journal-info');
});
app.get('/admin/library-journal-info-delete/:id', async (req, res) => {
  await promisePool.execute("DELETE FROM library_journal_info WHERE id=?", [req.params.id]);
  res.redirect('/admin/library-journal-info');
});

app.get('/library-objectives', async (req, res) => {
  const [introRows] = await promisePool.execute("SELECT * FROM libraryobj_page LIMIT 1");
  const libraryIntro = introRows[0] || {};

  const [libraryObjImages] = await promisePool.execute("SELECT * FROM libraryobj_images ORDER BY id ASC");

  const [libraryObjectives] = await promisePool.execute("SELECT * FROM libraryobj_objectives ORDER BY id ASC");

  const [libraryBottom] = await promisePool.execute("SELECT * FROM libraryobj_bottom ORDER BY id ASC");

  res.render("library-objectives", {
    libraryIntro,
    libraryObjImages,
    libraryObjectives,
    libraryBottom
  });
});
app.get('/admin/library-objectives', async (req, res) => {
  const [introRows] = await promisePool.execute("SELECT * FROM libraryobj_page LIMIT 1");
  const intro = introRows[0] || {};

  const [objectives] = await promisePool.execute("SELECT * FROM libraryobj_objectives ORDER BY id ASC");

  const [bottom] = await promisePool.execute("SELECT * FROM libraryobj_bottom ORDER BY id ASC");

  res.render("admin/library-objectives", {
    intro,
    objectives,
    bottom
  });
});
app.get('/admin/library-objectives-edit/new', (req, res) => {
  res.render("admin/library-objectives-edit", { data: {} });
});
app.get('/admin/library-objectives-edit/:id', async (req, res) => {
  const [rows] = await promisePool.execute("SELECT * FROM libraryobj_page WHERE id=?", [req.params.id]);
  res.render("admin/library-objectives-edit", { data: rows[0] });
});
app.post('/admin/library-objectives-update', async (req, res) => {
  const { id, intro_text } = req.body;

  if (id) {
    await promisePool.execute(
      "UPDATE libraryobj_page SET intro_text=? WHERE id=?",
      [intro_text, id]
    );
  } else {
    await promisePool.execute(
      "INSERT INTO libraryobj_page (intro_text) VALUES (?)",
      [intro_text]
    );
  }

  res.redirect('/admin/library-objectives');
});
app.get('/admin/library-objectives-point/new', (req, res) => {
  res.render("admin/library-objectives-point-edit", { data: {} });
});
app.get('/admin/library-objectives-point/:id', async (req, res) => {
  const [rows] = await promisePool.execute("SELECT * FROM libraryobj_objectives WHERE id=?", [req.params.id]);
  res.render("admin/library-objectives-point-edit", { data: rows[0] });
});
app.post('/admin/library-objectives-point-update', async (req, res) => {
  const { id, point_text } = req.body;

  if (id) {
    await promisePool.execute(
      "UPDATE libraryobj_objectives SET point_text=? WHERE id=?",
      [point_text, id]
    );
  } else {
    await promisePool.execute(
      "INSERT INTO libraryobj_objectives (point_text) VALUES (?)",
      [point_text]
    );
  }

  res.redirect('/admin/library-objectives');
});
app.get('/admin/library-objectives-point-delete/:id', async (req, res) => {
  await promisePool.execute("DELETE FROM libraryobj_objectives WHERE id=?", [req.params.id]);
  res.redirect('/admin/library-objectives');
});
app.get('/admin/library-objectives-bottom/new', (req, res) => {
  res.render("admin/library-objectives-bottom-edit", { data: {} });
});
app.get('/admin/library-objectives-bottom/:id', async (req, res) => {
  const [rows] = await promisePool.execute("SELECT * FROM libraryobj_bottom WHERE id=?", [req.params.id]);
  res.render("admin/library-objectives-bottom-edit", { data: rows[0] });
});
app.post('/admin/library-objectives-bottom-update', async (req, res) => {
  const { id, paragraph_text } = req.body;

  if (id) {
    await promisePool.execute(
      "UPDATE libraryobj_bottom SET paragraph_text=? WHERE id=?",
      [paragraph_text, id]
    );
  } else {
    await promisePool.execute(
      "INSERT INTO libraryobj_bottom (paragraph_text) VALUES (?)",
      [paragraph_text]
    );
  }

  res.redirect('/admin/library-objectives');
});
app.get('/admin/library-objectives-bottom-delete/:id', async (req, res) => {
  await promisePool.execute("DELETE FROM libraryobj_bottom WHERE id=?", [req.params.id]);
  res.redirect('/admin/library-objectives');
});

app.get('/journal-magazines', async (req, res) => {
    try {
        const [topRows] = await promisePool.execute(
            "SELECT * FROM journal_magazines_top LIMIT 1"
        );
        const top = topRows[0] || { image: '', title: '', content: '' };

        const [topList] = await promisePool.execute(
            "SELECT * FROM journal_magazines_top_list ORDER BY id ASC"
        );

        const [books] = await promisePool.execute(
            "SELECT * FROM journal_magazines_books ORDER BY id ASC"
        );

        const [journals] = await promisePool.execute(
            "SELECT * FROM journal_magazines_journals ORDER BY id ASC"
        );

        const [ejournals] = await promisePool.execute(
            "SELECT * FROM journal_magazines_ejournals ORDER BY id ASC"
        );

        const journalsTotal = journals.reduce((s, r) => s + (r.count || 0), 0);

        res.render("journal-magazines", {
            top,
            topList,
            books,
            journals,
            ejournals,
            journalsTotal
        });

    } catch (err) {
        console.error(err);
        res.status(500).send("Frontend error");
    }
});
app.get('/admin/journal-magazines', async (req, res) => {
    const [topRows] = await promisePool.execute("SELECT * FROM journal_magazines_top LIMIT 1");
    const top = topRows[0] || null;

    const [topList] = await promisePool.execute("SELECT * FROM journal_magazines_top_list ORDER BY id ASC");
    const [books] = await promisePool.execute("SELECT * FROM journal_magazines_books ORDER BY id ASC");
    const [journals] = await promisePool.execute("SELECT * FROM journal_magazines_journals ORDER BY id ASC");
    const [ejournals] = await promisePool.execute("SELECT * FROM journal_magazines_ejournals ORDER BY id ASC");

    res.render("admin/journal-magazines", {
        top,
        topList,
        books,
        journals,
        ejournals
    });
});
app.get('/admin/journal-magazines-top-edit/new', (req, res) => {
    res.render("admin/journal-magazines-top-edit", { data: {} });
});
app.get('/admin/journal-magazines-top-edit/:id', async (req, res) => {
    const [rows] = await promisePool.execute(
        "SELECT * FROM journal_magazines_top WHERE id=?",
        [req.params.id]
    );
    res.render("admin/journal-magazines-top-edit", { data: rows[0] || {} });
});
app.post('/admin/journal-magazines-top-update', upload.single('image'), async (req, res) => {
    const { id, title, content, current_image } = req.body;
    let imagePath = current_image || '';

    if (req.file) imagePath = "/images/" + req.file.filename;

    if (id) {
        await promisePool.execute(
            "UPDATE journal_magazines_top SET image=?, title=?, content=? WHERE id=?",
            [imagePath, title, content, id]
        );
    } else {
        await promisePool.execute(
            "INSERT INTO journal_magazines_top (image, title, content) VALUES (?, ?, ?)",
            [imagePath, title, content]
        );
    }

    res.redirect('/admin/journal-magazines');
});
app.get('/admin/journal-magazines-top-list-new', (req, res) => {
    res.render("admin/journal-magazines-top-list-edit", { data: {} });
});
app.get('/admin/journal-magazines-top-list/:id', async (req, res) => {
    const [rows] = await promisePool.execute(
        "SELECT * FROM journal_magazines_top_list WHERE id=?",
        [req.params.id]
    );
    res.render("admin/journal-magazines-top-list-edit", { data: rows[0] || {} });
});
app.post('/admin/journal-magazines-top-list-update', async (req, res) => {
    const { id, content } = req.body;

    if (id) {
        await promisePool.execute(
            "UPDATE journal_magazines_top_list SET content=? WHERE id=?",
            [content, id]
        );
    } else {
        await promisePool.execute(
            "INSERT INTO journal_magazines_top_list (content) VALUES (?)",
            [content]
        );
    }

    res.redirect('/admin/journal-magazines');
});
app.get('/admin/journal-magazines-top-list-delete/:id', async (req, res) => {
    await promisePool.execute(
        "DELETE FROM journal_magazines_top_list WHERE id=?",
        [req.params.id]
    );
    res.redirect('/admin/journal-magazines');
});
app.get('/admin/journal-magazines-books-new', (req, res) => {
    res.render("admin/journal-magazines-books-edit", { data: {} });
});
app.get('/admin/journal-magazines-books/:id', async (req, res) => {
    const [rows] = await promisePool.execute(
        "SELECT * FROM journal_magazines_books WHERE id=?",
        [req.params.id]
    );
    res.render("admin/journal-magazines-books-edit", { data: rows[0] || {} });
});
app.post('/admin/journal-magazines-books-update', async (req, res) => {
    const { id, description, count } = req.body;

    if (id) {
        await promisePool.execute(
            "UPDATE journal_magazines_books SET description=?, count=? WHERE id=?",
            [description, count, id]
        );
    } else {
        await promisePool.execute(
            "INSERT INTO journal_magazines_books (description, count) VALUES (?, ?)",
            [description, count]
        );
    }

    res.redirect('/admin/journal-magazines');
});
app.get('/admin/journal-magazines-books-delete/:id', async (req, res) => {
    await promisePool.execute(
        "DELETE FROM journal_magazines_books WHERE id=?",
        [req.params.id]
    );
    res.redirect('/admin/journal-magazines');
});
app.get('/admin/journal-magazines-journals-new', (req, res) => {
    res.render("admin/journal-magazines-journals-edit", { data: {} });
});
app.get('/admin/journal-magazines-journals/:id', async (req, res) => {
    const [rows] = await promisePool.execute(
        "SELECT * FROM journal_magazines_journals WHERE id=?",
        [req.params.id]
    );
    res.render("admin/journal-magazines-journals-edit", { data: rows[0] || {} });
});
app.post('/admin/journal-magazines-journals-update', async (req, res) => {
    const { id, description, count } = req.body;

    if (id) {
        await promisePool.execute(
            "UPDATE journal_magazines_journals SET description=?, count=? WHERE id=?",
            [description, count, id]
        );
    } else {
        await promisePool.execute(
            "INSERT INTO journal_magazines_journals (description, count) VALUES (?, ?)",
            [description, count]
        );
    }

    res.redirect('/admin/journal-magazines');
});
app.get('/admin/journal-magazines-journals-delete/:id', async (req, res) => {
    await promisePool.execute(
        "DELETE FROM journal_magazines_journals WHERE id=?",
        [req.params.id]
    );
    res.redirect('/admin/journal-magazines');
});
app.get('/admin/journal-magazines-ejournals-new', (req, res) => {
    res.render("admin/journal-magazines-ejournals-edit", { data: {} });
});
app.get('/admin/journal-magazines-ejournals/:id', async (req, res) => {
    const [rows] = await promisePool.execute(
        "SELECT * FROM journal_magazines_ejournals WHERE id=?",
        [req.params.id]
    );
    res.render("admin/journal-magazines-ejournals-edit", { data: rows[0] || {} });
});
app.post('/admin/journal-magazines-ejournals-update', async (req, res) => {
    const { id, description, count } = req.body;

    if (id) {
        await promisePool.execute(
            "UPDATE journal_magazines_ejournals SET description=?, count=? WHERE id=?",
            [description, count, id]
        );
    } else {
        await promisePool.execute(
            "INSERT INTO journal_magazines_ejournals (description, count) VALUES (?, ?)",
            [description, count]
        );
    }

    res.redirect('/admin/journal-magazines');
});
app.get('/admin/journal-magazines-ejournals-delete/:id', async (req, res) => {
    await promisePool.execute(
        "DELETE FROM journal_magazines_ejournals WHERE id=?",
        [req.params.id]
    );
    res.redirect('/admin/journal-magazines');
});

app.get('/internet-wifi', async (req, res) => {
    const [images] = await promisePool.execute(
        "SELECT * FROM internet_wifi_image ORDER BY id ASC"
    );

    const [content] = await promisePool.execute(
        "SELECT * FROM internet_wifi_content ORDER BY id ASC"
    );

    res.render("internet-wifi", {
        images,
        content
    });
});
app.get('/admin/internet-wifi', async (req, res) => {
    const [images] = await promisePool.execute("SELECT * FROM internet_wifi_image ORDER BY id ASC");
    const [content] = await promisePool.execute("SELECT * FROM internet_wifi_content ORDER BY id ASC");

    res.render("admin/internet-wifi", { images, content });
});
app.get('/admin/internet-wifi-image-new', (req, res) => {
    res.render("admin/internet-wifi-image-edit");
});
app.post('/admin/internet-wifi-image-save', upload.single("image"), async (req, res) => {
    const imagePath = "/images/" + req.file.filename;

    await promisePool.execute(
        "INSERT INTO internet_wifi_image (image) VALUES (?)",
        [imagePath]
    );

    res.redirect('/admin/internet-wifi');
});
app.get('/admin/internet-wifi-image-delete/:id', async (req, res) => {
    await promisePool.execute("DELETE FROM internet_wifi_image WHERE id=?", [req.params.id]);
    res.redirect('/admin/internet-wifi');
});
app.get('/admin/internet-wifi-content-new', (req, res) => {
    res.render("admin/internet-wifi-content-edit", { data: {} });
});
app.get('/admin/internet-wifi-content-edit/:id', async (req, res) => {
    const [rows] = await promisePool.execute(
        "SELECT * FROM internet_wifi_content WHERE id=?",
        [req.params.id]
    );
    res.render("admin/internet-wifi-content-edit", { data: rows[0] });
});
app.post('/admin/internet-wifi-content-save', async (req, res) => {
    const { id, content } = req.body;

    if (id) {
        await promisePool.execute(
            "UPDATE internet_wifi_content SET content=? WHERE id=?",
            [content, id]
        );
    } else {
        await promisePool.execute(
            "INSERT INTO internet_wifi_content (content) VALUES (?)",
            [content]
        );
    }

    res.redirect('/admin/internet-wifi');
});
app.get('/admin/internet-wifi-content-delete/:id', async (req, res) => {
    await promisePool.execute(
        "DELETE FROM internet_wifi_content WHERE id=?",
        [req.params.id]
    );

    res.redirect('/admin/internet-wifi');
});

app.get('/transportation', async (req, res) => {
  try {
    const [[page]] = await promisePool.execute("SELECT * FROM transportation_page_info LIMIT 1");
    const [images] = await promisePool.execute("SELECT * FROM transportation_images ORDER BY id ASC");
    const [routes] = await promisePool.execute("SELECT * FROM transportation_routes ORDER BY id ASC");
    res.render('transportation', { page: page || {}, images, routes });
  } catch (err) { console.error(err); res.status(500).send('Server error'); }
});
app.get('/admin/transportation', async (req,res)=>{
  const [page] = await promisePool.execute("SELECT * FROM transportation_page_info LIMIT 1");
  const [images] = await promisePool.execute("SELECT * FROM transportation_images ORDER BY id ASC");
  const [routes] = await promisePool.execute("SELECT * FROM transportation_routes ORDER BY id ASC");
  res.render('admin/transportation', { page: page[0] || {}, images, routes });
});
app.get('/admin/transportation-page-edit/new', (req,res)=> res.render('admin/transportation-page-edit',{ data:{} }));
app.get('/admin/transportation-page-edit/:id', async (req,res)=> {
  const [rows] = await promisePool.execute("SELECT * FROM transportation_page_info WHERE id=?", [req.params.id]);
  res.render('admin/transportation-page-edit',{ data: rows[0] || {} });
});
app.post('/admin/transportation-page-save', async (req,res)=> {
  const { id, college_name, page_title } = req.body;
  const collegeName = college_name || null;
  const pageTitle = page_title || null;
  
  if (id) {
    await promisePool.execute("UPDATE transportation_page_info SET college_name=?, page_title=? WHERE id=?", [collegeName, pageTitle, id]);
  } else {
    await promisePool.execute("INSERT INTO transportation_page_info (college_name, page_title) VALUES (?,?)", [collegeName, pageTitle]);
  }
  res.redirect('/admin/transportation');
});
app.get('/admin/transportation-image-new', (req,res)=> res.render('admin/transportation-image-edit',{ data:{} }));
app.post('/admin/transportation-image-save', upload.single('image'), async (req,res)=> {
  const imagePath = "/images/" + req.file.filename;
  await promisePool.execute("INSERT INTO transportation_images (image_path) VALUES (?)", [imagePath]);
  res.redirect('/admin/transportation');
});
app.get('/admin/transportation-image-delete/:id', async (req,res)=> {
  const [rows] = await promisePool.execute("SELECT image_path FROM transportation_images WHERE id=?", [req.params.id]);
  if (rows[0]) { const f = path.join(__dirname,'public', rows[0].image_path); if (fs.existsSync(f)) fs.unlinkSync(f); }
  await promisePool.execute("DELETE FROM transportation_images WHERE id=?", [req.params.id]);
  res.redirect('/admin/transportation');
});
app.get('/admin/transportation-route-new', (req,res)=> res.render('admin/transportation-route-edit',{ data:{} }));
app.get('/admin/transportation-route/:id', async (req,res)=> {
  const [rows] = await promisePool.execute("SELECT * FROM transportation_routes WHERE id=?", [req.params.id]);
  res.render('admin/transportation-route-edit',{ data: rows[0] || {} });
});
app.post('/admin/transportation-route-save', async (req,res)=> {
  const { id, route_name, details } = req.body;
  if (id) await promisePool.execute("UPDATE transportation_routes SET route_name=?, details=? WHERE id=?", [route_name, details, id]);
  else await promisePool.execute("INSERT INTO transportation_routes (route_name, details) VALUES (?, ?)", [route_name, details]);
  res.redirect('/admin/transportation');
});
app.get('/admin/transportation-route-delete/:id', async (req,res)=> {
  await promisePool.execute("DELETE FROM transportation_routes WHERE id=?", [req.params.id]);
  res.redirect('/admin/transportation');
});

app.get('/mess-facilities', async (req, res) => {
  try {
    const [[info]] = await promisePool.execute("SELECT * FROM mess_facilities_info LIMIT 1");
    const [images] = await promisePool.execute("SELECT * FROM mess_facilities_images ORDER BY id ASC");

    res.render('mess-facilities', {
      info: info || {},
      images
    });

  } catch (err) {
    console.error(err);
    res.status(500).send("Server error");
  }
});
app.get('/admin/mess-facilities', async (req, res) => {
  const [rows] = await promisePool.execute("SELECT * FROM mess_facilities_info LIMIT 1");
  const [images] = await promisePool.execute("SELECT * FROM mess_facilities_images ORDER BY id ASC");

  res.render('admin/mess-facilities', {
    info: rows[0] || {},
    images
  });
});
app.get('/admin/mess-facilities-info-edit/:id', async (req,res) => {
  const [rows] = await promisePool.execute("SELECT * FROM mess_facilities_info WHERE id=?", [req.params.id]);
  res.render('admin/mess-facilities-info-edit', { data: rows[0] || {} });
});
app.post('/admin/mess-facilities-info-save', async (req,res)=> {

  const { id, college_name, section_title } = req.body;

  // Convert content list → JSON array
  const content_list = JSON.stringify(
    req.body.content_list_text
    .split('\n')
    .map(x => x.trim())
    .filter(x => x !== "")
  );

  // Convert dining timings → JSON object
  const timings = {};
  req.body.dining_timings_text.split('\n').forEach(line => {
    let [key, value] = line.split(':');
    if (key && value) {
      timings[key.trim()] = value.trim();
    }
  });

  const dining_timings = JSON.stringify(timings);

  if(id){
    await promisePool.execute(
      "UPDATE mess_facilities_info SET college_name=?, section_title=?, content_list=?, dining_timings=? WHERE id=?",
      [college_name, section_title, content_list, dining_timings, id]
    );
  } else {
    await promisePool.execute(
      "INSERT INTO mess_facilities_info (college_name, section_title, content_list, dining_timings) VALUES (?,?,?,?)",
      [college_name, section_title, content_list, dining_timings]
    );
  }

  res.redirect('/admin/mess-facilities');
});
app.get('/admin/mess-facilities-image-new', (req,res)=> 
  res.render('admin/mess-facilities-image-edit',{ data:{} })
);
app.post('/admin/mess-facilities-image-save', upload.single('image'), async (req,res)=> {
  const imagePath = "/images/" + req.file.filename;
  await promisePool.execute("INSERT INTO mess_facilities_images (image_path) VALUES (?)", [imagePath]);
  res.redirect('/admin/mess-facilities');
});
app.get('/admin/mess-facilities-image-delete/:id', async (req,res)=> {
  const [rows] = await promisePool.execute("SELECT image_path FROM mess_facilities_images WHERE id=?", [req.params.id]);

  if (rows[0]) {
    const f = path.join(__dirname,'public', rows[0].image_path);
    if (fs.existsSync(f)) fs.unlinkSync(f);
  }

  await promisePool.execute("DELETE FROM mess_facilities_images WHERE id=?", [req.params.id]);
  res.redirect('/admin/mess-facilities');
});

app.get('/hostel-facilities', async (req, res) => {
  try {
    const [[topInfo]] = await promisePool.execute("SELECT * FROM hostel_facilities_top_info LIMIT 1");
    const [topList] = await promisePool.execute("SELECT * FROM hostel_facilities_top_list ORDER BY position ASC");
    const [bottom] = await promisePool.execute("SELECT * FROM hostel_facilities_bottom_content ORDER BY position ASC");
    const [images] = await promisePool.execute("SELECT * FROM hostel_facilities_images ORDER BY id ASC");

    res.render('hostel-facilities', {
      topInfo: topInfo || {},
      topList,
      bottom,
      images
    });
  } catch (err) {
    console.error(err);
    res.status(500).send('Server error');
  }
});
app.get('/admin/hostel-facilities', async (req, res) => {
  try {
    const [[topInfo]] = await promisePool.execute("SELECT * FROM hostel_facilities_top_info LIMIT 1");
    const [topList] = await promisePool.execute("SELECT * FROM hostel_facilities_top_list ORDER BY position ASC");
    const [bottom] = await promisePool.execute("SELECT * FROM hostel_facilities_bottom_content ORDER BY position ASC");
    const [images] = await promisePool.execute("SELECT * FROM hostel_facilities_images ORDER BY id ASC");

    res.render('admin/hostel-facilities', {
      topInfo: topInfo || {},
      topList,
      bottom,
      images
    });
  } catch (err) {
    console.error(err);
    res.status(500).send('Server error');
  }
});
app.get('/admin/hostel-top-info-edit', (req, res) => {
  res.render('admin/hostel-top-info-edit', { data: {} });
});
app.get('/admin/hostel-top-info-edit/:id', async (req, res) => {
  const [rows] = await promisePool.execute("SELECT * FROM hostel_facilities_top_info WHERE id=?", [req.params.id]);
  res.render('admin/hostel-top-info-edit', { data: rows[0] || {} });
});
app.post('/admin/hostel-top-info-save', async (req, res) => {
  const { id, title, intro } = req.body;
  if (id) {
    await promisePool.execute("UPDATE hostel_facilities_top_info SET title=?, intro=? WHERE id=?", [title, intro, id]);
  } else {
    await promisePool.execute("INSERT INTO hostel_facilities_top_info (title,intro) VALUES (?,?)", [title, intro]);
  }
  res.redirect('/admin/hostel-facilities');
});
app.get('/admin/hostel-top-list-new', (req, res) => res.render('admin/hostel-top-list-edit', { data: {} }));
app.get('/admin/hostel-top-list-edit/:id', async (req, res) => {
  const [rows] = await promisePool.execute("SELECT * FROM hostel_facilities_top_list WHERE id=?", [req.params.id]);
  res.render('admin/hostel-top-list-edit', { data: rows[0] || {} });
});
app.post('/admin/hostel-top-list-save', async (req, res) => {
  const { id, position, item } = req.body;
  const pos = position ? Number(position) : 999;
  if (id) {
    await promisePool.execute("UPDATE hostel_facilities_top_list SET position=?, item=? WHERE id=?", [pos, item, id]);
  } else {
    await promisePool.execute("INSERT INTO hostel_facilities_top_list (position,item) VALUES (?,?)", [pos, item]);
  }
  res.redirect('/admin/hostel-facilities');
});
app.get('/admin/hostel-top-list-delete/:id', async (req, res) => {
  await promisePool.execute("DELETE FROM hostel_facilities_top_list WHERE id=?", [req.params.id]);
  res.redirect('/admin/hostel-facilities');
});
app.get('/admin/hostel-bottom-new', (req, res) => res.render('admin/hostel-bottom-content-edit', { data: {} }));
app.get('/admin/hostel-bottom-edit/:id', async (req, res) => {
  const [rows] = await promisePool.execute("SELECT * FROM hostel_facilities_bottom_content WHERE id=?", [req.params.id]);
  res.render('admin/hostel-bottom-content-edit', { data: rows[0] || {} });
});
app.post('/admin/hostel-bottom-save', async (req, res) => {
  const { id, position, content } = req.body;
  const pos = position ? Number(position) : 999;
  if (id) {
    await promisePool.execute("UPDATE hostel_facilities_bottom_content SET position=?, content=? WHERE id=?", [pos, content, id]);
  } else {
    await promisePool.execute("INSERT INTO hostel_facilities_bottom_content (position,content) VALUES (?,?)", [pos, content]);
  }
  res.redirect('/admin/hostel-facilities');
});
app.get('/admin/hostel-bottom-delete/:id', async (req, res) => {
  await promisePool.execute("DELETE FROM hostel_facilities_bottom_content WHERE id=?", [req.params.id]);
  res.redirect('/admin/hostel-facilities');
});
app.get('/admin/hostel-image-new', (req, res) => res.render('admin/hostel-image-edit', { data: {} }));
app.post('/admin/hostel-image-save', upload.single('image'), async (req, res) => {
  const imagePath = "/images/" + req.file.filename;
  await promisePool.execute("INSERT INTO hostel_facilities_images (image) VALUES (?)", [imagePath]);
  res.redirect('/admin/hostel-facilities');
});
app.get('/admin/hostel-image-delete/:id', async (req, res) => {
  const [rows] = await promisePool.execute("SELECT image FROM hostel_facilities_images WHERE id=?", [req.params.id]);
  if (rows[0]) {
    const file = path.join(__dirname, 'public', rows[0].image);
    if (fs.existsSync(file)) fs.unlinkSync(file);
  }
  await promisePool.execute("DELETE FROM hostel_facilities_images WHERE id=?", [req.params.id]);
  res.redirect('/admin/hostel-facilities');
});

app.get('/tec-gallery', async (req, res) => {
    try {
        const [alumni] = await promisePool.execute("SELECT * FROM tec_gallery_alumni ORDER BY id DESC");
        const [civil]  = await promisePool.execute("SELECT * FROM tec_gallery_civil_alumni ORDER BY id DESC");
        const [cse]    = await promisePool.execute("SELECT * FROM tec_gallery_cse_alumni ORDER BY id DESC");
        const [ece]    = await promisePool.execute("SELECT * FROM tec_gallery_ece_alumni ORDER BY id DESC");
        const [eee]    = await promisePool.execute("SELECT * FROM tec_gallery_eee_alumni ORDER BY id DESC");
        const [mech]   = await promisePool.execute("SELECT * FROM tec_gallery_mech_alumni ORDER BY id DESC");

        res.render('tec-gallery', { alumni, civil, cse, ece, eee, mech });
    } catch (err) {
        console.log(err);
        res.send("Error loading gallery");
    }
});
app.get('/admin/tec-gallery', async (req, res) => {
    const [alumni] = await promisePool.execute("SELECT * FROM tec_gallery_alumni ORDER BY id DESC");
    const [civil]  = await promisePool.execute("SELECT * FROM tec_gallery_civil_alumni ORDER BY id DESC");
    const [cse]    = await promisePool.execute("SELECT * FROM tec_gallery_cse_alumni ORDER BY id DESC");
    const [ece]    = await promisePool.execute("SELECT * FROM tec_gallery_ece_alumni ORDER BY id DESC");
    const [eee]    = await promisePool.execute("SELECT * FROM tec_gallery_eee_alumni ORDER BY id DESC");
    const [mech]   = await promisePool.execute("SELECT * FROM tec_gallery_mech_alumni ORDER BY id DESC");

    res.render('admin/tec-gallery', { alumni, civil, cse, ece, eee, mech });
});
function createGalleryRoutes(routeName, tableName) {

  // Render Add Image Form
  app.get(`/admin/${routeName}`, (req, res) =>
    res.render(`admin/${routeName}-edit`)
  );

  // Save Uploaded Image
  app.post(`/admin/${routeName}-save`, upload.single('image'), async (req, res) => {
    const imgPath = "/images/" + req.file.filename;

    await promisePool.execute(
      `INSERT INTO ${tableName} (image) VALUES (?)`,
      [imgPath]
    );

    res.redirect("/admin/tec-gallery");
  });

  // Delete Image
  app.get(`/admin/${routeName}-delete/:id`, async (req, res) => {

    const [rows] = await promisePool.execute(
      `SELECT image FROM ${tableName} WHERE id=?`,
      [req.params.id]
    );

    if (rows.length) {
      const filePath = path.join(__dirname, "public", rows[0].image);
      if (fs.existsSync(filePath)) fs.unlinkSync(filePath);
    }

    await promisePool.execute(
      `DELETE FROM ${tableName} WHERE id=?`,
      [req.params.id]
    );

    res.redirect("/admin/tec-gallery");
  });
}
app.get('/admin/tec-gallery-alumni', (req, res) =>
    res.render('admin/tec-gallery-alumni')
);
app.get('/admin/tec-gallery-civil', (req, res) =>
    res.render('admin/tec-gallery-civil')
);
app.get('/admin/tec-gallery-cse', (req, res) =>
    res.render('admin/tec-gallery-cse')
);
app.get('/admin/tec-gallery-ece', (req, res) =>
    res.render('admin/tec-gallery-ece')
);
app.get('/admin/tec-gallery-eee', (req, res) =>
    res.render('admin/tec-gallery-eee')
);
app.get('/admin/tec-gallery-mech', (req, res) =>
    res.render('admin/tec-gallery-mech')
);
app.post('/admin/tec-gallery-alumni-save', upload.array('images'), async (req, res) => {
    const files = req.files;
    if (!files || !files.length) return res.redirect('/admin/tec-gallery');

    for (let file of files) {
        await promisePool.execute(
            "INSERT INTO tec_gallery_alumni (image) VALUES (?)",
            ['/images/' + file.filename]
        );
    }

    res.redirect('/admin/tec-gallery');
});
app.post('/admin/tec-gallery-civil-save', upload.array('images'), async (req, res) => {
    const files = req.files;
    for (let file of files) {
        await promisePool.execute(
            "INSERT INTO tec_gallery_civil_alumni (image) VALUES (?)",
            ['/images/' + file.filename]
        );
    }
    res.redirect('/admin/tec-gallery');
});
app.post('/admin/tec-gallery-cse-save', upload.array('images'), async (req, res) => {
    const files = req.files;
    for (let file of files) {
        await promisePool.execute(
            "INSERT INTO tec_gallery_cse_alumni (image) VALUES (?)",
            ['/images/' + file.filename]
        );
    }
    res.redirect('/admin/tec-gallery');
});
app.post('/admin/tec-gallery-ece-save', upload.array('images'), async (req, res) => {
    const files = req.files;
    for (let file of files) {
        await promisePool.execute(
            "INSERT INTO tec_gallery_ece_alumni (image) VALUES (?)",
            ['/images/' + file.filename]
        );
    }
    res.redirect('/admin/tec-gallery');
});
app.post('/admin/tec-gallery-eee-save', upload.array('images'), async (req, res) => {
    const files = req.files;
    for (let file of files) {
        await promisePool.execute(
            "INSERT INTO tec_gallery_eee_alumni (image) VALUES (?)",
            ['/images/' + file.filename]
        );
    }
    res.redirect('/admin/tec-gallery');
});
app.post('/admin/tec-gallery-mech-save', upload.array('images'), async (req, res) => {
    const files = req.files;
    for (let file of files) {
        await promisePool.execute(
            "INSERT INTO tec_gallery_mech_alumni (image) VALUES (?)",
            ['/images/' + file.filename]
        );
    }
    res.redirect('/admin/tec-gallery');
});

// Delete routes for TEC Gallery
app.get('/admin/tec-gallery-alumni-delete/:id', async (req, res) => {
    const [rows] = await promisePool.execute("SELECT image FROM tec_gallery_alumni WHERE id=?", [req.params.id]);
    if (rows[0]) {
        const filePath = path.join(__dirname, "public", rows[0].image);
        if (fs.existsSync(filePath)) fs.unlinkSync(filePath);
    }
    await promisePool.execute("DELETE FROM tec_gallery_alumni WHERE id=?", [req.params.id]);
    res.redirect('/admin/tec-gallery');
});

app.get('/admin/tec-gallery-civil-delete/:id', async (req, res) => {
    const [rows] = await promisePool.execute("SELECT image FROM tec_gallery_civil_alumni WHERE id=?", [req.params.id]);
    if (rows[0]) {
        const filePath = path.join(__dirname, "public", rows[0].image);
        if (fs.existsSync(filePath)) fs.unlinkSync(filePath);
    }
    await promisePool.execute("DELETE FROM tec_gallery_civil_alumni WHERE id=?", [req.params.id]);
    res.redirect('/admin/tec-gallery');
});

app.get('/admin/tec-gallery-cse-delete/:id', async (req, res) => {
    const [rows] = await promisePool.execute("SELECT image FROM tec_gallery_cse_alumni WHERE id=?", [req.params.id]);
    if (rows[0]) {
        const filePath = path.join(__dirname, "public", rows[0].image);
        if (fs.existsSync(filePath)) fs.unlinkSync(filePath);
    }
    await promisePool.execute("DELETE FROM tec_gallery_cse_alumni WHERE id=?", [req.params.id]);
    res.redirect('/admin/tec-gallery');
});

app.get('/admin/tec-gallery-ece-delete/:id', async (req, res) => {
    const [rows] = await promisePool.execute("SELECT image FROM tec_gallery_ece_alumni WHERE id=?", [req.params.id]);
    if (rows[0]) {
        const filePath = path.join(__dirname, "public", rows[0].image);
        if (fs.existsSync(filePath)) fs.unlinkSync(filePath);
    }
    await promisePool.execute("DELETE FROM tec_gallery_ece_alumni WHERE id=?", [req.params.id]);
    res.redirect('/admin/tec-gallery');
});

app.get('/admin/tec-gallery-eee-delete/:id', async (req, res) => {
    const [rows] = await promisePool.execute("SELECT image FROM tec_gallery_eee_alumni WHERE id=?", [req.params.id]);
    if (rows[0]) {
        const filePath = path.join(__dirname, "public", rows[0].image);
        if (fs.existsSync(filePath)) fs.unlinkSync(filePath);
    }
    await promisePool.execute("DELETE FROM tec_gallery_eee_alumni WHERE id=?", [req.params.id]);
    res.redirect('/admin/tec-gallery');
});

app.get('/admin/tec-gallery-mech-delete/:id', async (req, res) => {
    const [rows] = await promisePool.execute("SELECT image FROM tec_gallery_mech_alumni WHERE id=?", [req.params.id]);
    if (rows[0]) {
        const filePath = path.join(__dirname, "public", rows[0].image);
        if (fs.existsSync(filePath)) fs.unlinkSync(filePath);
    }
    await promisePool.execute("DELETE FROM tec_gallery_mech_alumni WHERE id=?", [req.params.id]);
    res.redirect('/admin/tec-gallery');
});

app.get('/green-campus-policy', async (req, res) => {
  try {
    const [sections] = await promisePool.execute(
      "SELECT * FROM green_campus_policy ORDER BY order_index ASC, id ASC"
    );
    res.render("green-campus-policy", { sectionsList: sections });
  } catch (error) {
    console.error("Error loading green campus policy:", error);
    res.status(500).send("Error loading page");
  }
});
app.get('/admin/green-campus-policy', async (req, res) => {
  try {
    const [sections] = await promisePool.execute(
      "SELECT * FROM green_campus_policy ORDER BY order_index ASC, id ASC"
    );
    res.render("admin/green-campus-policy", { sections });
  } catch (error) {
    console.error("Error loading green campus policy:", error);
    res.status(500).send("Error loading green campus policy");
  }
});
app.get('/admin/green-campus-policy/edit/:key', async (req, res) => {
  try {
    const { key } = req.params;
    const [rows] = await promisePool.execute(
      "SELECT * FROM green_campus_policy WHERE section_key = ?",
      [key]
    );
    res.render("admin/green-campus-policy-edit", {
      section: rows[0] || { section_key: key }
    });
  } catch (error) {
    console.error("Error loading green campus edit:", error);
    res.status(500).send("Error loading green campus edit");
  }
});
app.post('/admin/green-campus-policy/update', async (req, res) => {
  try {
    const { section_key, section_title, content, order_index } = req.body;

    const [existing] = await promisePool.execute(
      "SELECT id FROM green_campus_policy WHERE section_key = ?",
      [section_key]
    );

    if (existing.length > 0) {
      await promisePool.execute(
        "UPDATE green_campus_policy SET section_title=?, content=?, order_index=? WHERE section_key=?",
        [section_title, content, order_index || 0, section_key]
      );
    } else {
      await promisePool.execute(
        "INSERT INTO green_campus_policy (section_key, section_title, content, order_index) VALUES (?, ?, ?, ?)",
        [section_key, section_title, content, order_index || 0]
      );
    }

    res.redirect('/admin/green-campus-policy');
  } catch (error) {
    console.error("Error updating green campus policy:", error);
    res.status(500).send("Error updating green campus policy");
  }
});
app.get('/admin/green-campus-policy/delete/:key', async (req, res) => {
  try {
    const { key } = req.params;
    await promisePool.execute(
      "DELETE FROM green_campus_policy WHERE section_key = ?",
      [key]
    );
    res.redirect('/admin/green-campus-policy');
  } catch (error) {
    console.error("Error deleting policy:", error);
    res.status(500).send("Error deleting policy");
  }
});

app.get('/naac', async (req, res) => {
  const [rows] = await promisePool.execute("SELECT * FROM naac LIMIT 1");
  res.render('naac', { data: rows[0] || {} });
});
app.get('/admin/naac', async (req, res) => {
  const [rows] = await promisePool.execute("SELECT * FROM naac LIMIT 1");
  res.render('admin/naac', { data: rows[0] || {} });
});
app.get('/admin/naac-edit/:id', async (req, res) => {
  const [rows] = await promisePool.execute("SELECT * FROM naac WHERE id = ?", [req.params.id]);
  res.render('admin/naac-edit', { data: rows[0] || {} });
});
app.post('/admin/naac-update', async (req, res) => {
  const f = req.body;

  if (f.id) {
    await promisePool.execute(`
      UPDATE naac SET 
        page_title=?, 
        objective_title=?, objective_content=?,
        responsibility_title=?, responsibility_content=?
      WHERE id=?
    `, [
      f.page_title, f.objective_title, f.objective_content,
      f.responsibility_title, f.responsibility_content,
      f.id
    ]);
  }

  res.redirect('/admin/naac');
});

app.get('/peer-team-visit', async (req, res) => {
  try {
    const [videos] = await promisePool.execute(
      "SELECT * FROM peer_team_visit ORDER BY day_number ASC, order_index ASC"
    );

    const [members] = await promisePool.execute(
      "SELECT * FROM peer_team_members ORDER BY order_index ASC"
    );

    res.render('peer-team-visit', { videos, members });

  } catch (error) {
    console.error("Error loading peer team visit page:", error);
    res.status(500).send("Error loading page");
  }
});
app.get('/admin/peer-team-visit', async (req, res) => {
  try {
    const [videos] = await promisePool.execute('SELECT * FROM peer_team_visit ORDER BY day_number ASC, order_index ASC');
    const [members] = await promisePool.execute('SELECT * FROM peer_team_members ORDER BY order_index ASC');
    res.render('admin/peer-team-visit', { videos, members });
  } catch (error) {
    console.error('Error loading peer team visit:', error);
    res.status(500).send('Error loading peer team visit');
  }
});
app.post('/admin/peer-team-visit/video/add', async (req, res) => {
  try {
    const { visit_date, day_number, video_title, video_url, order_index } = req.body;
    await promisePool.execute(
      'INSERT INTO peer_team_visit (visit_date, day_number, video_title, video_url, order_index) VALUES (?, ?, ?, ?, ?)',
      [visit_date, day_number || 0, video_title, video_url, order_index || 0]
    );
    res.redirect('/admin/peer-team-visit');
  } catch (error) {
    console.error('Error adding video:', error);
    res.status(500).send('Error adding video');
  }
});
app.get('/admin/peer-team-visit/video/edit/:id', async (req, res) => {
  try {
    if (req.params.id === 'new') {
      res.render('admin/peer-team-visit-video-edit', { video: {} });
    } else {
      const [rows] = await promisePool.execute('SELECT * FROM peer_team_visit WHERE id = ?', [req.params.id]);
      res.render('admin/peer-team-visit-video-edit', { video: rows[0] || {} });
    }
  } catch (error) {
    console.error('Error loading video edit:', error);
    res.status(500).send('Error loading video edit');
  }
});
app.post('/admin/peer-team-visit/video/update/:id', async (req, res) => {
  try {
    const { visit_date, day_number, video_title, video_url, order_index } = req.body;
    await promisePool.execute(
      'UPDATE peer_team_visit SET visit_date = ?, day_number = ?, video_title = ?, video_url = ?, order_index = ? WHERE id = ?',
      [visit_date, day_number || 0, video_title, video_url, order_index || 0, req.params.id]
    );
    res.redirect('/admin/peer-team-visit');
  } catch (error) {
    console.error('Error updating video:', error);
    res.status(500).send('Error updating video');
  }
});
app.get('/admin/peer-team-visit/video/delete/:id', async (req, res) => {
  try {
    await promisePool.execute('DELETE FROM peer_team_visit WHERE id = ?', [req.params.id]);
    res.redirect('/admin/peer-team-visit');
  } catch (error) {
    console.error('Error deleting video:', error);
    res.status(500).send('Error deleting video');
  }
});
app.post('/admin/peer-team-visit/member/add', async (req, res) => {
  try {
    const { member_name, role, designation, institution, location, order_index } = req.body;
    await promisePool.execute(
      'INSERT INTO peer_team_members (member_name, role, designation, institution, location, order_index) VALUES (?, ?, ?, ?, ?, ?)',
      [member_name, role, designation, institution, location, order_index || 0]
    );
    res.redirect('/admin/peer-team-visit');
  } catch (error) {
    console.error('Error adding member:', error);
    res.status(500).send('Error adding member');
  }
});
app.get('/admin/peer-team-visit/member/edit/:id', async (req, res) => {
  try {
    if (req.params.id === 'new') {
      res.render('admin/peer-team-visit-member-edit', { member: {} });
    } else {
      const [rows] = await promisePool.execute('SELECT * FROM peer_team_members WHERE id = ?', [req.params.id]);
      res.render('admin/peer-team-visit-member-edit', { member: rows[0] || {} });
    }
  } catch (error) {
    console.error('Error loading member edit:', error);
    res.status(500).send('Error loading member edit');
  }
});
app.post('/admin/peer-team-visit/member/update/:id', async (req, res) => {
  try {
    const { member_name, role, designation, institution, location, order_index } = req.body;
    await promisePool.execute(
      'UPDATE peer_team_members SET member_name = ?, role = ?, designation = ?, institution = ?, location = ?, order_index = ? WHERE id = ?',
      [member_name, role, designation, institution, location, order_index || 0, req.params.id]
    );
    res.redirect('/admin/peer-team-visit');
  } catch (error) {
    console.error('Error updating member:', error);
    res.status(500).send('Error updating member');
  }
});
app.get('/admin/peer-team-visit/member/delete/:id', async (req, res) => {
  try {
    await promisePool.execute('DELETE FROM peer_team_members WHERE id = ?', [req.params.id]);
    res.redirect('/admin/peer-team-visit');
  } catch (error) {
    console.error('Error deleting member:', error);
    res.status(500).send('Error deleting member');
  }
});

app.get('/iqac', async (req, res) => {
  const [rows] = await promisePool.execute("SELECT * FROM iqac LIMIT 1");
  res.render('iqac', { data: rows[0] || null });
});
app.get('/admin/iqac', async (req, res) => {
  const [rows] = await promisePool.execute("SELECT * FROM iqac LIMIT 1");
  res.render('admin/iqac', { data: rows[0] || {} });
});
app.get('/admin/iqac-edit', (req, res) => {
  res.render('admin/iqac-edit', { data: {} });
});
app.get('/admin/iqac-edit/:id', async (req, res) => {
  const [rows] = await promisePool.execute("SELECT * FROM iqac WHERE id = ?", [req.params.id]);
  res.render('admin/iqac-edit', { data: rows[0] || {} });
});
app.post('/admin/iqac-update', async (req, res) => {
  const form = req.body;

  if (form.id) {
    await promisePool.execute(`
      UPDATE iqac SET 
        page_title = ?, 
        objective_title = ?, objective_content = ?, 
        responsibility_title = ?, responsibility_content = ?, 
        link_text = ?, link_url = ?
      WHERE id = ?
    `, [
      form.page_title,
      form.objective_title, form.objective_content,
      form.responsibility_title, form.responsibility_content,
      form.link_text, form.link_url,
      form.id
    ]);
  } else {
    await promisePool.execute(`
      INSERT INTO iqac 
        (page_title, objective_title, objective_content, responsibility_title, responsibility_content, link_text, link_url)
      VALUES (?, ?, ?, ?, ?, ?, ?)
    `, [
      form.page_title,
      form.objective_title, form.objective_content,
      form.responsibility_title, form.responsibility_content,
      form.link_text, form.link_url
    ]);
  }

  res.redirect('/admin/iqac');
});

app.get("/qif", async (req, res) => {
  try {
    const [criteria] = await promisePool.execute(
      "SELECT * FROM qif_criteria ORDER BY order_index ASC, id ASC"
    );
    res.render("qif", { criteria });
  } catch (error) {
    console.error("Error loading QIF:", error);
    res.status(500).send("Error loading QIF page");
  }
});
app.get('/admin/qif', async (req, res) => {
  try {
    const [criteria] = await promisePool.execute('SELECT * FROM qif_criteria ORDER BY order_index ASC, id ASC');
    res.render('admin/qif', { criteria });
  } catch (error) {
    console.error('Error loading QIF:', error);
    res.status(500).send('Error loading QIF');
  }
});
app.post('/admin/qif/add', async (req, res) => {
  try {
    const { criterion_number, criterion_name, criterion_link, order_index } = req.body;
    await promisePool.execute(
      'INSERT INTO qif_criteria (criterion_number, criterion_name, criterion_link, order_index) VALUES (?, ?, ?, ?)',
      [criterion_number, criterion_name, criterion_link, order_index || 0]
    );
    res.redirect('/admin/qif');
  } catch (error) {
    console.error('Error adding QIF criterion:', error);
    res.status(500).send('Error adding QIF criterion');
  }
});
app.get('/admin/qif/edit/:id', async (req, res) => {
  try {
    if (req.params.id === 'new') {
      res.render('admin/qif-edit', { criterion: {} });
    } else {
      const [rows] = await promisePool.execute('SELECT * FROM qif_criteria WHERE id = ?', [req.params.id]);
      res.render('admin/qif-edit', { criterion: rows[0] || {} });
    }
  } catch (error) {
    console.error('Error loading QIF edit:', error);
    res.status(500).send('Error loading QIF edit');
  }
});
app.post('/admin/qif/update/:id', async (req, res) => {
  try {
    const { criterion_number, criterion_name, criterion_link, order_index } = req.body;
    await promisePool.execute(
      'UPDATE qif_criteria SET criterion_number = ?, criterion_name = ?, criterion_link = ?, order_index = ? WHERE id = ?',
      [criterion_number, criterion_name, criterion_link, order_index || 0, req.params.id]
    );
    res.redirect('/admin/qif');
  } catch (error) {
    console.error('Error updating QIF criterion:', error);
    res.status(500).send('Error updating QIF criterion');
  }
});
app.get('/admin/qif/delete/:id', async (req, res) => {
  try {
    await promisePool.execute('DELETE FROM qif_criteria WHERE id = ?', [req.params.id]);
    res.redirect('/admin/qif');
  } catch (error) {
    console.error('Error deleting QIF criterion:', error);
    res.status(500).send('Error deleting QIF criterion');
  }
});

app.get('/dvv-clarifications', async (req, res) => {
  const [extended] = await promisePool.execute("SELECT * FROM dvv_extended_profile ORDER BY id ASC");
  const [metric] = await promisePool.execute("SELECT * FROM dvv_metric_level ORDER BY id ASC");
  res.render("dvv-clarifications", { extended, metric });
});
app.get('/admin/dvv-clarifications', async (req, res) => {
  const [extended] = await promisePool.execute("SELECT * FROM dvv_extended_profile ORDER BY id ASC");
  const [metric] = await promisePool.execute("SELECT * FROM dvv_metric_level ORDER BY id ASC");
  res.render("admin/dvv-clarifications", { extended, metric });
});
app.get('/admin/dvv-clarifications-edit/:type/:id', async (req, res) => {
  const { type, id } = req.params;
  const tableName = type === "extended" ? "dvv_extended_profile" : "dvv_metric_level";

  if (id === "new") {
    return res.render("admin/dvv-clarifications-edit", { type, data: {} });
  }

  const [rows] = await promisePool.execute(`SELECT * FROM ${tableName} WHERE id = ?`, [id]);
  res.render("admin/dvv-clarifications-edit", { type, data: rows[0] });
});
app.post('/admin/dvv-clarifications-update', async (req, res) => {
  const { id, type, value, pdf_url } = req.body;
  const tableName = type === "extended" ? "dvv_extended_profile" : "dvv_metric_level";

  if (id) {
    await promisePool.execute(
      `UPDATE ${tableName} SET value=?, pdf_url=? WHERE id=?`,
      [value, pdf_url, id]
    );
  } else {
    await promisePool.execute(
      `INSERT INTO ${tableName} (value, pdf_url) VALUES (?, ?)`,
      [value, pdf_url]
    );
  }

  res.redirect('/admin/dvv-clarifications');
});
app.get('/admin/dvv-clarifications-delete/:type/:id', async (req, res) => {
  const { type, id } = req.params;
  const tableName = type === "extended" ? "dvv_extended_profile" : "dvv_metric_level";
  await promisePool.execute(`DELETE FROM ${tableName} WHERE id=?`, [id]);
  res.redirect('/admin/dvv-clarifications');
});

app.get('/strategic-plan', async (req, res) => {
  const [data] = await promisePool.execute(
    "SELECT id, title, pdf_file FROM strategic_plan ORDER BY id DESC"
  );
  res.render('strategic-plan', { data });
});
app.get('/admin/strategic-plan', async (req, res) => {
  const [data] = await promisePool.execute(
    "SELECT id, title, pdf_file FROM strategic_plan ORDER BY id DESC"
  );
  res.render('admin/strategic-plan', { data });
});
app.get('/admin/strategic-plan-edit', (req, res) => {
  res.render('admin/strategic-plan-edit', { data: {} });
});
app.get('/admin/strategic-plan-edit/:id', async (req, res) => {
  const [rows] = await promisePool.execute(
    "SELECT * FROM strategic_plan WHERE id = ?",
    [req.params.id]
  );
  res.render('admin/strategic-plan-edit', { data: rows[0] });
});
app.post('/admin/strategic-plan-update', upload.single("pdf_file"), async (req, res) => {
  const { id, title, oldFile } = req.body;

  let fileName = oldFile;
  if (req.file) {
    fileName = req.file.filename;
  }

  if (id) {
    await promisePool.execute(
      "UPDATE strategic_plan SET title = ?, pdf_file = ? WHERE id = ?",
      [title, fileName, id]
    );
  } else {
    await promisePool.execute(
      "INSERT INTO strategic_plan (title, pdf_file) VALUES (?, ?)",
      [title, fileName]
    );
  }

  res.redirect('/admin/strategic-plan');
});
app.get('/admin/strategic-plan-delete/:id', async (req, res) => {
  await promisePool.execute("DELETE FROM strategic_plan WHERE id = ?", [
    req.params.id,
  ]);
  res.redirect('/admin/strategic-plan');
});

app.get('/annual-report', async (req,res)=>{
  const [data] = await promisePool.execute("SELECT * FROM annual_report ORDER BY id DESC");
  res.render('annual-report', { data });
});
app.get('/admin/annual-report', async (req,res)=>{
  const [data] = await promisePool.execute("SELECT * FROM annual_report ORDER BY id DESC");
  res.render('admin/annual-report', { data });
});
app.get('/admin/annual-report-edit', (req,res)=>{
  res.render('admin/annual-report-edit', { data:{} });
});
app.get('/admin/annual-report-edit/:id', async (req,res)=>{
  const [rows] = await promisePool.execute("SELECT * FROM annual_report WHERE id=?", [req.params.id]);
  res.render('admin/annual-report-edit', { data:rows[0] });
});
app.post('/admin/annual-report-update', upload.single("pdf_file"), async (req,res)=>{
  const { id, title, oldFile } = req.body;

  let fileName = oldFile;
  if (req.file) fileName = req.file.filename;

  if (id) {
    await promisePool.execute("UPDATE annual_report SET title=?, pdf_file=? WHERE id=?", [title, fileName, id]);
  } else {
    await promisePool.execute("INSERT INTO annual_report (title, pdf_file) VALUES (?,?)", [title, fileName]);
  }

  res.redirect('/admin/annual-report');
});

app.get("/institutional-distinctiveness", async (req, res) => {
  try {
    const [topRows] = await promisePool.execute(
      "SELECT * FROM institutional_distinctiveness_top LIMIT 1"
    );

    const [tableRows] = await promisePool.execute(
      "SELECT * FROM institutional_distinctiveness_table LIMIT 1"
    );

    const [bottomRows] = await promisePool.execute(
      "SELECT * FROM institutional_distinctiveness_bottom LIMIT 1"
    );

    res.render("institutional-distinctiveness", {
      top: topRows[0] || null,
      table: tableRows[0] || null,
      bottom: bottomRows[0] || null,
    });
  } catch (err) {
    console.log("Error loading institutional distinctiveness:", err);
    res.status(500).send("Error loading page");
  }
});
app.get("/admin/institutional-distinctiveness", async (req, res) => {
  try {
    const sections = [];

    const [top] = await promisePool.execute(
      "SELECT *, 'top' AS section_type FROM institutional_distinctiveness_top"
    );
    const [table] = await promisePool.execute(
      "SELECT *, 'table' AS section_type FROM institutional_distinctiveness_table"
    );
    const [bottom] = await promisePool.execute(
      "SELECT *, 'bottom' AS section_type FROM institutional_distinctiveness_bottom"
    );

    sections.push(...top, ...table, ...bottom);

    res.render("admin/institutional-distinctiveness", { sections });
  } catch (err) {
    console.log("Admin load error:", err);
    res.status(500).send("Error loading admin page");
  }
});
app.get("/admin/institutional-distinctiveness/edit/:id", async (req, res) => {
  try {
    const id = req.params.id;

    if (id === "new") {
      return res.render("admin/institutional-distinctiveness-edit", {
        data: {},
      });
    }

    // Find record in ALL 3 tables
    let record;

    const [top] = await promisePool.execute(
      "SELECT *, 'top' AS section_type FROM institutional_distinctiveness_top WHERE id = ?",
      [id]
    );
    if (top.length > 0) record = top[0];

    const [table] = await promisePool.execute(
      "SELECT *, 'table' AS section_type FROM institutional_distinctiveness_table WHERE id = ?",
      [id]
    );
    if (!record && table.length > 0) record = table[0];

    const [bottom] = await promisePool.execute(
      "SELECT *, 'bottom' AS section_type FROM institutional_distinctiveness_bottom WHERE id = ?",
      [id]
    );
    if (!record && bottom.length > 0) record = bottom[0];

    res.render("admin/institutional-distinctiveness-edit", {
      data: record || {},
    });
  } catch (err) {
    console.log("Admin edit error:", err);
    res.status(500).send("Error loading edit page");
  }
});
app.post("/admin/institutional-distinctiveness/update", async (req, res) => {
  try {
    const { id, section_type, title, subtitle, content } = req.body;

    // ---------------- UPDATE EXISTING RECORD ----------------
    if (id) {
      if (section_type === "top") {
        await promisePool.execute(
          "UPDATE institutional_distinctiveness_top SET title=?, subtitle=?, content=? WHERE id=?",
          [title, subtitle, content, id]
        );
      }

      if (section_type === "table") {
        await promisePool.execute(
          "UPDATE institutional_distinctiveness_table SET title=?, subtitle=?, content=? WHERE id=?",
          [title, subtitle, content, id]
        );
      }

      if (section_type === "bottom") {
        await promisePool.execute(
          "UPDATE institutional_distinctiveness_bottom SET title=?, subtitle=?, content=? WHERE id=?",
          [title, subtitle, content, id]
        );
      }

      return res.redirect("/admin/institutional-distinctiveness");
    }

    // ---------------- INSERT NEW RECORD ----------------
    if (section_type === "top") {
      await promisePool.execute(
        "INSERT INTO institutional_distinctiveness_top (title, subtitle, content) VALUES (?, ?, ?)",
        [title, subtitle, content]
      );
    }

    if (section_type === "table") {
      await promisePool.execute(
        "INSERT INTO institutional_distinctiveness_table (title, subtitle, content) VALUES (?, ?, ?)",
        [title, subtitle, content]
      );
    }

    if (section_type === "bottom") {
      await promisePool.execute(
        "INSERT INTO institutional_distinctiveness_bottom (title, subtitle, content) VALUES (?, ?, ?)",
        [title, subtitle, content]
      );
    }

    res.redirect("/admin/institutional-distinctiveness");
  } catch (err) {
    console.log("Update error:", err);
    res.status(500).send("Error saving data");
  }
});
app.get("/admin/institutional-distinctiveness/delete/:id", async (req, res) => {
  try {
    const id = req.params.id;

    await promisePool.execute(
      "DELETE FROM institutional_distinctiveness_top WHERE id=?",
      [id]
    );
    await promisePool.execute(
      "DELETE FROM institutional_distinctiveness_table WHERE id=?",
      [id]
    );
    await promisePool.execute(
      "DELETE FROM institutional_distinctiveness_bottom WHERE id=?",
      [id]
    );

    res.redirect("/admin/institutional-distinctiveness");
  } catch (err) {
    console.log("Delete error:", err);
    res.status(500).send("Error deleting section");
  }
});

app.get("/best-practices", async (req, res) => {
  const [rows] = await promisePool.execute("SELECT * FROM best_practices LIMIT 1");
  res.render("best-practices", { data: rows[0] });
});
app.get("/admin/best-practices", async (req, res) => {
  const [rows] = await promisePool.execute("SELECT * FROM best_practices LIMIT 1");
  res.render("admin/best-practices", { data: rows[0] });
});
app.get("/admin/best-practices-edit/:id", async (req, res) => {
  const [rows] = await promisePool.execute("SELECT * FROM best_practices WHERE id=?", [req.params.id]);
  res.render("admin/best-practices-edit", { data: rows[0] });
});
app.post("/admin/best-practices-update",
  upload.fields([{ name: "pdf1_file" }, { name: "pdf2_file" }]),
  async (req, res) => {

    let pdf1 = req.body.oldPdf1;
    let pdf2 = req.body.oldPdf2;

    if (req.files.pdf1_file) pdf1 = req.files.pdf1_file[0].filename;
    if (req.files.pdf2_file) pdf2 = req.files.pdf2_file[0].filename;

    await promisePool.execute(
      `UPDATE best_practices SET 
       content1=?, pdf1_title=?, pdf1_file=?, 
       content2=?, pdf2_title=?, pdf2_file=? 
       WHERE id=?`,
      [
        req.body.content1,
        req.body.pdf1_title,
        pdf1,
        req.body.content2,
        req.body.pdf2_title,
        pdf2,
        req.body.id
      ]
    );

    res.redirect("/admin/best-practices");
  }
);

function crudRoutes(table, route) {
  // FRONTEND
  app.get(`/${route}`, async (req, res) => {
    const [list] = await promisePool.execute(`SELECT * FROM ${table}`);
    res.render(route, { list });
  });

  // ADMIN
  app.get(`/admin/${route}`, async (req, res) => {
    const [list] = await promisePool.execute(`SELECT * FROM ${table}`);
    res.render(`admin/${route}`, { list });
  });

  // ADD
  app.get(`/admin/${route}-add`, (req, res) => {
    res.render(`admin/${route}-edit`, { data: {} });
  });

  // EDIT
  app.get(`/admin/${route}-edit/:id`, async (req, res) => {
    const [rows] = await promisePool.execute(`SELECT * FROM ${table} WHERE id=?`, [req.params.id]);
    res.render(`admin/${route}-edit`, { data: rows[0] });
  });

  // UPDATE
  app.post(`/admin/${route}-update`,
    upload.single("title_link"),
    async (req, res) => {
      let file = req.body.oldFile;
      if (req.file) file = req.file.filename;

      if (req.body.id) {
        await promisePool.execute(
          `UPDATE ${table} SET title=?, title_link=? WHERE id=?`,
          [req.body.title, file, req.body.id]
        );
      } else {
        await promisePool.execute(
          `INSERT INTO ${table} (title, title_link) VALUES (?, ?)`,
          [req.body.title, file]
        );
      }

      res.redirect(`/admin/${route}`);
    }
  );

  // DELETE
  app.post(`/admin/${route}-delete`, async (req, res) => {
    await promisePool.execute(`DELETE FROM ${table} WHERE id=?`, [req.body.id]);
    res.redirect(`/admin/${route}`);
  });
}
crudRoutes("feedback", "feedback");
crudRoutes("auditor_statement", "auditor-statement");
crudRoutes("student_satisfactory_survey", "student-satisfactory-survey");

app.get("/committee", (req, res) => {
  db.query("SELECT * FROM committee_top LIMIT 1", (err, topResult) => {
    if (err) throw err;

    db.query("SELECT * FROM committee_minutes ORDER BY id ASC", (err, minutes) => {
      if (err) throw err;

      res.render("committee", {
        top: topResult[0],
        minutes
      });
    });
  });
});
app.get("/admin/committee", (req, res) => {
  db.query("SELECT * FROM committee_top LIMIT 1", (err, top) => {
    if (err) throw err;

    db.query("SELECT * FROM committee_minutes ORDER BY id ASC", (err, minutes) => {
      if (err) throw err;

      res.render("admin/committee", {
        top: top[0],
        minutes
      });
    });
  });
});
app.get("/admin/committee-top-edit/:id", (req, res) => {
  db.query(
    "SELECT * FROM committee_top WHERE id=?",
    [req.params.id],
    (err, result) => {
      if (err) throw err;
      res.render("admin/committee-edit", {
        data: result[0],
        type: "top"
      });
    }
  );
});
app.get("/admin/committee-minutes-add", (req, res) => {
  res.render("admin/committee-edit", {
    data: {},
    type: "minutes"
  });
});
app.get("/admin/committee-minutes-edit/:id", (req, res) => {
  db.query(
    "SELECT * FROM committee_minutes WHERE id=?",
    [req.params.id],
    (err, rows) => {
      if (err) throw err;
      res.render("admin/committee-edit", {
        data: rows[0],
        type: "minutes"
      });
    }
  );
});
app.post("/admin/committee-update",
  uploadCommittee.single("committee_details"),
  (req, res) => {
    const { id, type } = req.body;

    // ===== TOP SECTION =====
    if (type === "top") {
      const { title, content, minutes_heading } = req.body;

      db.query(
        "UPDATE committee_top SET title=?, content=?, minutes_heading=? WHERE id=?",
        [title, content, minutes_heading, id],
        () => res.redirect("/admin/committee")
      );

    // ===== MINUTES SECTION =====
    } else {
      const { committee_name, oldPdf } = req.body;
      const pdfFile = req.file ? req.file.filename : oldPdf;

      // ADD
      if (!id) {
        db.query(
          "INSERT INTO committee_minutes (committee_name, committee_details) VALUES (?,?)",
          [committee_name, pdfFile],
          () => res.redirect("/admin/committee")
        );
      }
      // UPDATE
      else {
        db.query(
          "UPDATE committee_minutes SET committee_name=?, committee_details=? WHERE id=?",
          [committee_name, pdfFile, id],
          () => res.redirect("/admin/committee")
        );
      }
    }
  }
);
app.post("/admin/committee-minutes-delete", (req, res) => {
  db.query(
    "DELETE FROM committee_minutes WHERE id=?",
    [req.body.id],
    () => res.redirect("/admin/committee")
  );
});

app.get('/admin/committees', async (req, res) => {
  try {
    const [committees] = await promisePool.execute(`
      SELECT committee_slug, COUNT(*) as member_count 
      FROM committee_members 
      GROUP BY committee_slug 
      ORDER BY committee_slug
    `);
    
    const committeeList = [
      { slug: 'grievance-redressal-committee', name: 'Grievance Redressal Committee' },
      { slug: 'women-empowerment-cell', name: 'Women Empowerment Cell' },
      { slug: 'anti-ragging-committee', name: 'Anti-Ragging Committee' },
      { slug: 'training-placement-cell', name: 'Training & Placement Cell' },
      { slug: 'students-grievance-cell', name: 'Students Grievance Cell' },
      { slug: 'startup-innovation-cell', name: 'Startup & Innovation Cell' },
      { slug: 'staff-selection-committee', name: 'Staff Selection Committee' },
      { slug: 'sc-st-cell', name: 'SC/ST Cell' },
      { slug: 'alumni-cell', name: 'Alumni Cell' },
      { slug: 'anti-ragging-cell', name: 'Anti-Ragging Cell' },
      { slug: 'class-committee-meeting', name: 'Class Committee Meeting' },
      { slug: 'college-maintenance-committee', name: 'College Maintenance Committee' },
      { slug: 'complaints-cum-redressal', name: 'Complaints cum Redressal' },
      { slug: 'department-advisory-committee', name: 'Department Advisory Committee' },
      { slug: 'discipline-and-welfare', name: 'Discipline and Welfare' },
      { slug: 'discipline-committee', name: 'Discipline Committee' },
      { slug: 'exam-cell', name: 'Exam Cell' },
      { slug: 'admission-committee', name: 'Admission Committee' },
      { slug: 'excess-fee-inspection-committee', name: 'Excess Fee Inspection Committee' },
      { slug: 'faculty-welfare-cell', name: 'Faculty Welfare Cell' },
      { slug: 'governing-council', name: 'Governing Council' },
      { slug: 'internal-complaint-committee', name: 'Internal Complaint Committee' },
      { slug: 'planning-and-monitoring', name: 'Planning and Monitoring' },
    ];

    const committeesWithCount = committeeList.map(committee => {
      const found = committees.find(c => c.committee_slug === committee.slug);
      return {
        ...committee,
        member_count: found ? found.member_count : 0
      };
    });

    res.render('admin/committees', { committees: committeesWithCount });
  } catch (error) {
    console.error('Error loading committees:', error);
    res.status(500).send('Error loading committees');
  }
});
app.get('/admin/committees/:slug', async (req, res) => {
  try {
    const { slug } = req.params;
    const [members] = await promisePool.execute(
      'SELECT * FROM committee_members WHERE committee_slug = ? ORDER BY category, id ASC',
      [slug]
    );

    const committeeNames = {
      'grievance-redressal-committee': 'Grievance Redressal Committee',
      'women-empowerment-cell': 'Women Empowerment Cell',
      'anti-ragging-committee': 'Anti-Ragging Committee',
      'training-placement-cell': 'Training & Placement Cell',
      'students-grievance-cell': 'Students Grievance Cell',
      'startup-innovation-cell': 'Startup & Innovation Cell',
      'staff-selection-committee': 'Staff Selection Committee',
      'sc-st-cell': 'SC/ST Cell',
      'alumni-cell': 'Alumni Cell',
      'anti-ragging-cell': 'Anti-Ragging Cell',
      'class-committee-meeting': 'Class Committee Meeting',
      'college-maintenance-committee': 'College Maintenance Committee',
      'complaints-cum-redressal': 'Complaints cum Redressal',
      'department-advisory-committee': 'Department Advisory Committee',
      'discipline-and-welfare': 'Discipline and Welfare',
      'discipline-committee': 'Discipline Committee',
      'exam-cell': 'Exam Cell',
      'admission-committee': 'Admission Committee',
      'excess-fee-inspection-committee': 'Excess Fee Inspection Committee',
      'faculty-welfare-cell': 'Faculty Welfare Cell',
      'governing-council': 'Governing Council',
      'internal-complaint-committee': 'Internal Complaint Committee',
      'planning-and-monitoring': 'Planning and Monitoring'
    };

    res.render('admin/committee-members', {
      slug,
      committeeName: committeeNames[slug] || slug,
      members
    });
  } catch (error) {
    console.error('Error loading committee members:', error);
    res.status(500).send('Error loading committee members');
  }
});
app.post('/admin/committees/:slug/add', async (req, res) => {
  try {
    const { slug } = req.params;
    const { member_name, role, designation, category, year } = req.body;
    await promisePool.execute(
      'INSERT INTO committee_members (committee_slug, member_name, role, designation, category, year) VALUES (?, ?, ?, ?, ?, ?)',
      [slug, member_name, role || null, designation || null, category || null, year || null]
    );
    res.redirect(`/admin/committees/${slug}`);
  } catch (error) {
    console.error('Error adding committee member:', error);
    res.status(500).send('Error adding committee member');
  }
});
app.get('/admin/committees/:slug/edit/:id', async (req, res) => {
  try {
    const { slug, id } = req.params;
    const [rows] = await promisePool.execute('SELECT * FROM committee_members WHERE id = ? AND committee_slug = ?', [id, slug]);
    
    const committeeNames = {
      'grievance-redressal-committee': 'Grievance Redressal Committee',
      'women-empowerment-cell': 'Women Empowerment Cell',
      'anti-ragging-committee': 'Anti-Ragging Committee',
      'training-placement-cell': 'Training & Placement Cell',
      'students-grievance-cell': 'Students Grievance Cell',
      'startup-innovation-cell': 'Startup & Innovation Cell',
      'staff-selection-committee': 'Staff Selection Committee',
      'sc-st-cell': 'SC/ST Cell',
      'alumni-cell': 'Alumni Cell',
      'anti-ragging-cell': 'Anti-Ragging Cell',
      'class-committee-meeting': 'Class Committee Meeting',
      'college-maintenance-committee': 'College Maintenance Committee',
      'complaints-cum-redressal': 'Complaints cum Redressal',
      'department-advisory-committee': 'Department Advisory Committee',
      'discipline-and-welfare': 'Discipline and Welfare',
      'discipline-committee': 'Discipline Committee',
      'exam-cell': 'Exam Cell',
      'admission-committee': 'Admission Committee',
      'excess-fee-inspection-committee': 'Excess Fee Inspection Committee',
      'faculty-welfare-cell': 'Faculty Welfare Cell',
      'governing-council': 'Governing Council',
      'internal-complaint-committee': 'Internal Complaint Committee',
      'planning-and-monitoring': 'Planning and Monitoring'
    };

    res.render('admin/committee-member-edit', {
      slug,
      committeeName: committeeNames[slug] || slug,
      member: rows[0] || {}
    });
  } catch (error) {
    console.error('Error loading committee member edit:', error);
    res.status(500).send('Error loading committee member edit');
  }
});
app.post('/admin/committees/:slug/update/:id', async (req, res) => {
  try {
    const { slug, id } = req.params;
    const { member_name, role, designation, category, year } = req.body;
    await promisePool.execute(
      'UPDATE committee_members SET member_name = ?, role = ?, designation = ?, category = ?, year = ? WHERE id = ? AND committee_slug = ?',
      [member_name, role || null, designation || null, category || null, year || null, id, slug]
    );
    res.redirect(`/admin/committees/${slug}`);
  } catch (error) {
    console.error('Error updating committee member:', error);
    res.status(500).send('Error updating committee member');
  }
});
app.get('/admin/committees/:slug/delete/:id', async (req, res) => {
  try {
    const { slug, id } = req.params;
    await promisePool.execute('DELETE FROM committee_members WHERE id = ? AND committee_slug = ?', [id, slug]);
    res.redirect(`/admin/committees/${slug}`);
  } catch (error) {
    console.error('Error deleting committee member:', error);
    res.status(500).send('Error deleting committee member');
  }
});

app.get("/grievance-redressal-committee", (req, res) => {
  const q1 = "SELECT * FROM grievance_redressal_committee_student";
  const q2 = "SELECT * FROM grievance_redressal_committee_staff";

  db.query(q1, (err, student) => {
    if (err) throw err;

    db.query(q2, (err, staff) => {
      if (err) throw err;

      res.render("grievance-redressal-committee", {
        student,
        staff
      });
    });
  });
});
app.get("/admin/grievance-redressal-committee", (req, res) => {
  const sQuery = "SELECT * FROM grievance_redressal_committee_student";
  const fQuery = "SELECT * FROM grievance_redressal_committee_staff";

  db.query(sQuery, (err, student) => {
    if (err) throw err;

    db.query(fQuery, (err, staff) => {
      if (err) throw err;

      res.render("admin/grievance-redressal-committee", {
        student,
        staff
      });
    });
  });
});
app.get("/admin/grievance-redressal-committee-student-add", (req, res) => {
  res.render("admin/grievance-redressal-committee-edit", {
    data: {},
    type: "student"
  });
});
app.post("/admin/grievance-redressal-committee-student-add", (req, res) => {
  const { title, member_name, role, designation } = req.body;

  const sql = `
      INSERT INTO grievance_redressal_committee_student 
      (title, member_name, role, designation) VALUES (?, ?, ?, ?)
  `;

  db.query(sql, [title, member_name, role, designation], (err) => {
    if (err) throw err;

    res.redirect("/admin/grievance-redressal-committee");
  });
});
app.get("/admin/grievance-redressal-committee-staff-add", (req, res) => {
  res.render("admin/grievance-redressal-committee-edit", {
    data: {},
    type: "staff"
  });
});
app.post("/admin/grievance-redressal-committee-staff-add", (req, res) => {
  const { title, member_name, role, designation } = req.body;

  const sql = `
      INSERT INTO grievance_redressal_committee_staff
      (title, member_name, role, designation) VALUES (?, ?, ?, ?)
  `;

  db.query(sql, [title, member_name, role, designation], (err) => {
    if (err) throw err;

    res.redirect("/admin/grievance-redressal-committee");
  });
});
app.get("/admin/grievance-redressal-committee-edit/:id", (req, res) => {
  const { id } = req.params;
  const { type } = req.query;

  const table =
    type === "student"
      ? "grievance_redressal_committee_student"
      : "grievance_redressal_committee_staff";

  db.query(`SELECT * FROM ${table} WHERE id = ?`, [id], (err, result) => {
    if (err) throw err;

    res.render("admin/grievance-redressal-committee-edit", {
      data: result[0],
      type
    });
  });
});
app.post("/admin/grievance-redressal-committee-update", (req, res) => {
  const { id, type, title, member_name, role, designation } = req.body;

  const table =
    type === "student"
      ? "grievance_redressal_committee_student"
      : "grievance_redressal_committee_staff";

  const sql = `
      UPDATE ${table} SET 
      title = ?, member_name = ?, role = ?, designation = ?
      WHERE id = ?
  `;

  db.query(sql, [title, member_name, role, designation, id], (err) => {
    if (err) throw err;

    res.redirect("/admin/grievance-redressal-committee");
  });
});
app.post("/admin/grievance-redressal-committee-delete", (req, res) => {
  const { id, type } = req.body;

  const table =
    type === "student"
      ? "grievance_redressal_committee_student"
      : "grievance_redressal_committee_staff";

  db.query(`DELETE FROM ${table} WHERE id = ?`, [id], (err) => {
    if (err) throw err;

    res.redirect("/admin/grievance-redressal-committee");
  });
});

app.get("/women-empowerment-cell", (req, res) => {
    db.query("SELECT * FROM women_empowerment_cell_content LIMIT 1", (err, content) => {
        if (err) throw err;

        db.query("SELECT * FROM women_empowerment_cell_table", (err, list) => {
            res.render("women-empowerment-cell", {
                content: content[0],
                list
            });
        });
    });
});
app.get("/admin/women-empowerment-cell", (req, res) => {
    db.query("SELECT * FROM women_empowerment_cell_content LIMIT 1", (err, content) => {
        db.query("SELECT * FROM women_empowerment_cell_table", (err, list) => {
            res.render("admin/women-empowerment-cell", {
                content: content[0],
                list
            });
        });
    });
});
app.get("/admin/women-empowerment-cell-content-edit/:id", (req, res) => {
    db.query("SELECT * FROM women_empowerment_cell_content WHERE id=?", [req.params.id], (err, data) => {
        res.render("admin/women-empowerment-cell-edit", {
            data: data[0],
            type: "content"
        });
    });
});
app.get("/admin/women-empowerment-cell-edit/:id", (req, res) => {
    db.query("SELECT * FROM women_empowerment_cell_table WHERE id=?", [req.params.id], (err, data) => {
        res.render("admin/women-empowerment-cell-edit", {
            data: data[0],
            type: "table"
        });
    });
});
app.get("/admin/women-empowerment-cell-add", (req, res) => {
    res.render("admin/women-empowerment-cell-edit", {
        data: {},
        type: "table"
    });
});
app.post("/admin/women-empowerment-cell-update", upload.single("image"), upload.single("link"), (req, res) => {
    const { id, type } = req.body;

    if (type === "content") {

        let imageFile = req.body.oldImage;
        if (req.file) imageFile = req.file.filename;

        db.query(
            `UPDATE women_empowerment_cell_content 
             SET title=?, image=?, content=? WHERE id=?`,
            [
                req.body.title,
                imageFile,
                req.body.content,
                id
            ], () => res.redirect("/admin/women-empowerment-cell")
        );

    } else {

        let pdfFile = req.body.oldPdf;
        if (req.file) pdfFile = req.file.filename;

        db.query(
            `UPDATE women_empowerment_cell_table 
             SET year=?, details1=?, details2=?, link=? WHERE id=?`,
            [
                req.body.year,
                req.body.details1,
                req.body.details2,
                pdfFile,
                id
            ],
            () => res.redirect("/admin/women-empowerment-cell")
        );
    }
});
app.post("/admin/women-empowerment-cell-delete", (req, res) => {
    db.query("DELETE FROM women_empowerment_cell_table WHERE id=?", [req.body.id], () => {
        res.redirect("/admin/women-empowerment-cell");
    });
});

app.get("/anti-ragging-committee", (req, res) => {
    db.query("SELECT * FROM anti_ragging_header LIMIT 1", (err, header) => {
        db.query("SELECT * FROM anti_ragging_committee", (err, members) => {
            res.render("anti-ragging-committee", {
                header: header[0],
                members
            });
        });
    });
});
app.get("/admin/anti-ragging-committee", (req, res) => {
    db.query("SELECT * FROM anti_ragging_header LIMIT 1", (err, header) => {
        db.query("SELECT * FROM anti_ragging_committee", (err, members) => {
            res.render("admin/anti-ragging-committee", {
                header: header[0],
                members
            });
        });
    });
});
app.get("/admin/anti-ragging-header-edit/:id", (req, res) => {
    db.query("SELECT * FROM anti_ragging_header WHERE id=?", [req.params.id], (err, data) => {
        res.render("admin/anti-ragging-edit", {
            data: data[0],
            type: "header"
        });
    });
});
app.get("/admin/anti-ragging-member-edit/:id", (req, res) => {
    db.query("SELECT * FROM anti_ragging_committee WHERE id=?", [req.params.id], (err, data) => {
        res.render("admin/anti-ragging-edit", {
            data: data[0],
            type: "member"
        });
    });
});
app.get("/admin/anti-ragging-member-add", (req, res) => {
    res.render("admin/anti-ragging-edit", {
        data: {},
        type: "member"
    });
});
app.post("/admin/anti-ragging-update", (req, res) => {
    const { id, type } = req.body;

    if (type === "header") {
        db.query(
            `UPDATE anti_ragging_header SET title=?, subtitle=?, contact_no=? WHERE id=?`,
            [req.body.title, req.body.subtitle, req.body.contact_no, id],
            () => res.redirect("/admin/anti-ragging-committee")
        );
    } else {
        db.query(
            `UPDATE anti_ragging_committee 
             SET committee_type=?, committee_member=?, profession=?, associated_details=? 
             WHERE id=?`,
            [
                req.body.committee_type,
                req.body.committee_member,
                req.body.profession,
                req.body.associated_details,
                id
            ],
            () => res.redirect("/admin/anti-ragging-committee")
        );
    }
});
app.post("/admin/anti-ragging-member-delete", (req, res) => {
    db.query("DELETE FROM anti_ragging_committee WHERE id=?", [req.body.id], () => {
        res.redirect("/admin/anti-ragging-committee");
    });
});

app.get("/training-placement-cell", (req, res) => {
    db.query("SELECT * FROM training_placement_vision LIMIT 1", (err, vision) => {
        db.query("SELECT * FROM training_placement_mission LIMIT 1", (err, mission) => {
            db.query("SELECT * FROM training_placement_coordinators", (err, coordinators) => {
                db.query("SELECT * FROM training_placement_documents", (err, documents) => {
                    res.render("training-placement-cell", {
                        vision: vision[0],
                        mission: mission[0],
                        coordinators,
                        documents
                    });
                });
            });
        });
    });
});
app.get("/admin/training-placement-cell", (req, res) => {
    db.query("SELECT * FROM training_placement_vision LIMIT 1", (err, vision) => {
        db.query("SELECT * FROM training_placement_mission LIMIT 1", (err, mission) => {
            db.query("SELECT * FROM training_placement_coordinators", (err, coordinators) => {
                db.query("SELECT * FROM training_placement_documents", (err, documents) => {
                    res.render("admin/training-placement-cell", {
                        vision: vision[0],
                        mission: mission[0],
                        coordinators,
                        documents
                    });
                });
            });
        });
    });
});
app.get("/admin/training-placement-vision-edit/:id", (req, res) => {
    db.query("SELECT * FROM training_placement_vision WHERE id=?", [req.params.id], (err, data) => {
        res.render("admin/training-placement-cell-edit", {
            data: data[0],
            type: "vision"
        });
    });
});
app.get("/admin/training-placement-mission-edit/:id", (req, res) => {
    db.query("SELECT * FROM training_placement_mission WHERE id=?", [req.params.id], (err, data) => {
        res.render("admin/training-placement-cell-edit", {
            data: data[0],
            type: "mission"
        });
    });
});
app.get("/admin/training-placement-coordinator-edit/:id", (req, res) => {
    db.query("SELECT * FROM training_placement_coordinators WHERE id=?", [req.params.id], (err, data) => {
        res.render("admin/training-placement-cell-edit", {
            data: data[0],
            type: "coordinator"
        });
    });
});
app.get("/admin/training-placement-coordinator-add", (req, res) => {
    res.render("admin/training-placement-cell-edit", {
        data: {},
        type: "coordinator"
    });
});
app.get("/admin/training-placement-document-edit/:id", (req, res) => {
    db.query("SELECT * FROM training_placement_documents WHERE id=?", [req.params.id], (err, data) => {
        res.render("admin/training-placement-cell-edit", {
            data: data[0],
            type: "document"
        });
    });
});
app.get("/admin/training-placement-document-add", (req, res) => {
    res.render("admin/training-placement-cell-edit", {
        data: {},
        type: "document"
    });
});
app.post("/admin/training-placement-update", upload.single("link"), (req, res) => {
    const { id, type } = req.body;

    if (type === "vision" || type === "mission") {
        db.query(
            `UPDATE training_placement_${type} SET title=?, content=? WHERE id=?`,
            [req.body.title, req.body.content, id],
            () => res.redirect("/admin/training-placement-cell")
        );

    } else if (type === "coordinator") {
        db.query(
            `UPDATE training_placement_coordinators SET coordinator=?, department=? WHERE id=?`,
            [req.body.coordinator, req.body.department, id],
            () => res.redirect("/admin/training-placement-cell")
        );

    } else if (type === "document") {
        let pdf = req.body.oldPdf;
        if (req.file) pdf = req.file.filename;

        db.query(
            `UPDATE training_placement_documents SET title=?, link=? WHERE id=?`,
            [req.body.title, pdf, id],
            () => res.redirect("/admin/training-placement-cell")
        );
    }
});
app.post("/admin/training-placement-coordinator-delete", (req, res) => {
    db.query("DELETE FROM training_placement_coordinators WHERE id=?", [req.body.id], () => {
        res.redirect("/admin/training-placement-cell");
    });
});
app.post("/admin/training-placement-document-delete", (req, res) => {
    db.query("DELETE FROM training_placement_documents WHERE id=?", [req.body.id], () => {
        res.redirect("/admin/training-placement-cell");
    });
});

app.get("/students-grievance-cell", (req, res) => {
    db.query("SELECT * FROM students_grievance_cell_content LIMIT 1", (err, content) => {
        db.query("SELECT * FROM students_grievance_cell_table", (err, list) => {
            res.render("students-grievance-cell", {
                content: content[0],
                list
            });
        });
    });
});
app.get("/admin/students-grievance-cell", (req, res) => {
    db.query("SELECT * FROM students_grievance_cell_content LIMIT 1", (err, content) => {
        db.query("SELECT * FROM students_grievance_cell_table", (err, list) => {
            res.render("admin/students-grievance-cell", {
                content: content[0],
                list
            });
        });
    });
});
app.get("/admin/students-grievance-cell-content-edit/:id", (req, res) => {
    db.query("SELECT * FROM students_grievance_cell_content WHERE id=?", [req.params.id], (err, data) => {
        res.render("admin/students-grievance-cell-edit", { data: data[0], type: "content" });
    });
});
app.get("/admin/students-grievance-cell-edit/:id", (req, res) => {
    db.query("SELECT * FROM students_grievance_cell_table WHERE id=?", [req.params.id], (err, data) => {
        res.render("admin/students-grievance-cell-edit", { data: data[0], type: "table" });
    });
});
app.get("/admin/students-grievance-cell-add", (req, res) => {
    res.render("admin/students-grievance-cell-edit", { data: {}, type: "table" });
});
app.post("/admin/students-grievance-cell-update", upload.single("image"), upload.single("link"), (req, res) => {
    const { id, type } = req.body;

    if (type === "content") {

        let image = req.body.oldImage;
        if (req.file) image = req.file.filename;

        db.query(
            `UPDATE students_grievance_cell_content 
             SET title=?, image=?, content=? WHERE id=?`,
            [req.body.title, image, req.body.content, id],
            () => res.redirect("/admin/students-grievance-cell")
        );

    } else {

        let pdf = req.body.oldPdf;
        if (req.file) pdf = req.file.filename;

        db.query(
            `UPDATE students_grievance_cell_table 
             SET year=?, details1=?, details2=?, link=? WHERE id=?`,
            [req.body.year, req.body.details1, req.body.details2, pdf, id],
            () => res.redirect("/admin/students-grievance-cell")
        );
    }
});
app.post("/admin/students-grievance-cell-delete", (req, res) => {
    db.query("DELETE FROM students_grievance_cell_table WHERE id=?", [req.body.id], () => {
        res.redirect("/admin/students-grievance-cell");
    });
});

app.get("/staff-selection-committee", (req, res) => {
    db.query("SELECT * FROM staff_selection_committee_content LIMIT 1", (err, data) => {
        res.render("staff-selection-committee", { content: data[0] });
    });
});
app.get("/admin/staff-selection-committee", (req, res) => {
    db.query("SELECT * FROM staff_selection_committee_content LIMIT 1", (err, data) => {
        res.render("admin/staff-selection-committee", { content: data[0] });
    });
});
app.get("/admin/staff-selection-committee-edit/:id", (req, res) => {
    db.query(
        "SELECT * FROM staff_selection_committee_content WHERE id=?",
        [req.params.id],
        (err, data) => {
            res.render("admin/staff-selection-committee-edit", { data: data[0] });
        }
    );
});
app.post("/admin/staff-selection-committee-update",
    upload.single("image"),
    (req, res) => {
        let image = req.body.oldImage;
        if (req.file) image = req.file.filename;

        db.query(
            `UPDATE staff_selection_committee_content 
             SET title=?, image=?, content=? WHERE id=?`,
            [req.body.title, image, req.body.content, req.body.id],
            () => res.redirect("/admin/staff-selection-committee")
        );
});

app.get("/sc-st-cell", (req, res) => {
    db.query("SELECT * FROM sc_st_cell_content LIMIT 1", (err, content) => {
        db.query("SELECT * FROM sc_st_cell_table", (err, table) => {
            res.render("sc-st-cell", { content: content[0], table });
        });
    });
});
app.get("/admin/sc-st-cell", (req, res) => {
    db.query("SELECT * FROM sc_st_cell_content LIMIT 1", (err, content) => {
        db.query("SELECT * FROM sc_st_cell_table", (err, table) => {
            res.render("admin/sc-st-cell", { content: content[0], table });
        });
    });
});
app.get("/admin/sc-st-cell-content-edit/:id", (req, res) => {
    db.query("SELECT * FROM sc_st_cell_content WHERE id=?",
        [req.params.id],
        (err, data) => {
            res.render("admin/sc-st-cell-edit", { data: data[0], type: "content" });
        });
});
app.get("/admin/sc-st-cell-edit/:id", (req, res) => {
    db.query("SELECT * FROM sc_st_cell_table WHERE id=?",
        [req.params.id],
        (err, data) => {
            res.render("admin/sc-st-cell-edit", { data: data[0], type: "table" });
        });
});
app.get("/admin/sc-st-cell-add", (req, res) => {
    res.render("admin/sc-st-cell-edit", { data: {}, type: "table" });
});
app.post("/admin/sc-st-cell-update",
    upload.single("image"),
    (req, res) => {
        const { id, type } = req.body;

        if (type === "content") {

            let img = req.body.oldImage;
            if (req.file) img = req.file.filename;

            db.query(
                `UPDATE sc_st_cell_content SET title=?, image=?, content=? WHERE id=?`,
                [req.body.title, img, req.body.content, id],
                () => res.redirect("/admin/sc-st-cell")
            );

        } else {

            db.query(
                `UPDATE sc_st_cell_table 
                 SET committee_member=?, profession=?, associated_details=? WHERE id=?`,
                [
                    req.body.committee_member,
                    req.body.profession,
                    req.body.associated_details,
                    id
                ],
                () => res.redirect("/admin/sc-st-cell")
            );
        }
});
app.post("/admin/sc-st-cell-delete", (req, res) => {
    db.query("DELETE FROM sc_st_cell_table WHERE id=?", [req.body.id], () => {
        res.redirect("/admin/sc-st-cell");
    });
});

app.get("/alumni-cell", (req, res) => {
    db.query("SELECT * FROM alumni_cell_content LIMIT 1", (err, content) => {
        db.query("SELECT * FROM alumni_cell_table", (err, table) => {
            res.render("alumni-cell", { content: content[0], table });
        });
    });
});
app.get("/admin/alumni-cell", (req, res) => {
    db.query("SELECT * FROM alumni_cell_content LIMIT 1", (err, content) => {
        db.query("SELECT * FROM alumni_cell_table", (err, table) => {
            res.render("admin/alumni-cell", { content: content[0], table });
        });
    });
});
app.get("/admin/alumni-cell-content-edit/:id", (req, res) => {
    db.query("SELECT * FROM alumni_cell_content WHERE id=?",
        [req.params.id],
        (err, data) => {
            res.render("admin/alumni-cell-edit", { data: data[0], type: "content" });
        });
});
app.get("/admin/alumni-cell-edit/:id", (req, res) => {
    db.query("SELECT * FROM alumni_cell_table WHERE id=?",
        [req.params.id],
        (err, data) => {
            res.render("admin/alumni-cell-edit", { data: data[0], type: "table" });
        });
});
app.get("/admin/alumni-cell-add", (req, res) => {
    res.render("admin/alumni-cell-edit", { data: {}, type: "table" });
});
app.post("/admin/alumni-cell-update",
    upload.single("image"),
    upload.single("link"),
    (req, res) => {

        const { id, type } = req.body;

        if (type === "content") {

            let img = req.body.oldImage;
            if (req.file) img = req.file.filename;

            db.query(
                `UPDATE alumni_cell_content 
                 SET title=?, image=?, content=? WHERE id=?`,
                [req.body.title, img, req.body.content, id],
                () => res.redirect("/admin/alumni-cell")
            );

        } else {

            let pdf = req.body.oldPdf;
            if (req.file) pdf = req.file.filename;

            db.query(
                `UPDATE alumni_cell_table 
                 SET details1=?, details2=?, link=? WHERE id=?`,
                [req.body.details1, req.body.details2, pdf, id],
                () => res.redirect("/admin/alumni-cell")
            );
        }
});
app.post("/admin/alumni-cell-delete", (req, res) => {
    db.query("DELETE FROM alumni_cell_table WHERE id=?", [req.body.id], () => {
        res.redirect("/admin/alumni-cell");
    });
});

app.get("/anti-ragging-cell", (req, res) => {
    db.query("SELECT * FROM anti_ragging_cell_content LIMIT 1", (err, content) => {
        db.query("SELECT * FROM anti_ragging_cell_table", (err, table) => {
            res.render("anti-ragging-cell", { content: content[0], table });
        });
    });
});
app.get("/admin/anti-ragging-cell", (req, res) => {
    db.query("SELECT * FROM anti_ragging_cell_content LIMIT 1", (err, content) => {
        db.query("SELECT * FROM anti_ragging_cell_table", (err, table) => {
            res.render("admin/anti-ragging-cell", { content: content[0], table });
        });
    });
});
app.get("/admin/anti-ragging-cell-content-edit/:id", (req, res) => {
    db.query("SELECT * FROM anti_ragging_cell_content WHERE id=?", [req.params.id], (err, data) => {
        res.render("admin/anti-ragging-cell-edit", { data: data[0], type: "content" });
    });
});
app.get("/admin/anti-ragging-cell-edit/:id", (req, res) => {
    db.query("SELECT * FROM anti_ragging_cell_table WHERE id=?", [req.params.id], (err, data) => {
        res.render("admin/anti-ragging-cell-edit", { data: data[0], type: "table" });
    });
});
app.get("/admin/anti-ragging-cell-add", (req, res) => {
    res.render("admin/anti-ragging-cell-edit", { data: {}, type: "table" });
});
app.post("/admin/anti-ragging-cell-update",
    upload.single("image"),
    upload.single("link"),
    (req, res) => {

        const { id, type } = req.body;

        if (type === "content") {

            let img = req.body.oldImage;
            if (req.file) img = req.file.filename;

            db.query(
                `UPDATE anti_ragging_cell_content 
                 SET title=?, image=?, content=? WHERE id=?`,
                [req.body.title, img, req.body.content, id],
                () => res.redirect("/admin/anti-ragging-cell")
            );

        } else {

            let pdf = req.body.oldPdf;
            if (req.file) pdf = req.file.filename;

            db.query(
                `UPDATE anti_ragging_cell_table 
                 SET details1=?, details2=?, link=? WHERE id=?`,
                [req.body.details1, req.body.details2, pdf, id],
                () => res.redirect("/admin/anti-ragging-cell")
            );
        }
});
app.post("/admin/anti-ragging-cell-delete", (req, res) => {
    db.query("DELETE FROM anti_ragging_cell_table WHERE id=?", [req.body.id], () => {
        res.redirect("/admin/anti-ragging-cell");
    });
});

app.get("/class-committee-meeting", (req, res) => {
    db.query("SELECT * FROM class_committee_meeting LIMIT 1", (err, data) => {
        res.render("class-committee-meeting", { data: data[0] });
    });
});
app.get("/admin/class-committee-meeting", (req, res) => {
    db.query("SELECT * FROM class_committee_meeting LIMIT 1", (err, data) => {
        res.render("admin/class-committee-meeting", { data: data[0] });
    });
});
app.get("/admin/class-committee-meeting-edit/:id", (req, res) => {
    db.query("SELECT * FROM class_committee_meeting WHERE id=?", [req.params.id], (err, data) => {
        res.render("admin/class-committee-meeting-edit", { data: data[0] });
    });
});
app.post("/admin/class-committee-meeting-update",
  upload.fields([{ name: "image1" }, { name: "image2" }]),
    (req, res) => {

        let img1 = req.body.oldImage1;
        let img2 = req.body.oldImage2;

        if (req.files.image1) img1 = req.files.image1[0].filename;
        if (req.files.image2) img2 = req.files.image2[0].filename;

        db.query(
            `UPDATE class_committee_meeting 
             SET title=?, image1=?, image2=?, content=? WHERE id=?`,
            [req.body.title, img1, img2, req.body.content, req.body.id],
            () => res.redirect("/admin/class-committee-meeting")
        );
});

app.get("/college-maintenance-committee", (req, res) => {
    db.query("SELECT * FROM college_maintenance_committee_content LIMIT 1", (err, content) => {
        db.query("SELECT * FROM college_maintenance_committee_table", (err, table) => {
            res.render("college-maintenance-committee", { content: content[0], table });
        });
    });
});
app.get("/admin/college-maintenance-committee", (req, res) => {
    db.query("SELECT * FROM college_maintenance_committee_content LIMIT 1", (err, content) => {
        db.query("SELECT * FROM college_maintenance_committee_table", (err, table) => {
            res.render("admin/college-maintenance-committee", { content: content[0], table });
        });
    });
});
app.get("/admin/college-maintenance-committee-content-edit/:id", (req, res) => {
    db.query(
        "SELECT * FROM college_maintenance_committee_content WHERE id=?",
        [req.params.id],
        (err, data) => res.render("admin/college-maintenance-committee-edit", { data: data[0], type: "content" })
    );
});
app.get("/admin/college-maintenance-committee-edit/:id", (req, res) => {
    db.query(
        "SELECT * FROM college_maintenance_committee_table WHERE id=?",
        [req.params.id],
        (err, data) => res.render("admin/college-maintenance-committee-edit", { data: data[0], type: "table" })
    );
});
app.get("/admin/college-maintenance-committee-add", (req, res) => {
    res.render("admin/college-maintenance-committee-edit", { data: {}, type: "table" });
});
app.post("/admin/college-maintenance-committee-update",
    upload.single("image"),
    upload.single("link"),
    (req, res) => {
        const { id, type } = req.body;

        if (type === "content") {
            let img = req.body.oldImage;
            if (req.file) img = req.file.filename;

            db.query(
                `UPDATE college_maintenance_committee_content 
                 SET title=?, image=?, content=? WHERE id=?`,
                [req.body.title, img, req.body.content, id],
                () => res.redirect("/admin/college-maintenance-committee")
            );
        } else {
            let pdf = req.body.oldPdf;
            if (req.file) pdf = req.file.filename;

            db.query(
                `UPDATE college_maintenance_committee_table 
                 SET details1=?, details2=?, link=? WHERE id=?`,
                [req.body.details1, req.body.details2, pdf, id],
                () => res.redirect("/admin/college-maintenance-committee")
            );
        }
    }
);
app.post("/admin/college-maintenance-committee-delete", (req, res) => {
    db.query("DELETE FROM college_maintenance_committee_table WHERE id=?", [req.body.id], () => {
        res.redirect("/admin/college-maintenance-committee");
    });
});

app.get("/complaints-cum-redressal", (req, res) => {
    db.query("SELECT * FROM complaints_cum_redressal", (err, list) => {
        res.render("complaints-cum-redressal", { list });
    });
});
app.get("/admin/complaints-cum-redressal", (req, res) => {
    db.query("SELECT * FROM complaints_cum_redressal", (err, list) => {
        res.render("admin/complaints-cum-redressal", { list });
    });
});
app.get("/admin/complaints-cum-redressal-add", (req, res) => {
    res.render("admin/complaints-cum-redressal-edit", { data: {} });
});
app.get("/admin/complaints-cum-redressal-edit/:id", (req, res) => {
    db.query("SELECT * FROM complaints_cum_redressal WHERE id=?", [req.params.id], (err, data) => {
        res.render("admin/complaints-cum-redressal-edit", { data: data[0] });
    });
});
app.post("/admin/complaints-cum-redressal-update", (req, res) => {
    db.query(
        `UPDATE complaints_cum_redressal 
         SET committee_member=?, designation=?, email=?, phone_no=? WHERE id=?`,
        [
            req.body.committee_member,
            req.body.designation,
            req.body.email,
            req.body.phone_no,
            req.body.id
        ],
        () => res.redirect("/admin/complaints-cum-redressal")
    );
});
app.post("/admin/complaints-cum-redressal-delete", (req, res) => {
    db.query("DELETE FROM complaints_cum_redressal WHERE id=?", [req.body.id], () => {
        res.redirect("/admin/complaints-cum-redressal");
    });
});

app.get("/department-advisory-committee", (req, res) => {
    db.query("SELECT * FROM department_advisory_committee LIMIT 1", (err, data) => {
        res.render("department-advisory-committee", { data: data[0] });
    });
});
app.get("/admin/department-advisory-committee", (req, res) => {
    db.query("SELECT * FROM department_advisory_committee LIMIT 1", (err, data) => {
        res.render("admin/department-advisory-committee", { data: data[0] });
    });
});
app.get("/admin/department-advisory-committee-edit/:id", (req, res) => {
    db.query("SELECT * FROM department_advisory_committee WHERE id=?", [req.params.id], (err, data) => {
        res.render("admin/department-advisory-committee-edit", { data: data[0] });
    });
});
app.post("/admin/department-advisory-committee-update",
    upload.single("image"),
    upload.single("pdf_link"),
    (req, res) => {
        let img = req.body.oldImage;
        let pdf = req.body.oldPdf;

        if (req.files?.image) img = req.files.image[0].filename;
        if (req.file) pdf = req.file.filename;

        db.query(
            `UPDATE department_advisory_committee 
             SET title=?, image=?, content=?, pdf_title=?, pdf_link=? WHERE id=?`,
            [
                req.body.title,
                img,
                req.body.content,
                req.body.pdf_title,
                pdf,
                req.body.id
            ],
            () => res.redirect("/admin/department-advisory-committee")
        );
    }
);

app.get("/discipline-and-welfare", (req, res) => {
    db.query("SELECT * FROM discipline_and_welfare", (err, list) => {
        res.render("discipline-and-welfare", { list });
    });
});
app.get("/admin/discipline-and-welfare", (req, res) => {
    db.query("SELECT * FROM discipline_and_welfare", (err, list) => {
        res.render("admin/discipline-and-welfare", { list });
    });
});
app.get("/admin/discipline-and-welfare-add", (req, res) => {
    res.render("admin/discipline-and-welfare-edit", { data: {} });
});
app.get("/admin/discipline-and-welfare-edit/:id", (req, res) => {
    db.query("SELECT * FROM discipline_and_welfare WHERE id=?", [req.params.id], (err, data) => {
        res.render("admin/discipline-and-welfare-edit", { data: data[0] });
    });
});
app.post("/admin/discipline-and-welfare-update", (req, res) => {
    db.query(
        `UPDATE discipline_and_welfare 
         SET committee_member=?, designation=?, email=?, phone_no=? WHERE id=?`,
        [
            req.body.committee_member,
            req.body.designation,
            req.body.email,
            req.body.phone_no,
            req.body.id
        ],
        () => res.redirect("/admin/discipline-and-welfare")
    );
});
app.post("/admin/discipline-and-welfare-delete", (req, res) => {
    db.query("DELETE FROM discipline_and_welfare WHERE id=?", [req.body.id], () => {
        res.redirect("/admin/discipline-and-welfare");
    });
});

app.get("/discipline-committee", (req, res) => {
    db.query("SELECT * FROM discipline_committee_content LIMIT 1", (err, content) => {
        db.query("SELECT * FROM discipline_committee_table", (err, table) => {
            res.render("discipline-committee", { content: content[0], table });
        });
    });
});
app.get("/admin/discipline-committee", (req, res) => {
    db.query("SELECT * FROM discipline_committee_content LIMIT 1", (err, content) => {
        db.query("SELECT * FROM discipline_committee_table", (err, table) => {
            res.render("admin/discipline-committee", { content: content[0], table });
        });
    });
});
app.get("/admin/discipline-committee-content-edit/:id", (req, res) => {
    db.query("SELECT * FROM discipline_committee_content WHERE id=?", [req.params.id], (err, data) => {
        res.render("admin/discipline-committee-edit", { data: data[0], type: "content" });
    });
});
app.get("/admin/discipline-committee-edit/:id", (req, res) => {
    db.query("SELECT * FROM discipline_committee_table WHERE id=?", [req.params.id], (err, data) => {
        res.render("admin/discipline-committee-edit", { data: data[0], type: "table" });
    });
});
app.get("/admin/discipline-committee-add", (req, res) => {
    res.render("admin/discipline-committee-edit", { data: {}, type: "table" });
});
app.post("/admin/discipline-committee-update",
    upload.single("image"),
    upload.single("link"),
    (req, res) => {
        const { id, type } = req.body;

        if (type === "content") {
            let img = req.body.oldImage;
            if (req.file) img = req.file.filename;

            db.query(
                `UPDATE discipline_committee_content 
                 SET title=?, image=?, content=? WHERE id=?`,
                [req.body.title, img, req.body.content, id],
                () => res.redirect("/admin/discipline-committee")
            );
        } else {
            let pdf = req.body.oldPdf;
            if (req.file) pdf = req.file.filename;

            db.query(
                `UPDATE discipline_committee_table 
                 SET details1=?, details2=?, link=? WHERE id=?`,
                [req.body.details1, req.body.details2, pdf, id],
                () => res.redirect("/admin/discipline-committee")
            );
        }
    }
);
app.post("/admin/discipline-committee-delete", (req, res) => {
    db.query("DELETE FROM discipline_committee_table WHERE id=?", [req.body.id], () => {
        res.redirect("/admin/discipline-committee");
    });
});

app.get("/start-up-and-innovation-cell", (req, res) => {
    db.query("SELECT * FROM start_up_and_innovation_cell_content LIMIT 1", (err, content) => {
        db.query("SELECT * FROM start_up_and_innovation_cell_table", (err, table) => {
            res.render("start-up-and-innovation-cell", { content: content[0], table });
        });
    });
});
app.get("/admin/start-up-and-innovation-cell", (req, res) => {
    db.query("SELECT * FROM start_up_and_innovation_cell_content LIMIT 1", (err, content) => {
        db.query("SELECT * FROM start_up_and_innovation_cell_table", (err, table) => {
            res.render("admin/start-up-and-innovation-cell", { content: content[0], table });
        });
    });
});
app.get("/admin/start-up-and-innovation-cell-content-edit/:id", (req, res) => {
    db.query("SELECT * FROM start_up_and_innovation_cell_content WHERE id=?", [req.params.id], (err, data) => {
        res.render("admin/start-up-and-innovation-cell-edit", { data: data[0], type: "content" });
    });
});
app.get("/admin/start-up-and-innovation-cell-edit/:id", (req, res) => {
    db.query("SELECT * FROM start_up_and_innovation_cell_table WHERE id=?", [req.params.id], (err, data) => {
        res.render("admin/start-up-and-innovation-cell-edit", { data: data[0], type: "table" });
    });
});
app.get("/admin/start-up-and-innovation-cell-add", (req, res) => {
    res.render("admin/start-up-and-innovation-cell-edit", { data: {}, type: "table" });
});
app.post("/admin/start-up-and-innovation-cell-update",
    upload.single("image"),
    upload.single("link"),
    (req, res) => {

        const { id, type } = req.body;

        if (type === "content") {
            let img = req.body.oldImage;
            if (req.file) img = req.file.filename;

            db.query(
                `UPDATE start_up_and_innovation_cell_content 
                 SET title=?, image=?, content=? WHERE id=?`,
                [req.body.title, img, req.body.content, id],
                () => res.redirect("/admin/start-up-and-innovation-cell")
            );
        } else {
            let pdf = req.body.oldPdf;
            if (req.file) pdf = req.file.filename;

            db.query(
                `UPDATE start_up_and_innovation_cell_table 
                 SET details1=?, details2=?, link=? WHERE id=?`,
                [req.body.details1, req.body.details2, pdf, id],
                () => res.redirect("/admin/start-up-and-innovation-cell")
            );
        }
    }
);
app.post("/admin/start-up-and-innovation-cell-delete", (req, res) => {
    db.query("DELETE FROM start_up_and_innovation_cell_table WHERE id=?", [req.body.id], () => {
        res.redirect("/admin/start-up-and-innovation-cell");
    });
});

app.get("/exam-cell", (req, res) => {
    db.query("SELECT * FROM exam_cell LIMIT 1", (err, data) => {
        res.render("exam-cell", { data: data[0] });
    });
});
app.get("/admin/exam-cell", (req, res) => {
    db.query("SELECT * FROM exam_cell LIMIT 1", (err, data) => {
        res.render("admin/exam-cell", { data: data[0] });
    });
});
app.get("/admin/exam-cell-edit/:id", (req, res) => {
    db.query("SELECT * FROM exam_cell WHERE id=?", [req.params.id], (err, data) => {
        res.render("admin/exam-cell-edit", { data: data[0] });
    });
});
app.post("/admin/exam-cell-update",
    upload.single("image"),
    upload.single("pdf_link"),
    (req, res) => {

        let img = req.body.oldImage;
        let pdf = req.body.oldPdf;

        if (req.files?.image) img = req.files.image[0].filename;
        if (req.file) pdf = req.file.filename;

        db.query(
            `UPDATE exam_cell 
             SET title=?, image=?, content=?, pdf_title=?, pdf_link=? WHERE id=?`,
            [
                req.body.title,
                img,
                req.body.content,
                req.body.pdf_title,
                pdf,
                req.body.id
            ],
            () => res.redirect("/admin/exam-cell")
        );
    }
);

app.get("/admission-committee", (req, res) => {
    db.query("SELECT * FROM admission_committee_details LIMIT 1", (err, details) => {
        db.query("SELECT * FROM admission_committee_list", (err, list) => {
            db.query("SELECT * FROM admission_committee_table", (err, table) => {
                res.render("admission-committee", {
                    details: details[0],
                    list,
                    table
                });
            });
        });
    });
});
app.get("/admin/admission-committee", (req, res) => {
    db.query("SELECT * FROM admission_committee_details LIMIT 1", (err, details) => {
        db.query("SELECT * FROM admission_committee_list", (err, list) => {
            db.query("SELECT * FROM admission_committee_table", (err, table) => {
                res.render("admin/admission-committee", {
                    details: details[0],
                    list,
                    table
                });
            });
        });
    });
});
app.get("/admin/admission-committee-details-edit/:id", (req, res) => {
    db.query("SELECT * FROM admission_committee_details WHERE id=?", [req.params.id], (err, data) => {
        res.render("admin/admission-committee-details-edit", { data: data[0] });
    });
});
app.post("/admin/admission-committee-details-update",
    upload.single("image"),
    (req, res) => {

        let img = req.body.oldImage;
        if (req.file) img = req.file.filename;

        db.query(
            `UPDATE admission_committee_details 
             SET title=?, image=?, content=? WHERE id=?`,
            [req.body.title, img, req.body.content, req.body.id],
            () => res.redirect("/admin/admission-committee")
        );
    }
);
app.get("/admin/admission-committee-list-add", (req, res) => {
    res.render("admin/admission-committee-list-edit", { data: {} });
});
app.get("/admin/admission-committee-list-edit/:id", (req, res) => {
    db.query("SELECT * FROM admission_committee_list WHERE id=?", [req.params.id], (err, data) => {
        res.render("admin/admission-committee-list-edit", { data: data[0] });
    });
});
app.post("/admin/admission-committee-list-update", (req, res) => {
    db.query(
        `UPDATE admission_committee_list SET title=?, content=? WHERE id=?`,
        [req.body.title, req.body.content, req.body.id],
        () => res.redirect("/admin/admission-committee")
    );
});
app.post("/admin/admission-committee-list-delete", (req, res) => {
    db.query("DELETE FROM admission_committee_list WHERE id=?", [req.body.id], () => {
        res.redirect("/admin/admission-committee");
    });
});
app.get("/admin/admission-committee-table-add", (req, res) => {
    res.render("admin/admission-committee-table-edit", { data: {} });
});
app.get("/admin/admission-committee-table-edit/:id", (req, res) => {
    db.query("SELECT * FROM admission_committee_table WHERE id=?", [req.params.id], (err, data) => {
        res.render("admin/admission-committee-table-edit", { data: data[0] });
    });
});
app.post("/admin/admission-committee-table-update", (req, res) => {
    db.query(
        `UPDATE admission_committee_table 
         SET name_designation=?, role=? WHERE id=?`,
        [req.body.name_designation, req.body.role, req.body.id],
        () => res.redirect("/admin/admission-committee")
    );
});
app.post("/admin/admission-committee-table-delete", (req, res) => {
    db.query("DELETE FROM admission_committee_table WHERE id=?", [req.body.id], () => {
        res.redirect("/admin/admission-committee");
    });
});

app.get("/excess-fee-inspection-committee", (req, res) => {
    db.query("SELECT * FROM excess_fee_inspection_committee LIMIT 1", (err, data) => {
        res.render("excess-fee-inspection-committee", { data: data[0] });
    });
});
app.get("/admin/excess-fee-inspection-committee", (req, res) => {
    db.query("SELECT * FROM excess_fee_inspection_committee LIMIT 1", (err, data) => {
        res.render("admin/excess-fee-inspection-committee", { data: data[0] });
    });
});
app.get("/admin/excess-fee-inspection-committee-edit/:id", (req, res) => {
    db.query("SELECT * FROM excess_fee_inspection_committee WHERE id=?", [req.params.id], (err, data) => {
        res.render("admin/excess-fee-inspection-committee-edit", { data: data[0] });
    });
});
app.post("/admin/excess-fee-inspection-committee-update",
    upload.single("pdf"),
    (req, res) => {

        let pdf = req.body.oldPdf;
        if (req.file) pdf = req.file.filename;

        db.query(
            `UPDATE excess_fee_inspection_committee 
             SET title=?, pdf=? WHERE id=?`,
            [req.body.title, pdf, req.body.id],
            () => res.redirect("/admin/excess-fee-inspection-committee")
        );
    }
);

app.get("/faculty-welfare-cell", (req, res) => {
    db.query("SELECT * FROM faculty_welfare_cell_content LIMIT 1", (err, data) => {
        res.render("faculty-welfare-cell", { data: data[0] });
    });
});
app.get("/admin/faculty-welfare-cell", (req, res) => {
    db.query("SELECT * FROM faculty_welfare_cell_content LIMIT 1", (err, data) => {
        res.render("admin/faculty-welfare-cell", { data: data[0] });
    });
});
app.get("/admin/faculty-welfare-cell-edit/:id", (req, res) => {
    db.query("SELECT * FROM faculty_welfare_cell_content WHERE id=?", [req.params.id], (err, data) => {
        res.render("admin/faculty-welfare-cell-edit", { data: data[0] });
    });
});
app.post("/admin/faculty-welfare-cell-update",
    upload.fields([{ name: "image" }, { name: "image2" }]),
    (req, res) => {

        let image = req.body.oldImage;
        let image2 = req.body.oldImage2;

        if (req.files.image) image = req.files.image[0].filename;
        if (req.files.image2) image2 = req.files.image2[0].filename;

        db.query(
            `UPDATE faculty_welfare_cell_content 
             SET title=?, image=?, content=?, image2=? WHERE id=?`,
            [
                req.body.title,
                image,
                req.body.content,
                image2,
                req.body.id
            ],
            () => res.redirect("/admin/faculty-welfare-cell")
        );
    }
);

app.get("/governing-council", (req, res) => {
    db.query("SELECT * FROM governing_council", (err, list) => {
        res.render("governing-council", { list });
    });
});
app.get("/admin/governing-council", (req, res) => {
    db.query("SELECT * FROM governing_council", (err, list) => {
        res.render("admin/governing-council", { list });
    });
});
app.get("/admin/governing-council-add", (req, res) => {
    res.render("admin/governing-council-edit", { data: {} });
});
app.get("/admin/governing-council-edit/:id", (req, res) => {
    db.query("SELECT * FROM governing_council WHERE id=?", [req.params.id], (err, data) => {
        res.render("admin/governing-council-edit", { data: data[0] });
    });
});
app.post("/admin/governing-council-update", (req, res) => {
    const { id, title, name_address, status, designation } = req.body;

    if (id) {
        db.query(
            `UPDATE governing_council 
             SET title=?, name_address=?, status=?, designation=? WHERE id=?`,
            [title, name_address, status, designation, id],
            () => res.redirect("/admin/governing-council")
        );
    } else {
        db.query(
            `INSERT INTO governing_council 
             (title, name_address, status, designation) VALUES (?, ?, ?, ?)`,
            [title, name_address, status, designation],
            () => res.redirect("/admin/governing-council")
        );
    }
});
app.post("/admin/governing-council-delete", (req, res) => {
    db.query("DELETE FROM governing_council WHERE id=?", [req.body.id], () => {
        res.redirect("/admin/governing-council");
    });
});

app.get("/internal-complaint-committee", (req, res) => {
    db.query("SELECT * FROM internal_complaint_committee_content LIMIT 1", (err, content) => {
        db.query("SELECT * FROM internal_complaint_committee_table", (err, table) => {
            res.render("internal-complaint-committee", { content: content[0], table });
        });
    });
});
app.get("/admin/internal-complaint-committee", (req, res) => {
    db.query("SELECT * FROM internal_complaint_committee_content LIMIT 1", (err, content) => {
        db.query("SELECT * FROM internal_complaint_committee_table", (err, table) => {
            res.render("admin/internal-complaint-committee", { content: content[0], table });
        });
    });
});
app.get("/admin/internal-complaint-committee-content-edit/:id", (req, res) => {
    db.query("SELECT * FROM internal_complaint_committee_content WHERE id=?", [req.params.id], (err, data) => {
        res.render("admin/internal-complaint-committee-content-edit", { data: data[0] });
    });
});
app.post("/admin/internal-complaint-committee-content-update",
    upload.single("image"),
    (req, res) => {
        let image = req.body.oldImage;
        if (req.file) image = req.file.filename;

        db.query(
            `UPDATE internal_complaint_committee_content 
             SET title=?, image=?, content=? WHERE id=?`,
            [req.body.title, image, req.body.content, req.body.id],
            () => res.redirect("/admin/internal-complaint-committee")
        );
    }
);
app.get("/admin/internal-complaint-committee-add", (req, res) => {
    res.render("admin/internal-complaint-committee-edit", { data: {} });
});
app.get("/admin/internal-complaint-committee-edit/:id", (req, res) => {
    db.query("SELECT * FROM internal_complaint_committee_table WHERE id=?", [req.params.id], (err, data) => {
        res.render("admin/internal-complaint-committee-edit", { data: data[0] });
    });
});
app.post("/admin/internal-complaint-committee-update", (req, res) => {
    const { id, sec_title, committee_member, profession, associated_with, designation } = req.body;

    if (id) {
        db.query(
            `UPDATE internal_complaint_committee_table 
             SET sec_title=?, committee_member=?, profession=?, associated_with=?, designation=? 
             WHERE id=?`,
            [sec_title, committee_member, profession, associated_with, designation, id],
            () => res.redirect("/admin/internal-complaint-committee")
        );
    } else {
        db.query(
            `INSERT INTO internal_complaint_committee_table 
             (sec_title, committee_member, profession, associated_with, designation)
             VALUES (?, ?, ?, ?, ?)`,
            [sec_title, committee_member, profession, associated_with, designation],
            () => res.redirect("/admin/internal-complaint-committee")
        );
    }
});
app.post("/admin/internal-complaint-committee-delete", (req, res) => {
    db.query("DELETE FROM internal_complaint_committee_table WHERE id=?", [req.body.id], () => {
        res.redirect("/admin/internal-complaint-committee");
    });
});

app.get("/planning-and-monitoring", (req, res) => {
    db.query("SELECT * FROM planning_and_monitoring", (err, list) => {
        res.render("planning-and-monitoring", { list });
    });
});
app.get("/admin/planning-and-monitoring", (req, res) => {
    db.query("SELECT * FROM planning_and_monitoring", (err, list) => {
        res.render("admin/planning-and-monitoring", { list });
    });
});
app.get("/admin/planning-and-monitoring-add", (req, res) => {
    res.render("admin/planning-and-monitoring-edit", { data: {} });
});
app.get("/admin/planning-and-monitoring-edit/:id", (req, res) => {
    db.query("SELECT * FROM planning_and_monitoring WHERE id=?", [req.params.id], (err, data) => {
        res.render("admin/planning-and-monitoring-edit", { data: data[0] });
    });
});
app.post("/admin/planning-and-monitoring-update", (req, res) => {
    const { id, committee_member, designation, email, phone_number } = req.body;

    if (id) {
        db.query(
            `UPDATE planning_and_monitoring 
             SET committee_member=?, designation=?, email=?, phone_number=? WHERE id=?`,
            [committee_member, designation, email, phone_number, id],
            () => res.redirect("/admin/planning-and-monitoring")
        );

    } else {
        db.query(
            `INSERT INTO planning_and_monitoring 
             (committee_member, designation, email, phone_number)
             VALUES (?, ?, ?, ?)`,
            [committee_member, designation, email, phone_number],
            () => res.redirect("/admin/planning-and-monitoring")
        );
    }
});
app.post("/admin/planning-and-monitoring-delete", (req, res) => {
    db.query("DELETE FROM planning_and_monitoring WHERE id=?", [req.body.id], () => {
        res.redirect("/admin/planning-and-monitoring");
    });
});

app.get('/aqar23-iqac', async (req, res) => {
  const [data] = await promisePool.execute(
    "SELECT * FROM aqar23_iqac LIMIT 1"
  );
  res.render('aqar23-iqac', { data });
});
app.get('/admin/aqar23-iqac', async (req, res) => {
  const [data] = await promisePool.execute(
    "SELECT * FROM aqar23_iqac"
  );
  res.render('admin/aqar23-iqac', { data });
});
app.get('/admin/aqar23-iqac/edit/:id', async (req, res) => {
  const [rows] = await promisePool.execute(
    "SELECT * FROM aqar23_iqac WHERE id = ?",
    [req.params.id]
  );
  res.render('admin/aqar23-iqac-edit', { row: rows[0] });
});
app.post('/admin/aqar23-iqac/update/:id', async (req, res) => {
  const { details_coordinator, objective, responsibilities } = req.body;

  await promisePool.execute(
    `UPDATE aqar23_iqac 
     SET details_coordinator = ?, objective = ?, responsibilities = ?
     WHERE id = ?`,
    [details_coordinator, objective, responsibilities, req.params.id]
  );

  res.redirect('/admin/aqar23-iqac');
});

app.get('/aqar23-institutional-distinctiveness', async (req,res)=>{
  const [data] = await promisePool.execute(
    "SELECT * FROM aqar23_institutional_distinctiveness"
  );
  res.render('aqar23-institutional-distinctiveness',{data});
});
app.get('/admin/aqar23-institutional-distinctiveness', async (req,res)=>{
  const [data] = await promisePool.execute(
    "SELECT * FROM aqar23_institutional_distinctiveness"
  );
  res.render('admin/aqar23-institutional-distinctiveness',{data});
});
app.get('/admin/aqar23-institutional-distinctiveness/edit/:id', async (req,res)=>{
  const [rows] = await promisePool.execute(
    "SELECT * FROM aqar23_institutional_distinctiveness WHERE id=?",
    [req.params.id]
  );
  res.render('admin/aqar23-institutional-distinctiveness-edit',{row:rows[0]});
});
app.post('/admin/aqar23-institutional-distinctiveness/update/:id', async (req,res)=>{
  const {title,content} = req.body;
  await promisePool.execute(
    "UPDATE aqar23_institutional_distinctiveness SET title=?, content=? WHERE id=?",
    [title,content,req.params.id]
  );
  res.redirect('/admin/aqar23-institutional-distinctiveness');
});

app.get('/aqar23-best-practices', async (req, res) => {
  try {
    const [data] = await promisePool.execute(
      "SELECT * FROM aqar23_best_practices ORDER BY id ASC"
    );

    res.render('aqar23-best-practices', { data });
  } catch (err) {
    console.error("AQAR23 frontend error:", err);
    res.status(500).send("Error loading page");
  }
});
app.get('/admin/aqar23-best-practices', async (req, res) => {
  const [data] = await promisePool.execute(
    "SELECT * FROM aqar23_best_practices ORDER BY id ASC"
  );
  res.render('admin/aqar23-best-practices', { data });
});
app.get('/admin/aqar23-best-practices/add', (req, res) => {
  res.render('admin/aqar23-best-practices-add');
});
app.post('/admin/aqar23-best-practices/add',
  upload.single('document_file'),
  async (req, res) => {

    const {
      practice_title,
      practice_name,
      description,
      document_title
    } = req.body;

    const document_file = req.file ? req.file.filename : null;

    await promisePool.execute(
      `INSERT INTO aqar23_best_practices
       (practice_title, practice_name, description, document_title, document_file)
       VALUES (?,?,?,?,?)`,
      [practice_title, practice_name, description, document_title, document_file]
    );

    res.redirect('/admin/aqar23-best-practices');
});
app.get('/admin/aqar23-best-practices/edit/:id', async (req, res) => {
  const [rows] = await promisePool.execute(
    "SELECT * FROM aqar23_best_practices WHERE id = ?",
    [req.params.id]
  );

  if (!rows.length) {
    return res.send('Record not found');
  }

  res.render('admin/aqar23-best-practices-edit', {
    row: rows[0]
  });
});
app.post('/admin/aqar23-best-practices/update/:id',
  upload.single('document_file'),
  async (req, res) => {

    const {
      practice_title,
      practice_name,
      description,
      document_title,
      old_document_file
    } = req.body;

    const finalFile = req.file ? req.file.filename : old_document_file;

    // delete old pdf if replaced
    if (req.file && old_document_file) {
      const oldPath = `public/pdf/${old_document_file}`;
      if (fs.existsSync(oldPath)) {
        fs.unlinkSync(oldPath);
      }
    }

    await promisePool.execute(
      `UPDATE aqar23_best_practices
       SET practice_title=?, practice_name=?, description=?, document_title=?, document_file=?
       WHERE id=?`,
      [practice_title, practice_name, description, document_title, finalFile, req.params.id]
    );

    res.redirect('/admin/aqar23-best-practices');
});
app.get('/admin/aqar23-best-practices/delete/:id', async (req, res) => {
  const [rows] = await promisePool.execute(
    "SELECT document_file FROM aqar23_best_practices WHERE id=?",
    [req.params.id]
  );

  if (rows.length && rows[0].document_file) {
    const filePath = `public/pdf/${rows[0].document_file}`;
    if (fs.existsSync(filePath)) {
      fs.unlinkSync(filePath);
    }
  }

  await promisePool.execute(
    "DELETE FROM aqar23_best_practices WHERE id=?",
    [req.params.id]
  );

  res.redirect('/admin/aqar23-best-practices');
});

app.use(async (req, res, next) => {
  try {
    const [sp] = await promisePool.execute(
      "SELECT pdf_file FROM aqar23_strategic_plan LIMIT 1"
    );
    res.locals.aqar23StrategicPlan = sp.length ? sp[0].pdf_file : null;

    const [ar] = await promisePool.execute(
      "SELECT pdf_file FROM aqar23_annual_report LIMIT 1"
    );
    res.locals.aqar23AnnualReport = ar.length ? ar[0].pdf_file : null;

    const [as] = await promisePool.execute(
      "SELECT pdf_file FROM aqar23_audited_statement LIMIT 1"
    );
    res.locals.aqar23AuditedStatement = as.length ? as[0].pdf_file : null;

  } catch (err) {
    res.locals.aqar23StrategicPlan = null;
    res.locals.aqar23AnnualReport = null;
    res.locals.aqar23AuditedStatement = null;
  }
  next();
});
app.get('/admin/aqar23-strategic-plan', async (req, res) => {
  const [rows] = await promisePool.execute(
    "SELECT * FROM aqar23_strategic_plan LIMIT 1"
  );
  res.render('admin/aqar23-strategic-plan', {
    plan: rows.length ? rows[0] : null
  });
});
app.post('/admin/aqar23-strategic-plan/add',
  uploadPDF.single('pdf'),
  async (req, res) => {

    const [rows] = await promisePool.execute(
      "SELECT id FROM aqar23_strategic_plan LIMIT 1"
    );

    if (rows.length > 0) {
      return res.redirect('/admin/aqar23-strategic-plan');
    }

    await promisePool.execute(
      "INSERT INTO aqar23_strategic_plan (pdf_file) VALUES (?)",
      [req.file.filename]
    );

    res.redirect('/admin/aqar23-strategic-plan');
  }
);
app.post('/admin/aqar23-strategic-plan/edit/:id',
  uploadPDF.single('pdf'),
  async (req, res) => {

    const { old_file } = req.body;

    // delete old pdf
    if (old_file) {
      const oldPath = path.join(__dirname, 'public/pdf', old_file);
      if (fs.existsSync(oldPath)) fs.unlinkSync(oldPath);
    }

    await promisePool.execute(
      "UPDATE aqar23_strategic_plan SET pdf_file=? WHERE id=?",
      [req.file.filename, req.params.id]
    );

    res.redirect('/admin/aqar23-strategic-plan');
  }
);

app.get('/admin/aqar23-annual-report', async (req, res) => {
  const [rows] = await promisePool.execute(
    "SELECT * FROM aqar23_annual_report LIMIT 1"
  );
  res.render('admin/aqar23-annual-report', {
    report: rows.length ? rows[0] : null
  });
});
app.post('/admin/aqar23-annual-report/add',
  uploadPDF.single('pdf'),
  async (req, res) => {

    const [rows] = await promisePool.execute(
      "SELECT id FROM aqar23_annual_report LIMIT 1"
    );
    if (rows.length > 0) return res.redirect('/admin/aqar23-annual-report');

    await promisePool.execute(
      "INSERT INTO aqar23_annual_report (pdf_file) VALUES (?)",
      [req.file.filename]
    );
    res.redirect('/admin/aqar23-annual-report');
  }
);
app.post('/admin/aqar23-annual-report/edit/:id',
  uploadPDF.single('pdf'),
  async (req, res) => {

    const oldPath = path.join(__dirname, 'public/pdf', req.body.old_file);
    if (fs.existsSync(oldPath)) fs.unlinkSync(oldPath);

    await promisePool.execute(
      "UPDATE aqar23_annual_report SET pdf_file=? WHERE id=?",
      [req.file.filename, req.params.id]
    );
    res.redirect('/admin/aqar23-annual-report');
  }
);

app.get('/admin/aqar23-audited-statement', async (req, res) => {
  const [rows] = await promisePool.execute(
    "SELECT * FROM aqar23_audited_statement LIMIT 1"
  );
  res.render('admin/aqar23-audited-statement', {
    statement: rows.length ? rows[0] : null
  });
});
app.post('/admin/aqar23-audited-statement/add',
  uploadPDF.single('pdf'),
  async (req, res) => {

    const [rows] = await promisePool.execute(
      "SELECT id FROM aqar23_audited_statement LIMIT 1"
    );
    if (rows.length > 0) return res.redirect('/admin/aqar23-audited-statement');

    await promisePool.execute(
      "INSERT INTO aqar23_audited_statement (pdf_file) VALUES (?)",
      [req.file.filename]
    );
    res.redirect('/admin/aqar23-audited-statement');
  }
);
app.post('/admin/aqar23-audited-statement/edit/:id',
  uploadPDF.single('pdf'),
  async (req, res) => {

    const oldPath = path.join(__dirname, 'public/pdf', req.body.old_file);
    if (fs.existsSync(oldPath)) fs.unlinkSync(oldPath);

    await promisePool.execute(
      "UPDATE aqar23_audited_statement SET pdf_file=? WHERE id=?",
      [req.file.filename, req.params.id]
    );
    res.redirect('/admin/aqar23-audited-statement');
  }
);

app.get('/aqar23-feedback', async (req, res) => {
  const [links] = await promisePool.execute(
    "SELECT * FROM aqar23_feedback_link ORDER BY id ASC"
  );
  const [pdf] = await promisePool.execute(
    "SELECT * FROM aqar23_feedback_pdf LIMIT 1"
  );

  res.render('aqar23-feedback', {
    feedbackLinks: links,
    feedbackPDF: pdf.length ? pdf[0] : null
  });
});
app.get('/admin/aqar23-feedback', async (req, res) => {
  const [links] = await promisePool.execute(
    "SELECT * FROM aqar23_feedback_link ORDER BY id ASC"
  );
  const [pdf] = await promisePool.execute(
    "SELECT * FROM aqar23_feedback_pdf LIMIT 1"
  );

  res.render('admin/aqar23-feedback', {
    links,
    pdf: pdf.length ? pdf[0] : null
  });
});
app.get('/admin/aqar23-feedback-edit/:id', async (req, res) => {
  const [rows] = await promisePool.execute(
    "SELECT * FROM aqar23_feedback_link WHERE id=?",
    [req.params.id]
  );
  res.render('admin/aqar23-feedback-edit', { data: rows[0] });
});
app.post('/admin/aqar23-feedback-link-update', async (req, res) => {
  const { id, feedback_title, feedback_title_url } = req.body;
  await promisePool.execute(
    "UPDATE aqar23_feedback_link SET feedback_title=?, feedback_title_url=? WHERE id=?",
    [feedback_title, feedback_title_url, id]
  );
  res.redirect('/admin/aqar23-feedback');
});
app.get('/admin/aqar23-feedback-pdf-edit/:id', async (req, res) => {
  const [rows] = await promisePool.execute(
    "SELECT * FROM aqar23_feedback_pdf WHERE id=?",
    [req.params.id]
  );
  res.render('admin/aqar23-feedback-pdf-edit', { data: rows[0] });
});
app.post('/admin/aqar23-feedback-pdf-update',
  uploadPDF.single('pdf_file'),
  async (req, res) => {

    const { id, pdf_title, oldFile } = req.body;
    const pdf = req.file ? req.file.filename : oldFile;

    await promisePool.execute(
      "UPDATE aqar23_feedback_pdf SET pdf_title=?, pdf_file=? WHERE id=?",
      [pdf_title, pdf, id]
    );

    res.redirect('/admin/aqar23-feedback');
  }
);

app.get('/admin/qif-aqar23/add', (req, res) => {
  res.render('admin/qif-aqar23-add');
});
app.post('/admin/qif-aqar23/add', async (req, res) => {
  const { criterion, name, page_link } = req.body;

  await promisePool.execute(
    "INSERT INTO qif_aqar23 (criterion, name, page_link) VALUES (?,?,?)",
    [criterion, name, page_link]
  );

  res.redirect('/admin/qif-aqar23');
});
app.get('/qif-aqar23', async (req, res) => {
  const [data] = await promisePool.execute(
    "SELECT * FROM qif_aqar23 ORDER BY id ASC"
  );
  res.render('qif-aqar23', { data });
});
app.get('/qif-aqar23-criteria/:id', async (req, res) => {
  const criteriaId = req.params.id;

  const [data] = await promisePool.execute(
    "SELECT * FROM qif_aqar23_criteria WHERE id = ?",
    [criteriaId]
  );

  res.render('qif-aqar23-criteria', { data });
});
app.get('/admin/qif-aqar23', async (req, res) => {
  const [data] = await promisePool.execute("SELECT * FROM qif_aqar23");
  res.render('admin/qif-aqar23', { data });
});
app.post('/admin/qif-aqar23/add', async (req, res) => {
  const { criterion, name, page_link } = req.body;

  await promisePool.execute(
    "INSERT INTO qif_aqar23 (criterion, name, page_link) VALUES (?,?,?)",
    [criterion, name, page_link]
  );

  res.redirect('/admin/qif-aqar23');
});
app.get('/admin/qif-aqar23/edit/:id', async (req, res) => {
  const [row] = await promisePool.execute(
    "SELECT * FROM qif_aqar23 WHERE id = ?",
    [req.params.id]
  );

  res.render('admin/qif-aqar23-edit', { row: row[0] });
});
app.post('/admin/qif-aqar23/edit/:id', async (req, res) => {
  const { criterion, name, page_link } = req.body;

  await promisePool.execute(
    "UPDATE qif_aqar23 SET criterion=?, name=?, page_link=? WHERE id=?",
    [criterion, name, page_link, req.params.id]
  );

  res.redirect('/admin/qif-aqar23');
});
app.get('/admin/qif-aqar23/delete/:id', async (req, res) => {
  await promisePool.execute(
    "DELETE FROM qif_aqar23 WHERE id = ?",
    [req.params.id]
  );

  res.redirect('/admin/qif-aqar23');
});
app.get('/admin/qif-aqar23-criteria', async (req, res) => {
  const [data] = await promisePool.execute(
    "SELECT * FROM qif_aqar23_criteria"
  );
  res.render('admin/qif-aqar23-criteria', { data });
});
app.post('/admin/qif-aqar23-criteria/add',
  uploadPDF.single('download_link'),
  async (req, res) => {

    const { criteria_number, criteria_name } = req.body;
    const pdfPath = '/pdf/' + req.file.filename;

    await promisePool.execute(
      "INSERT INTO qif_aqar23_criteria (criteria_number, criteria_name, download_link) VALUES (?,?,?)",
      [criteria_number, criteria_name, pdfPath]
    );

    res.redirect('/admin/qif-aqar23-criteria');
  }
);
app.get('/admin/qif-aqar23-criteria/edit/:id', async (req, res) => {
  const [row] = await promisePool.execute(
    "SELECT * FROM qif_aqar23_criteria WHERE id = ?",
    [req.params.id]
  );

  res.render('admin/qif-aqar23-criteria-edit', { row: row[0] });
});
app.post('/admin/qif-aqar23-criteria/edit/:id',
  uploadPDF.single('download_link'),
  async (req, res) => {

    const { criteria_number, criteria_name } = req.body;

    if (req.file) {
      const pdfPath = '/pdf/' + req.file.filename;

      await promisePool.execute(
        "UPDATE qif_aqar23_criteria SET criteria_number=?, criteria_name=?, download_link=? WHERE id=?",
        [criteria_number, criteria_name, pdfPath, req.params.id]
      );
    } else {
      await promisePool.execute(
        "UPDATE qif_aqar23_criteria SET criteria_number=?, criteria_name=? WHERE id=?",
        [criteria_number, criteria_name, req.params.id]
      );
    }

    res.redirect('/admin/qif-aqar23-criteria');
  }
);
app.get('/admin/qif-aqar23-criteria/delete/:id', async (req, res) => {
  await promisePool.execute(
    "DELETE FROM qif_aqar23_criteria WHERE id = ?",
    [req.params.id]
  );

  res.redirect('/admin/qif-aqar23-criteria');
});
app.get('/admin/qif-aqar23-criteria/add', (req, res) => {
  res.render('admin/qif-aqar23-criteria-add');
});
app.post('/admin/qif-aqar23-criteria/add',
  uploadPDF.single('download_link'),
  async (req, res) => {

    const { criteria_number, criteria_name } = req.body;
    const pdfPath = '/pdf/' + req.file.filename;

    await promisePool.execute(
      "INSERT INTO qif_aqar23_criteria (criteria_number, criteria_name, download_link) VALUES (?,?,?)",
      [criteria_number, criteria_name, pdfPath]
    );

    res.redirect('/admin/qif-aqar23-criteria');
  }
);
app.use(express.static('public'));

app.get('/training', async (req, res) => {
  try {
    const [visionRows] = await promisePool.execute('SELECT * FROM training_vision LIMIT 1');
    const [missionRows] = await promisePool.execute('SELECT * FROM training_mission LIMIT 1');
    const [objectivesRows] = await promisePool.execute('SELECT * FROM training_objectives LIMIT 1');
    const [policyRows] = await promisePool.execute('SELECT * FROM training_placement_policy LIMIT 1');
    const [processRows] = await promisePool.execute('SELECT * FROM training_process_chart LIMIT 1');
    const [coordRows] = await promisePool.execute('SELECT * FROM training_coordinators ORDER BY id ASC');

    const vision = visionRows[0] || {};
    const mission = missionRows[0] || {};
    const objectives = objectivesRows[0] || {};
    const placementPolicy = policyRows[0] || {};
    const processChart = processRows[0] || {};
    const coordinators = coordRows || [];

    res.render('training', {
      vision,
      mission,
      objectives,
      placementPolicy,
      processChart,
      coordinators
    });
  } catch (err) {
    console.error('Error loading training page:', err);
    res.status(500).send('Server Error');
  }
});
app.get('/admin/training', async (req, res) => {
  try {
    const [visionRows] = await promisePool.execute('SELECT * FROM training_vision LIMIT 1');
    const [missionRows] = await promisePool.execute('SELECT * FROM training_mission LIMIT 1');
    const [objectivesRows] = await promisePool.execute('SELECT * FROM training_objectives LIMIT 1');
    const [policyRows] = await promisePool.execute('SELECT * FROM training_placement_policy LIMIT 1');
    const [processRows] = await promisePool.execute('SELECT * FROM training_process_chart LIMIT 1');
    const [coordRows] = await promisePool.execute('SELECT * FROM training_coordinators ORDER BY id ASC');

    res.render('admin/training', {
      vision: visionRows[0] || {},
      mission: missionRows[0] || {},
      objectives: objectivesRows[0] || {},
      placementPolicy: policyRows[0] || {},
      processChart: processRows[0] || {},
      coordinators: coordRows || []
    });
  } catch (err) {
    console.error('Error loading admin training page:', err);
    res.status(500).send('Server Error');
  }
});
app.get('/admin/training-edit', async (req, res) => {
  try {
    const [visionRows] = await promisePool.execute('SELECT * FROM training_vision LIMIT 1');
    const [missionRows] = await promisePool.execute('SELECT * FROM training_mission LIMIT 1');
    const [objectivesRows] = await promisePool.execute('SELECT * FROM training_objectives LIMIT 1');
    const [policyRows] = await promisePool.execute('SELECT * FROM training_placement_policy LIMIT 1');
    const [processRows] = await promisePool.execute('SELECT * FROM training_process_chart LIMIT 1');
    const [coordRows] = await promisePool.execute('SELECT * FROM training_coordinators ORDER BY id ASC');

    res.render('admin/training-edit', {
      vision: visionRows[0] || {},
      mission: missionRows[0] || {},
      objectives: objectivesRows[0] || {},
      placementPolicy: policyRows[0] || {},
      processChart: processRows[0] || {},
      coordinators: coordRows || []
    });
  } catch (err) {
    console.error('Error loading training edit page:', err);
    res.status(500).send('Server Error');
  }
});
app.post('/admin/training-update', async (req, res) => {
  try {
    const {
      vision_id,
      vision_content,
      mission_id,
      mission_content,
      objectives_id,
      objectives_content,
      policy_id,
      policy_content,
      process_id,
      process_content
    } = req.body;

    await promisePool.execute(
      'UPDATE training_vision SET content = ? WHERE id = ?',
      [vision_content || '', vision_id || 1]
    );

    await promisePool.execute(
      'UPDATE training_mission SET content = ? WHERE id = ?',
      [mission_content || '', mission_id || 1]
    );

    await promisePool.execute(
      'UPDATE training_objectives SET content = ? WHERE id = ?',
      [objectives_content || '', objectives_id || 1]
    );

    await promisePool.execute(
      'UPDATE training_placement_policy SET content = ? WHERE id = ?',
      [policy_content || '', policy_id || 1]
    );

    await promisePool.execute(
      'UPDATE training_process_chart SET content = ? WHERE id = ?',
      [process_content || '', process_id || 1]
    );

    res.redirect('/admin/training');
  } catch (err) {
    console.error('Error updating training page:', err);
    res.status(500).send('Server Error');
  }
});

// Training Coordinators Management
app.get('/admin/training-coordinator-add', (req, res) => {
  res.render('admin/training-coordinator-edit', {
    coordinator: {},
    isEdit: false
  });
});

app.get('/admin/training-coordinator-edit/:id', async (req, res) => {
  try {
    const [rows] = await promisePool.execute('SELECT * FROM training_coordinators WHERE id = ?', [req.params.id]);
    res.render('admin/training-coordinator-edit', {
      coordinator: rows[0] || {},
      isEdit: true
    });
  } catch (err) {
    console.error('Error loading coordinator:', err);
    res.status(500).send('Server Error');
  }
});

app.post('/admin/training-coordinator-add', async (req, res) => {
  try {
    const { name, department } = req.body;
    await promisePool.execute(
      'INSERT INTO training_coordinators (name, department) VALUES (?, ?)',
      [name, department]
    );
    res.redirect('/admin/training-edit');
  } catch (err) {
    console.error('Error adding coordinator:', err);
    res.status(500).send('Server Error');
  }
});

app.post('/admin/training-coordinator-update/:id', async (req, res) => {
  try {
    const { name, department } = req.body;
    await promisePool.execute(
      'UPDATE training_coordinators SET name = ?, department = ? WHERE id = ?',
      [name, department, req.params.id]
    );
    res.redirect('/admin/training-edit');
  } catch (err) {
    console.error('Error updating coordinator:', err);
    res.status(500).send('Server Error');
  }
});

app.post('/admin/training-coordinator-delete', async (req, res) => {
  try {
    await promisePool.execute('DELETE FROM training_coordinators WHERE id = ?', [req.body.id]);
    res.redirect('/admin/training-edit');
  } catch (err) {
    console.error('Error deleting coordinator:', err);
    res.status(500).send('Server Error');
  }
});

// FRONTEND RECORD
app.get("/record", (req, res) => {
  db.query(
    "SELECT * FROM placement_records ORDER BY id DESC",
    (err, records) => {
      if (err) return res.send("DB Error");
      res.render("record", { records });
    }
  );
});

/* ======================
   ADMIN RECORD LIST
====================== */
app.get("/admin/record", (req, res) => {
  db.query(
    "SELECT * FROM placement_records ORDER BY id DESC",
    (err, records) => {
      if (err) return res.send("DB Error");
      res.render("admin/record", { records });
    }
  );
});

/* ======================
   ADD FORM PAGE
====================== */
app.get("/admin/record/new", (req, res) => {
  res.render("admin/record-new");
});

/* ======================
   INSERT RECORD
====================== */
app.post(
  "/admin/record/add",
  upload.fields([
    { name: "stu_img" },
    { name: "stu_company_logo" }
  ]),
  (req, res) => {

    const { stu_name, stu_course, stu_company } = req.body;

    const stuImg = req.files.stu_img[0].filename;
    const compLogo = req.files.stu_company_logo[0].filename;

    const sql = `
      INSERT INTO placement_records
      (stu_img, stu_name, stu_course, stu_company, stu_company_logo)
      VALUES (?, ?, ?, ?, ?)
    `;

    db.query(
      sql,
      [stuImg, stu_name, stu_course, stu_company, compLogo],
      err => {
        if (err) return res.send("Insert Error");
        res.redirect("/admin/record");
      }
    );
  }
);

/* ======================
   EDIT PAGE
====================== */
app.get("/admin/record/edit/:id", (req, res) => {
  db.query(
    "SELECT * FROM placement_records WHERE id=?",
    [req.params.id],
    (err, rows) => {
      if (err || rows.length === 0)
        return res.send("Record not found");

      res.render("admin/record-edit", { record: rows[0] });
    }
  );
});

/* ======================
   UPDATE
====================== */
app.post(
  "/admin/record/update/:id",
  upload.fields([
    { name: "stu_img" },
    { name: "stu_company_logo" }
  ]),
  (req, res) => {

    const { stu_name, stu_course, stu_company } = req.body;

    let sql =
      "UPDATE placement_records SET stu_name=?, stu_course=?, stu_company=?";
    let params = [stu_name, stu_course, stu_company];

    if (req.files.stu_img) {
      sql += ", stu_img=?";
      params.push(req.files.stu_img[0].filename);
    }

    if (req.files.stu_company_logo) {
      sql += ", stu_company_logo=?";
      params.push(req.files.stu_company_logo[0].filename);
    }

    sql += " WHERE id=?";
    params.push(req.params.id);

    db.query(sql, params, err => {
      if (err) return res.send("Update error");
      res.redirect("/admin/record");
    });
  }
);

/* ======================
   DELETE
====================== */
app.get("/admin/record/delete/:id", (req, res) => {
  db.query(
    "DELETE FROM placement_records WHERE id=?",
    [req.params.id],
    err => {
      if (err) return res.send("Delete error");
      res.redirect("/admin/record");
    }
  );
});


app.get('/recruits', async (req, res) => {
  try {
    const [aboutData] = await promisePool.execute("SELECT * FROM recruits_about LIMIT 1");
    const [logos] = await promisePool.execute("SELECT * FROM recruits_logos ORDER BY id ASC");
    const [charts] = await promisePool.execute("SELECT * FROM recruits_charts ORDER BY id ASC");
    const [pageInfo] = await promisePool.execute("SELECT * FROM recruits_page_info LIMIT 1");

    res.render('recruits', {
      about: aboutData[0],
      logos,
      charts,
      pageInfo: pageInfo[0]
    });

  } catch (err) {
    console.log("Recruits Page Error:", err);
    res.status(500).send("Server Error");
  }
});
app.get('/admin/recruits', async (req, res) => {
  const [aboutData] = await promisePool.execute("SELECT * FROM recruits_about LIMIT 1");
  const [logos] = await promisePool.execute("SELECT * FROM recruits_logos ORDER BY id ASC");
  const [charts] = await promisePool.execute("SELECT * FROM recruits_charts ORDER BY id ASC");
  const [pageInfo] = await promisePool.execute("SELECT * FROM recruits_page_info LIMIT 1");

  res.render('admin/recruits', {
    about: aboutData[0],
    logos,
    charts,
    pageInfo: pageInfo[0]
  });
});
app.get('/admin/recruits-edit', async (req, res) => {
  const [aboutData] = await promisePool.execute("SELECT * FROM recruits_about LIMIT 1");
  const [logos] = await promisePool.execute("SELECT * FROM recruits_logos ORDER BY id ASC");
  const [charts] = await promisePool.execute("SELECT * FROM recruits_charts ORDER BY id ASC");
  const [pageInfo] = await promisePool.execute("SELECT * FROM recruits_page_info LIMIT 1");

  res.render('admin/recruits-edit', {
    about: aboutData[0],
    logos,
    charts,
    pageInfo: pageInfo[0]
  });
});
app.post("/admin/recruits-about-update",
  uploadRecruits.single("about_image"),
  async (req, res) => {

    const { id, old_image, description } = req.body;

    let newImage = old_image;
    if (req.file) {
      newImage = "uploads/recruits/" + req.file.filename;
    }

    await promisePool.execute(
      "UPDATE recruits_about SET image_path=?, description=? WHERE id=?",
      [newImage, description, id]
    );

    res.redirect("/admin/recruits-edit");
  }
);
app.post("/admin/recruits-pageinfo-update", async (req, res) => {
  const { title, subtitle } = req.body;

  await promisePool.execute(
    "UPDATE recruits_page_info SET title=?, subtitle=? WHERE id=1",
    [title, subtitle]
  );

  res.redirect("/admin/recruits-edit");
});
app.post("/admin/recruits-logo-add",
  uploadRecruits.single("logo_image"),
  async (req, res) => {
    if (!req.file) {
      return res.status(400).send("No file uploaded. Please select an image file.");
    }

    const logoPath = "uploads/recruits/" + req.file.filename;

    await promisePool.execute(
      "INSERT INTO recruits_logos (logo_path) VALUES (?)",
      [logoPath]
    );

    res.redirect("/admin/recruits-edit");
  }
);
app.post("/admin/recruits-logo-delete", async (req, res) => {
  const { id } = req.body;

  await promisePool.execute("DELETE FROM recruits_logos WHERE id = ?", [id]);

  res.redirect("/admin/recruits-edit");
});
app.post("/admin/recruits-charts-update", async (req, res) => {
  const { chart_id, department, offer_summary, company_placements } = req.body;

  for (let i = 0; i < chart_id.length; i++) {
    await promisePool.execute(
      "UPDATE recruits_charts SET department=?, offer_summary=?, company_placements=? WHERE id=?",
      [
        department[i],
        offer_summary[i],
        company_placements[i],
        chart_id[i]
      ]
    );
  }

  res.redirect("/admin/recruits-edit");
});

app.use('/images', express.static('public/images'));
app.use('/pdf', express.static('public/pdf'));
app.get('/clubs', async (req, res) => {
  const [gallery]    = await promisePool.execute("SELECT * FROM clubs_gallery");
  const [contents]   = await promisePool.execute("SELECT * FROM clubs_content");
  const [activities] = await promisePool.execute("SELECT * FROM clubs_activities");

  res.render('clubs', { gallery, contents, activities });
});
app.get('/clubs-gallery', async (req,res)=>{
  const [gallery] = await promisePool.execute("SELECT * FROM clubs_gallery");
  res.render('clubs-gallery',{gallery});
});
app.get('/admin/clubs-gallery', async (req,res)=>{
  const [gallery] = await promisePool.execute("SELECT * FROM clubs_gallery");
  res.render('admin/clubs-gallery',{gallery});
});
app.get('/admin/clubs-gallery/add', (req, res) => {
  res.render('admin/clubs-gallery-edit', { gallery: {} });
});
app.post('/admin/clubs-gallery/add', upload.single('image'), async (req, res) => {
  await promisePool.execute(
    "INSERT INTO clubs_gallery (image, content) VALUES (?, ?)",
    [req.file.filename, req.body.content]
  );
  res.redirect('/admin/clubs-gallery');
});
app.get('/admin/clubs-gallery/edit/:id', async (req,res)=>{
  const [[gallery]] = await promisePool.execute(
    "SELECT * FROM clubs_gallery WHERE id=?",[req.params.id]
  );
  res.render('admin/clubs-gallery-edit',{gallery});
});
app.post('/admin/clubs-gallery/update/:id', upload.single('image'), async (req,res)=>{
  const image = req.file ? req.file.filename : req.body.old_image;
  await promisePool.execute(
    "UPDATE clubs_gallery SET image=?, content=? WHERE id=?",
    [image, req.body.content, req.params.id]
  );
  res.redirect('/admin/clubs-gallery');
});
app.get('/admin/clubs-gallery/delete/:id', async (req,res)=>{
  await promisePool.execute("DELETE FROM clubs_gallery WHERE id=?", [req.params.id]);
  res.redirect('/admin/clubs-gallery');
});
app.get('/clubs-content', async (req,res)=>{
  const [contents] = await promisePool.execute("SELECT * FROM clubs_content");
  res.render('clubs-content',{contents});
});
app.get('/admin/clubs-content', async (req,res)=>{
  const [contents] = await promisePool.execute("SELECT * FROM clubs_content");
  res.render('admin/clubs-content',{contents});
});
app.get('/admin/clubs-content/add', (req, res) => {
  res.render('admin/clubs-content-edit', { content: {} });
});
app.post('/admin/clubs-content/add', async (req, res) => {
  const { club_name, no_events, document_link } = req.body;

  await promisePool.execute(
    "INSERT INTO clubs_content (club_name, no_events, document_link) VALUES (?, ?, ?)",
    [club_name, no_events, document_link]
  );

  res.redirect('/admin/clubs-content');
});
app.get('/admin/clubs-content/edit/:id', async (req,res)=>{
  const [[content]] = await promisePool.execute(
    "SELECT * FROM clubs_content WHERE id=?",[req.params.id]
  );
  res.render('admin/clubs-content-edit',{content});
});
app.post('/admin/clubs-content/update/:id', async (req,res)=>{
  const { club_name,no_events,document_link } = req.body;
  await promisePool.execute(
    "UPDATE clubs_content SET club_name=?, no_events=?, document_link=? WHERE id=?",
    [club_name,no_events,document_link,req.params.id]
  );
  res.redirect('/admin/clubs-content');
});
app.get('/admin/clubs-content/delete/:id', async (req,res)=>{
  await promisePool.execute("DELETE FROM clubs_content WHERE id=?", [req.params.id]);
  res.redirect('/admin/clubs-content');
});
app.get('/clubs-activities', async (req,res)=>{
  const [activities] = await promisePool.execute("SELECT * FROM clubs_activities");
  res.render('clubs-activities',{activities});
});
app.get('/admin/clubs-activities', async (req,res)=>{
  const [activities] = await promisePool.execute("SELECT * FROM clubs_activities");
  res.render('admin/clubs-activities',{activities});
});
app.get('/admin/clubs-activities/add', (req, res) => {
  res.render('admin/clubs-activities-edit', { activity: {} });
});
app.post('/admin/clubs-activities/add', uploadPDF.single('pdf_file'), async (req, res) => {
  const { title, ticket, content, pdf_name } = req.body;

  await promisePool.execute(
    "INSERT INTO clubs_activities (title, ticket, content, pdf_name, pdf_file) VALUES (?, ?, ?, ?, ?)",
    [title, ticket, content, pdf_name, req.file.filename]
  );

  res.redirect('/admin/clubs-activities');
});
app.get('/admin/clubs-activities/edit/:id', async (req,res)=>{
  const [[activity]] = await promisePool.execute(
    "SELECT * FROM clubs_activities WHERE id=?",[req.params.id]
  );
  res.render('admin/clubs-activities-edit',{activity});
});
app.post('/admin/clubs-activities/update/:id', uploadPDF.single('pdf_file'), async (req,res)=>{
  const pdf = req.file ? req.file.filename : req.body.old_pdf;
  const { title,ticket,content,pdf_name } = req.body;

  await promisePool.execute(
    "UPDATE clubs_activities SET title=?,ticket=?,content=?,pdf_name=?,pdf_file=? WHERE id=?",
    [title,ticket,content,pdf_name,pdf,req.params.id]
  );
  res.redirect('/admin/clubs-activities');
});
app.get('/admin/clubs-activities/delete/:id', async (req,res)=>{
  await promisePool.execute("DELETE FROM clubs_activities WHERE id=?", [req.params.id]);
  res.redirect('/admin/clubs-activities');
});

app.get('/digital-payment', async (req, res) => {
  try {
    await promisePool.execute(`
      CREATE TABLE IF NOT EXISTS digital_payment (
        id INT AUTO_INCREMENT PRIMARY KEY,
        image_path VARCHAR(255),
        name VARCHAR(255),
        account_number VARCHAR(255),
        bank VARCHAR(255),
        branch VARCHAR(255),
        ifsc_code VARCHAR(255),
        micr_code VARCHAR(255),
        state VARCHAR(255),
        district VARCHAR(255),
        city VARCHAR(255),
        branch_code VARCHAR(255)
      )
    `);

    const [count] = await promisePool.execute(`SELECT COUNT(*) AS count FROM digital_payment`);
    if (count[0].count === 0) {
      await promisePool.query(`
        INSERT INTO digital_payment 
        (image_path, name, account_number, bank, branch, ifsc_code, micr_code, state, district, city, branch_code)
        VALUES (
          'images/TEC_f_improf.jpg',
          'Thamirabharani Engineering College',
          '025150360865001',
          'Tamilnad Mercantile Bank Ltd',
          'Tirunelveli',
          'TMBL0000025',
          '627060002',
          'Tamil Nadu',
          'Tirunelveli',
          'Tirunelveli',
          '000025 (Last 6 Characters of the IFSC Code)'
        )
      `);
    }

    const [rows] = await promisePool.execute(`SELECT * FROM digital_payment LIMIT 1`);
    const payment = rows[0];

    res.render('digital-payment', { payment });

  } catch (err) {
    console.error("❌ Digital Payment Page Error:", err);
    res.status(500).send("Internal Server Error: " + err.message);
  }
});
app.get("/admin/digital-payment", async (req, res) => {
  try {
    const [rows] = await promisePool.execute("SELECT * FROM digital_payment LIMIT 1");
    res.render("admin/digital-payment", { payment: rows[0] });
  } catch (err) {
    console.log("Digital Payment Admin Error:", err);
    res.status(500).send("Server Error");
  }
});
app.get("/admin/digital-payment-edit", async (req, res) => {
  try {
    const [rows] = await promisePool.execute("SELECT * FROM digital_payment LIMIT 1");
    res.render("admin/digital-payment-edit", { payment: rows[0] });
  } catch (err) {
    console.log("Digital Payment Edit Error:", err);
    res.status(500).send("Server Error");
  }
});
app.post("/admin/digital-payment-update",
  uploadDigitalPayment.single("image_file"),
  async (req, res) => {
    try {
      const {
        id,
        old_image,
        name,
        account_number,
        bank,
        branch,
        ifsc_code,
        micr_code,
        state,
        district,
        city,
        branch_code
      } = req.body;

      let imagePath = old_image;
      if (req.file) {
        imagePath = "uploads/digital-payment/" + req.file.filename;
      }

      await promisePool.execute(
        `UPDATE digital_payment 
         SET image_path=?, name=?, account_number=?, bank=?, branch=?, ifsc_code=?, micr_code=?, state=?, district=?, city=?, branch_code=?
         WHERE id=?`,
        [
          imagePath,
          name,
          account_number,
          bank,
          branch,
          ifsc_code,
          micr_code,
          state,
          district,
          city,
          branch_code,
          id
        ]
      );

      res.redirect("/admin/digital-payment");

    } catch (err) {
      console.log("Digital Payment Update Error:", err);
      res.status(500).send("Server Error");
    }
  }
);

app.get("/external-links", async (req, res) => {
  const [list] = await promisePool.execute("SELECT * FROM external_links");
  res.render("external-links", { list });
});
app.get("/admin/external-links", async (req, res) => {
  const [list] = await promisePool.execute("SELECT * FROM external_links");
  res.render("admin/external-links", { list });
});
app.get("/admin/external-links-add", (req, res) => {
  res.render("admin/external-links-edit", { data: {} });
});
app.get("/admin/external-links-edit/:id", async (req, res) => {
  const [rows] = await promisePool.execute("SELECT * FROM external_links WHERE id=?", [req.params.id]);
  res.render("admin/external-links-edit", { data: rows[0] });
});
app.post("/admin/external-links-update",
  upload.single("pdf_file"),
  async (req, res) => {
    let pdf = req.body.oldPdf;
    if (req.file) pdf = req.file.filename;

    if (req.body.id) {
      await promisePool.execute(
        `UPDATE external_links SET pdf_title=?, pdf_file=? WHERE id=?`,
        [req.body.pdf_title, pdf, req.body.id]
      );
    } else {
      await promisePool.execute(
        `INSERT INTO external_links (pdf_title, pdf_file) VALUES (?, ?)`,
        [req.body.pdf_title, pdf]
      );
    }

    res.redirect("/admin/external-links");
  }
);
app.post("/admin/external-links-delete", async (req, res) => {
  await promisePool.execute(`DELETE FROM external_links WHERE id=?`, [req.body.id]);
  res.redirect("/admin/external-links");
});

app.get("/staff-cricket-2023", async (req, res) => {
  const [rows] = await promisePool.execute("SELECT * FROM staff_cricket_2023 LIMIT 1");
  res.render("staff-cricket-2023", { data: rows[0] });
});
app.get("/admin/staff-cricket-2023", async (req, res) => {
  const [rows] = await promisePool.execute("SELECT * FROM staff_cricket_2023 LIMIT 1");
  res.render("admin/staff-cricket-2023", { data: rows[0] });
});
app.get("/admin/staff-cricket-2023-edit/:id", async (req, res) => {
  const [rows] = await promisePool.execute("SELECT * FROM staff_cricket_2023 WHERE id=?", [req.params.id]);
  res.render("admin/staff-cricket-2023-edit", { data: rows[0] });
});
app.post("/admin/staff-cricket-2023-update",
  upload.fields([
    { name: "image" },
    { name: "register_link" },
    { name: "cover_letter_link" }
  ]),
  async (req, res) => {
    let image = req.body.oldImage;
    let reg = req.body.oldRegister;
    let cover = req.body.oldCover;

    if (req.files.image) image = req.files.image[0].filename;
    if (req.files.register_link) reg = req.files.register_link[0].filename;
    if (req.files.cover_letter_link) cover = req.files.cover_letter_link[0].filename;

    await promisePool.execute(
      `UPDATE staff_cricket_2023 
       SET image=?, content=?, register_title=?, register_link=?, 
           cover_letter_title=?, cover_letter_link=?, rules=? 
       WHERE id=?`,
      [
        image,
        req.body.content,
        req.body.register_title,
        reg,
        req.body.cover_letter_title,
        cover,
        req.body.rules,
        req.body.id
      ]
    );

    res.redirect("/admin/staff-cricket-2023");
  }
);

app.get("/dcc-2023", async (req, res) => {
  const [rows] = await promisePool.execute("SELECT * FROM dcc_2023 LIMIT 1");
  res.render("dcc-2023", { data: rows[0] });
});
app.get("/admin/dcc-2023", async (req, res) => {
  const [rows] = await promisePool.execute("SELECT * FROM dcc_2023 LIMIT 1");
  res.render("admin/dcc-2023", { data: rows[0] });
});
app.get("/admin/dcc-2023-edit/:id", async (req, res) => {
  const [rows] = await promisePool.execute("SELECT * FROM dcc_2023 WHERE id=?", [req.params.id]);
  res.render("admin/dcc-2023-edit", { data: rows[0] });
});
app.post("/admin/dcc-2023-update",
  upload.fields([
    { name: "image" },
    { name: "register_link" },
    { name: "team_register_link" }
  ]),
  async (req, res) => {

    let image = req.body.oldImage;
    let reg = req.body.oldRegister;
    let team = req.body.oldTeamRegister;

    if (req.files.image) image = req.files.image[0].filename;
    if (req.files.register_link) reg = req.files.register_link[0].filename;
    if (req.files.team_register_link) team = req.files.team_register_link[0].filename;

    await promisePool.execute(
      `UPDATE dcc_2023 
       SET image=?, content=?, register_title=?, register_link=?, 
           team_register_title=?, team_register_link=?, rules=? 
       WHERE id=?`,
      [
        image,
        req.body.content,
        req.body.register_title,
        reg,
        req.body.team_register_title,
        team,
        req.body.rules,
        req.body.id
      ]
    );

    res.redirect("/admin/dcc-2023");
  }
);

app.get("/contact-us", async (req, res) => {
  try {
    const [rows] = await promisePool.execute("SELECT * FROM contact_info LIMIT 1");
    res.render("contact-us", { contact: rows[0] });
  } catch (err) {
    console.log("Contact Front Error:", err);
    res.status(500).send("Server error");
  }
});
app.get("/admin/contact-us", async (req, res) => {
  try {
    const [rows] = await promisePool.execute("SELECT * FROM contact_info LIMIT 1");
    res.render("admin/contact-us", { contact: rows[0] });
  } catch (err) {
    console.log("Admin Contact Error:", err);
    res.status(500).send("Server error");
  }
});
app.get("/admin/contact-us-edit", async (req, res) => {
  try {
    const [rows] = await promisePool.execute("SELECT * FROM contact_info LIMIT 1");
    res.render("admin/contact-us-edit", { contact: rows[0] });
  } catch (err) {
    console.log("Contact Edit Error:", err);
    res.status(500).send("Server error");
  }
});
app.post("/admin/contact-us-update", async (req, res) => {
  try {
    const {
      id,
      description,
      address,
      phone,
      fax,
      email,
      facebook,
      instagram,
      linkedin,
      youtube,
      twitter
    } = req.body;

    await promisePool.execute(
      `UPDATE contact_info 
       SET description=?, address=?, phone=?, fax=?, email=?, facebook=?, instagram=?, linkedin=?, youtube=?, twitter=? 
       WHERE id=?`,
      [
        description,
        address,
        phone,
        fax,
        email,
        facebook,
        instagram,
        linkedin,
        youtube,
        twitter,
        id
      ]
    );

    res.redirect("/admin/contact-us");

  } catch (err) {
    console.log("Contact Update Error:", err);
    res.status(500).send("Server error");
  }
});

app.get('/ug', async (req, res) => {
  const [rows] = await promisePool.execute("SELECT * FROM programs_ug ORDER BY id ASC");
  res.render('ug', { programs: rows });
});

app.get('/pg', async (req, res) => {
  const [rows] = await promisePool.execute("SELECT * FROM programs_pg ORDER BY id ASC");
  res.render('pg', { programs: rows });
});

app.get('/course/:slug', async (req, res) => {
  const slug = req.params.slug;
  try {

    const [ugData] = await promisePool.execute("SELECT * FROM programs_ug WHERE slug = ?", [slug]);
    if (ugData.length > 0) {
      return res.render('course-details', { course: ugData[0] });
    }

    const [pgData] = await promisePool.execute("SELECT * FROM programs_pg WHERE slug = ?", [slug]);
    if (pgData.length > 0) {
      return res.render('course-details', { course: pgData[0] });
    }

    res.status(404).send('Course not found');
  } catch (err) {
    console.error('Error loading course:', err);
    res.status(500).send('Internal Server Error');
  }
});
app.get("/admin/ug", async (req, res) => {
  const [rows] = await promisePool.execute("SELECT * FROM programs_ug ORDER BY id ASC");
  res.render("admin/ug", { programs: rows });
});
app.get("/admin/ug-add", (req, res) => {
  res.render("admin/ug-add");
});
const ugStorage = multer.diskStorage({
  destination: (req, file, cb) => {
    cb(null, "public/images/courses");
  },
  filename: (req, file, cb) => {
    cb(null, Date.now() + "-" + file.originalname);
  }
});
const uploadUG = multer({ storage: ugStorage });
app.post("/admin/ug-add", uploadUG.single("image_file"), async (req, res) => {

  const { degree, slug, duration, intake, description } = req.body;
  const image_file = req.file.filename;

  await promisePool.execute(
    `INSERT INTO programs_ug (degree, slug, image_file, duration, intake, description) 
     VALUES (?, ?, ?, ?, ?, ?)`,
    [degree, slug, image_file, duration, intake, description]
  );

  res.redirect("/admin/ug");
});
app.get("/admin/ug-edit/:id", async (req, res) => {
  const { id } = req.params;
  const [rows] = await promisePool.execute("SELECT * FROM programs_ug WHERE id=?", [id]);
  res.render("admin/ug-edit", { program: rows[0] });
});
app.post("/admin/ug-edit", uploadUG.single("image_file"), async (req, res) => {

  const { id, degree, slug, duration, intake, description, old_image } = req.body;

  let image_file = old_image;
  if (req.file) image_file = req.file.filename;

  await promisePool.execute(
    `UPDATE programs_ug 
     SET degree=?, slug=?, image_file=?, duration=?, intake=?, description=?
     WHERE id=?`,
    [degree, slug, image_file, duration, intake, description, id]
  );

  res.redirect("/admin/ug");
});
app.post("/admin/ug-delete", async (req, res) => {
  const { id } = req.body;

  await promisePool.execute("DELETE FROM programs_ug WHERE id=?", [id]);

  res.redirect("/admin/ug");
});
const pgStorage = multer.diskStorage({
  destination: (req, file, cb) => {
    cb(null, "public/images/courses");
  },
  filename: (req, file, cb) => {
    cb(null, Date.now() + "-" + file.originalname);
  }
});
const uploadPG = multer({ storage: pgStorage });
app.get("/admin/pg", async (req, res) => {
  const [rows] = await promisePool.execute("SELECT * FROM programs_pg ORDER BY id ASC");
  res.render("admin/pg", { programs: rows });
});
app.get("/admin/pg-add", (req, res) => {
  res.render("admin/pg-add");
});
app.post("/admin/pg-add", uploadPG.single("image_file"), async (req, res) => {

  const { degree, slug, duration, intake, description } = req.body;
  const image_file = req.file.filename;

  await promisePool.execute(
    `INSERT INTO programs_pg (degree, slug, image_file, duration, intake, description) 
     VALUES (?, ?, ?, ?, ?, ?)`,
    [degree, slug, image_file, duration, intake, description]
  );

  res.redirect("/admin/pg");
});
app.get("/admin/pg-edit/:id", async (req, res) => {
  const { id } = req.params;
  const [rows] = await promisePool.execute("SELECT * FROM programs_pg WHERE id=?", [id]);
  res.render("admin/pg-edit", { program: rows[0] });
});
app.post("/admin/pg-edit", uploadPG.single("image_file"), async (req, res) => {

  const { id, degree, slug, duration, intake, description, old_image } = req.body;

  let image_file = old_image;
  if (req.file) image_file = req.file.filename;

  await promisePool.execute(
    `UPDATE programs_pg 
     SET degree=?, slug=?, image_file=?, duration=?, intake=?, description=?
     WHERE id=?`,
    [degree, slug, image_file, duration, intake, description, id]
  );

  res.redirect("/admin/pg");
});
app.post("/admin/pg-delete", async (req, res) => {
  const { id } = req.body;

  await promisePool.execute("DELETE FROM programs_pg WHERE id=?", [id]);

  res.redirect("/admin/pg");
});

// ==================================================
// CIVIL ENGINEERING FRONTEND
// ==================================================

app.get("/civil-engineering", async (req, res) => {
  try {
    const [sections] = await promisePool.query(
      "SELECT * FROM civil_engineering"
    );

    const [gallery] = await promisePool.query(
      "SELECT * FROM civil_gallery"
    );

    const [faculty] = await promisePool.query(
      "SELECT * FROM civil_faculty_profile"
    );

    // Transform sections into arrays expected by the view
    const sectionArrays = {
      about: [],
      vision: [],
      mission: [],
      objectives: [], // Core Values
      outcomes: [],
      specific: [],
      highlights: [],
      placement: [],
      events: [],
      facult_achievements: [],
      student_achievements: [],
      newsletter: [],
      fees: []
    };

    sections.forEach(row => {
      const key = row.section_key;
      if (sectionArrays[key]) {
        sectionArrays[key].push(row);
      }
    });

    res.render("civil-engineering", {
      ...sectionArrays,
      civilGallery: gallery || [],
      faculty: faculty || []
    });

  } catch (err) {
    console.log(err);
    res.send("Civil page error");
  }
});
app.get("/admin/civil-engineering", async (req, res) => {
  try {
    const [data] = await promisePool.execute(
      "SELECT * FROM civil_engineering ORDER BY id ASC"
    );
    res.render("admin/civil-engineering", { data: data || [], query: req.query });
  } catch (err) {
    console.error("Error loading civil engineering:", err);
    res.render("admin/civil-engineering", { data: [], query: req.query });
  }
});
app.post("/admin/civil-engineering/update", async (req, res) => {
  try {
    const { section_key, title, content } = req.body;

    // Validate section_key is not empty
    if (!section_key || section_key.trim() === "") {
      return res.redirect("/admin/civil-engineering?error=section_key_required");
    }

    // Check if record exists
    const [existing] = await promisePool.execute(
      "SELECT id FROM civil_engineering WHERE section_key = ?",
      [section_key]
    );

    if (existing.length > 0) {
      // Update existing
      await promisePool.execute(
        "UPDATE civil_engineering SET title = ?, content = ? WHERE section_key = ?",
        [title, content, section_key]
      );
    } else {
      // Insert new - calculate next ID since id field doesn't have AUTO_INCREMENT
      const [maxId] = await promisePool.execute(
        "SELECT COALESCE(MAX(id), 0) + 1 as next_id FROM civil_engineering"
      );
      const nextId = maxId[0].next_id;

      await promisePool.execute(
        "INSERT INTO civil_engineering (id, section_key, title, content) VALUES (?, ?, ?, ?)",
        [nextId, section_key, title, content]
      );
    }

    res.redirect("/admin/civil-engineering");
  } catch (err) {
    console.error("Error updating civil engineering:", err);
    res.redirect("/admin/civil-engineering?error=update_failed");
  }
});
app.get("/admin/civil-engineering/edit/:section_key", async (req, res) => {
  try {
    const { section_key } = req.params;

    let data = null;
    if (section_key !== "new") {
      const [rows] = await promisePool.execute(
        "SELECT * FROM civil_engineering WHERE section_key = ?",
        [section_key]
      );
      data = rows[0] || null;
    }

    res.render("admin/civil-engineering-edit", { 
      data: data || { section_key: section_key === "new" ? "" : section_key },
      query: req.query
    });
  } catch (err) {
    console.error("Error loading civil engineering edit:", err);
    res.render("admin/civil-engineering-edit", { 
      data: { section_key: req.params.section_key === "new" ? "" : req.params.section_key },
      query: req.query
    });
  }
});

app.get("/admin/civil-gallery", async (req, res) => {
  try {
    const [gallery] = await promisePool.query(
      "SELECT * FROM civil_gallery ORDER BY id DESC"
    );

    res.render("admin/civil-gallery", { gallery });

  } catch (err) {
    console.log(err);
    res.send("Civil gallery error");
  }
});
app.post("/admin/civil-gallery/add", upload.single("image"), async (req, res) => {
  try {
    if (!req.file) {
      return res.redirect("/admin/civil-gallery");
    }

    // Files uploaded via `upload` are stored under `public/images/`
    // Store a web-path (no leading slash) to be used as `/<%= image_path %>` in EJS
    const imagePath = `images/${req.file.filename}`;

    await promisePool.execute(
      "INSERT INTO civil_gallery (image_path) VALUES (?)",
      [imagePath]
    );
    res.redirect("/admin/civil-engineering");
  } catch (err) {
    console.error("Civil gallery add error:", err);
    res.redirect("/admin/civil-gallery");
  }
});
app.get("/admin/civil-gallery/delete/:id", async (req, res) => {
  await promisePool.query(
    "DELETE FROM civil_gallery WHERE id=?",
    [req.params.id]
  );
  res.redirect("/admin/civil-engineering");
});
app.get("/admin/civil-faculty_profile", async (req, res) => {
  try {
    const [faculty] = await promisePool.query(
      "SELECT * FROM civil_faculty_profile ORDER BY id DESC"
    );

    res.render("admin/civil-faculty_profile", { faculty });

  } catch (err) {
    console.log(err);
    res.send("Faculty page error");
  }
});
app.get("/admin/civil-faculty/add", (req, res) => {
  res.render("admin/civil-faculty_profile-edit", {
    data: null,
    query: req.query
  });
});
app.get("/admin/civil-faculty/edit/:id", async (req, res) => {
  try {
    const [rows] = await promisePool.query(
      "SELECT * FROM civil_faculty_profile WHERE id=?",
      [req.params.id]
    );
    res.render("admin/civil-faculty_profile-edit", { data: rows[0] || null });
  } catch (err) {
    console.error("Error loading civil faculty:", err);
    res.redirect("/admin/civil-engineering");
  }
});
app.post("/admin/civil-faculty/add", upload.single("teacher_image"), async (req, res) => {
  try {
    const { teacher_name, teacher_department } = req.body;

    // Validate required fields
    if (!teacher_name || !teacher_department) {
      return res.redirect("/admin/civil-faculty/add?error=fields_required");
    }

    if (!req.file) {
      return res.redirect("/admin/civil-faculty/add?error=image_required");
    }

    // Get next ID since id field doesn't have AUTO_INCREMENT
    const [maxId] = await promisePool.execute(
      "SELECT COALESCE(MAX(id), 0) + 1 as next_id FROM civil_faculty_profile"
    );
    const nextId = maxId[0].next_id;

    await promisePool.execute(
      `INSERT INTO civil_faculty_profile
       (id, teacher_image, teacher_name, teacher_department)
       VALUES (?, ?, ?, ?)`,
      [nextId, req.file.filename, teacher_name, teacher_department]
    );

    res.redirect("/admin/civil-engineering");
  } catch (err) {
    console.error("Error adding faculty:", err);
    res.redirect("/admin/civil-faculty/add?error=add_failed");
  }
});
app.post("/admin/civil-faculty/update/:id", upload.single("teacher_image"), async (req, res) => {
  try {
    const { teacher_name, teacher_department, old_image } = req.body;
    const id = req.params.id;

    const filename = req.file ? req.file.filename : (old_image || null);

    await promisePool.query(
      `UPDATE civil_faculty_profile
       SET teacher_image=?, teacher_name=?, teacher_department=?
       WHERE id=?`,
      [filename, teacher_name, teacher_department, id]
    );

    res.redirect("/admin/civil-engineering");
  } catch (err) {
    console.error("Error updating civil faculty:", err);
    res.redirect("/admin/civil-engineering");
  }
});
app.get("/admin/civil-faculty/delete/:id", async (req, res) => {
  await promisePool.query(
    "DELETE FROM civil_faculty_profile WHERE id=?",
    [req.params.id]
  );
  res.redirect("/admin/civil-engineering");
});
 
// ==================================================
// MECHANICAL ENGINEERING (same logic & flow as civil)
// ==================================================

app.get("/mechanical-engineering", async (req, res) => {
  try {
    await promisePool.execute(`
      CREATE TABLE IF NOT EXISTS mechanical_engineering (
        id INT PRIMARY KEY,
        section_key VARCHAR(100) UNIQUE,
        title VARCHAR(255),
        content LONGTEXT
      )
    `);
    await promisePool.execute(`
      CREATE TABLE IF NOT EXISTS mechanical_gallery (
        id INT AUTO_INCREMENT PRIMARY KEY,
        image_path VARCHAR(500)
      )
    `);
    await promisePool.execute(`
      CREATE TABLE IF NOT EXISTS mechanical_faculty_profile (
        id INT PRIMARY KEY,
        teacher_image VARCHAR(255),
        teacher_name VARCHAR(255),
        teacher_department VARCHAR(255)
      )
    `);

    const [sections] = await promisePool.query(
      "SELECT * FROM mechanical_engineering"
    );

    const [gallery] = await promisePool.query(
      "SELECT * FROM mechanical_gallery"
    );

    const [faculty] = await promisePool.query(
      "SELECT * FROM mechanical_faculty_profile"
    );

    const sectionArrays = {
      about: [],
      vision: [],
      mission: [],
      objectives: [],
      outcomes: [],
      specific: [],
      highlights: [],
      placement: [],
      events: [],
      facult_achievements: [],
      student_achievements: [],
      newsletter: [],
      fees: []
    };

    sections.forEach(row => {
      const key = row.section_key;
      if (sectionArrays[key]) {
        sectionArrays[key].push(row);
      }
    });

    res.render("mechanical-engineering", {
      ...sectionArrays,
      mechanicalGallery: gallery || [],
      faculty: faculty || []
    });

  } catch (err) {
    console.error("Mechanical Frontend Error:", err);
    res.send("Mechanical Page Error");
  }
});

app.get("/admin/mechanical-engineering", async (req, res) => {
  try {
    const [data] = await promisePool.execute(
      "SELECT * FROM mechanical_engineering ORDER BY id ASC"
    );
    res.render("admin/mechanical-engineering", { data: data || [], query: req.query });
  } catch (err) {
    console.error("Error loading mechanical engineering:", err);
    res.render("admin/mechanical-engineering", { data: [], query: req.query });
  }
});

app.post("/admin/mechanical-engineering/update", async (req, res) => {
  try {
    const { section_key, title, content } = req.body;

    if (!section_key || section_key.trim() === "") {
      return res.redirect("/admin/mechanical-engineering?error=section_key_required");
    }

    const [existing] = await promisePool.execute(
      "SELECT id FROM mechanical_engineering WHERE section_key = ?",
      [section_key]
    );

    if (existing.length > 0) {
      await promisePool.execute(
        "UPDATE mechanical_engineering SET title = ?, content = ? WHERE section_key = ?",
        [title, content, section_key]
      );
    } else {
      const [maxId] = await promisePool.execute(
        "SELECT COALESCE(MAX(id), 0) + 1 as next_id FROM mechanical_engineering"
      );
      const nextId = maxId[0].next_id;

      await promisePool.execute(
        "INSERT INTO mechanical_engineering (id, section_key, title, content) VALUES (?, ?, ?, ?)",
        [nextId, section_key, title, content]
      );
    }

    res.redirect("/admin/mechanical-engineering");
  } catch (err) {
    console.error("Error updating mechanical engineering:", err);
    res.redirect("/admin/mechanical-engineering?error=update_failed");
  }
});

app.get("/admin/mechanical-engineering/edit/:section_key", async (req, res) => {
  try {
    const { section_key } = req.params;

    let data = null;
    if (section_key !== "new") {
      const [rows] = await promisePool.execute(
        "SELECT * FROM mechanical_engineering WHERE section_key = ?",
        [section_key]
      );
      data = rows[0] || null;
    }

    res.render("admin/mechanical-engineering-edit", {
      data: data || { section_key: section_key === "new" ? "" : section_key },
      query: req.query
    });
  } catch (err) {
    console.error("Error loading mechanical engineering edit:", err);
    res.render("admin/mechanical-engineering-edit", {
      data: { section_key: req.params.section_key === "new" ? "" : req.params.section_key },
      query: req.query
    });
  }
});

app.get("/admin/mechanical-gallery", async (req, res) => {
  try {
    const [gallery] = await promisePool.query(
      "SELECT * FROM mechanical_gallery ORDER BY id DESC"
    );
    res.render("admin/mechanical-gallery", { gallery });
  } catch (err) {
    console.error(err);
    res.send("Mechanical gallery error");
  }
});

app.post("/admin/mechanical-gallery/add", upload.single("image"), async (req, res) => {
  try {
    if (!req.file) {
      return res.redirect("/admin/mechanical-gallery");
    }
    const imagePath = `images/${req.file.filename}`;
    await promisePool.execute(
      "INSERT INTO mechanical_gallery (image_path) VALUES (?)",
      [imagePath]
    );
    res.redirect("/admin/mechanical-engineering");
  } catch (err) {
    console.error("Mechanical gallery add error:", err);
    res.redirect("/admin/mechanical-gallery");
  }
});

app.get("/admin/mechanical-gallery/delete/:id", async (req, res) => {
  await promisePool.query(
    "DELETE FROM mechanical_gallery WHERE id=?",
    [req.params.id]
  );
  res.redirect("/admin/mechanical-engineering");
});

app.get("/admin/mechanical-faculty_profile", async (req, res) => {
  try {
    const [faculty] = await promisePool.query(
      "SELECT * FROM mechanical_faculty_profile ORDER BY id DESC"
    );
    res.render("admin/mechanical-faculty_profile", { faculty });
  } catch (err) {
    console.error(err);
    res.send("Faculty page error");
  }
});

app.get("/admin/mechanical-faculty/add", (req, res) => {
  res.render("admin/mechanical-faculty_profile-edit", {
    data: null,
    query: req.query
  });
});

app.get("/admin/mechanical-faculty/edit/:id", async (req, res) => {
  try {
    const [rows] = await promisePool.query(
      "SELECT * FROM mechanical_faculty_profile WHERE id=?",
      [req.params.id]
    );
    res.render("admin/mechanical-faculty_profile-edit", { data: rows[0] || null });
  } catch (err) {
    console.error("Error loading mechanical faculty:", err);
    res.redirect("/admin/mechanical-engineering");
  }
});

app.post("/admin/mechanical-faculty/add", upload.single("teacher_image"), async (req, res) => {
  try {
    const { teacher_name, teacher_department } = req.body;

    if (!teacher_name || !teacher_department) {
      return res.redirect("/admin/mechanical-faculty/add?error=fields_required");
    }
    if (!req.file) {
      return res.redirect("/admin/mechanical-faculty/add?error=image_required");
    }

    const [maxId] = await promisePool.execute(
      "SELECT COALESCE(MAX(id), 0) + 1 as next_id FROM mechanical_faculty_profile"
    );
    const nextId = maxId[0].next_id;

    await promisePool.execute(
      `INSERT INTO mechanical_faculty_profile
       (id, teacher_image, teacher_name, teacher_department)
       VALUES (?, ?, ?, ?)`,
      [nextId, req.file.filename, teacher_name, teacher_department]
    );

    res.redirect("/admin/mechanical-engineering");
  } catch (err) {
    console.error("Error adding faculty:", err);
    res.redirect("/admin/mechanical-faculty/add?error=add_failed");
  }
});

app.post("/admin/mechanical-faculty/update/:id", upload.single("teacher_image"), async (req, res) => {
  try {
    const { teacher_name, teacher_department, old_image } = req.body;
    const id = req.params.id;
    const filename = req.file ? req.file.filename : (old_image || null);

    await promisePool.query(
      `UPDATE mechanical_faculty_profile
       SET teacher_image=?, teacher_name=?, teacher_department=?
       WHERE id=?`,
      [filename, teacher_name, teacher_department, id]
    );

    res.redirect("/admin/mechanical-engineering");
  } catch (err) {
    console.error("Error updating mechanical faculty:", err);
    res.redirect("/admin/mechanical-engineering");
  }
});

app.get("/admin/mechanical-faculty/delete/:id", async (req, res) => {
  await promisePool.query(
    "DELETE FROM mechanical_faculty_profile WHERE id=?",
    [req.params.id]
  );
  res.redirect("/admin/mechanical-engineering");
});

// ==================================================
// EANDC (E and C Engineering - same logic & flow as civil)
// ==================================================

app.get("/eandc", async (req, res) => {
  try {
    const [sections] = await promisePool.query("SELECT * FROM eandc");
    const [gallery] = await promisePool.query("SELECT * FROM eandc_gallery");
    const [faculty] = await promisePool.query("SELECT * FROM eandc_faculty_profile");

    const sectionArrays = {
      about: [], vision: [], mission: [], objectives: [], outcomes: [], specific: [],
      highlights: [], placement: [], events: [], facult_achievements: [], student_achievements: [],
      newsletter: [], fees: []
    };
    sections.forEach(row => {
      if (sectionArrays[row.section_key]) sectionArrays[row.section_key].push(row);
    });

    res.render("eandc", {
      ...sectionArrays,
      eandcGallery: gallery || [],
      faculty: faculty || []
    });
  } catch (err) {
    console.error("EANDC ERROR:", err);
    res.status(500).send(err.message);
  }
});

app.get("/admin/eandc", async (req, res) => {
  try {
    const [data] = await promisePool.execute("SELECT * FROM eandc ORDER BY id ASC");
    res.render("admin/eandc", { data: data || [], query: req.query });
  } catch (err) {
    console.error("Error loading EANDC:", err);
    res.render("admin/eandc", { data: [], query: req.query });
  }
});

app.post("/admin/eandc/update", async (req, res) => {
  try {
    const { section_key, title, content } = req.body;
    if (!section_key || section_key.trim() === "") {
      return res.redirect("/admin/eandc?error=section_key_required");
    }
    const [existing] = await promisePool.execute("SELECT id FROM eandc WHERE section_key = ?", [section_key]);
    if (existing.length > 0) {
      await promisePool.execute("UPDATE eandc SET title = ?, content = ? WHERE section_key = ?", [title, content, section_key]);
    } else {
      const [maxId] = await promisePool.execute("SELECT COALESCE(MAX(id), 0) + 1 as next_id FROM eandc");
      await promisePool.execute("INSERT INTO eandc (id, section_key, title, content) VALUES (?, ?, ?, ?)", [maxId[0].next_id, section_key, title, content]);
    }
    res.redirect("/admin/eandc");
  } catch (err) {
    console.error("Error updating EANDC:", err);
    res.redirect("/admin/eandc?error=update_failed");
  }
});

app.get("/admin/eandc/edit/:section_key", async (req, res) => {
  try {
    const { section_key } = req.params;
    let data = null;
    if (section_key !== "new") {
      const [rows] = await promisePool.execute("SELECT * FROM eandc WHERE section_key = ?", [section_key]);
      data = rows[0] || null;
    }
    res.render("admin/eandc-edit", {
      data: data || { section_key: section_key === "new" ? "" : section_key },
      query: req.query
    });
  } catch (err) {
    console.error("Error loading EANDC edit:", err);
    res.render("admin/eandc-edit", { data: { section_key: req.params.section_key === "new" ? "" : req.params.section_key }, query: req.query });
  }
});

app.get("/admin/eandc-gallery", async (req, res) => {
  try {
    const [gallery] = await promisePool.query("SELECT * FROM eandc_gallery ORDER BY id DESC");
    res.render("admin/eandc-gallery", { gallery });
  } catch (err) {
    console.error(err);
    res.send("EANDC gallery error");
  }
});

app.post("/admin/eandc-gallery/add", upload.single("image"), async (req, res) => {
  try {
    if (!req.file) return res.redirect("/admin/eandc-gallery");
    const imagePath = `images/${req.file.filename}`;
    await promisePool.execute("INSERT INTO eandc_gallery (image_path) VALUES (?)", [imagePath]);
    res.redirect("/admin/eandc");
  } catch (err) {
    console.error("EANDC gallery add error:", err);
    res.redirect("/admin/eandc-gallery");
  }
});

app.get("/admin/eandc-gallery/delete/:id", async (req, res) => {
  await promisePool.query("DELETE FROM eandc_gallery WHERE id=?", [req.params.id]);
  res.redirect("/admin/eandc");
});

app.get("/admin/eandc-faculty_profile", async (req, res) => {
  try {
    const [faculty] = await promisePool.query("SELECT * FROM eandc_faculty_profile ORDER BY id DESC");
    res.render("admin/eandc-faculty_profile", { faculty });
  } catch (err) {
    console.error(err);
    res.send("EANDC faculty page error");
  }
});

app.get("/admin/eandc-faculty/add", (req, res) => {
  res.render("admin/eandc-faculty_profile-edit", { data: null, query: req.query });
});

app.get("/admin/eandc-faculty/edit/:id", async (req, res) => {
  try {
    const [rows] = await promisePool.query("SELECT * FROM eandc_faculty_profile WHERE id=?", [req.params.id]);
    res.render("admin/eandc-faculty_profile-edit", { data: rows[0] || null });
  } catch (err) {
    console.error("Error loading EANDC faculty:", err);
    res.redirect("/admin/eandc");
  }
});

app.post("/admin/eandc-faculty/add", upload.single("teacher_image"), async (req, res) => {
  try {
    const { teacher_name, teacher_department } = req.body;
    if (!teacher_name || !teacher_department) return res.redirect("/admin/eandc-faculty/add?error=fields_required");
    if (!req.file) return res.redirect("/admin/eandc-faculty/add?error=image_required");
    const [maxId] = await promisePool.execute("SELECT COALESCE(MAX(id), 0) + 1 as next_id FROM eandc_faculty_profile");
    await promisePool.execute(
      "INSERT INTO eandc_faculty_profile (id, teacher_image, teacher_name, teacher_department) VALUES (?, ?, ?, ?)",
      [maxId[0].next_id, req.file.filename, teacher_name, teacher_department]
    );
    res.redirect("/admin/eandc");
  } catch (err) {
    console.error("Error adding EANDC faculty:", err);
    res.redirect("/admin/eandc-faculty/add?error=add_failed");
  }
});

app.post("/admin/eandc-faculty/update/:id", upload.single("teacher_image"), async (req, res) => {
  try {
    const { teacher_name, teacher_department, old_image } = req.body;
    const id = req.params.id;
    const filename = req.file ? req.file.filename : (old_image || null);
    await promisePool.query(
      "UPDATE eandc_faculty_profile SET teacher_image=?, teacher_name=?, teacher_department=? WHERE id=?",
      [filename, teacher_name, teacher_department, id]
    );
    res.redirect("/admin/eandc");
  } catch (err) {
    console.error("Error updating EANDC faculty:", err);
    res.redirect("/admin/eandc");
  }
});

app.get("/admin/eandc-faculty/delete/:id", async (req, res) => {
  await promisePool.query("DELETE FROM eandc_faculty_profile WHERE id=?", [req.params.id]);
  res.redirect("/admin/eandc");
});

// ==================================================
// EEE (same logic & flow as civil; table name "eee" only)
// ==================================================

app.get("/eee", async (req, res) => {
  try {
    const [sections] = await promisePool.query("SELECT * FROM eee");
    const [gallery] = await promisePool.query("SELECT * FROM eee_gallery");
    const [faculty] = await promisePool.query("SELECT * FROM eee_faculty_profile");

    const sectionArrays = {
      about: [], vision: [], mission: [], objectives: [], outcomes: [], specific: [],
      highlights: [], placement: [], events: [], facult_achievements: [], student_achievements: [],
      newsletter: [], fees: []
    };
    sections.forEach(row => {
      if (sectionArrays[row.section_key]) sectionArrays[row.section_key].push(row);
    });

    res.render("eee", {
      ...sectionArrays,
      eeeGallery: gallery || [],
      faculty: faculty || []
    });
  } catch (err) {
    console.error("EEE ERROR:", err);
    res.status(500).send(err.message);
  }
});

app.get("/admin/eee", async (req, res) => {
  try {
    const [data] = await promisePool.execute("SELECT * FROM eee ORDER BY id ASC");
    res.render("admin/eee", { data: data || [], query: req.query });
  } catch (err) {
    console.error("Error loading EEE:", err);
    res.render("admin/eee", { data: [], query: req.query });
  }
});

app.post("/admin/eee/update", async (req, res) => {
  try {
    const { section_key, title, content } = req.body;
    if (!section_key || section_key.trim() === "") {
      return res.redirect("/admin/eee?error=section_key_required");
    }
    const [existing] = await promisePool.execute("SELECT id FROM eee WHERE section_key = ?", [section_key]);
    if (existing.length > 0) {
      await promisePool.execute("UPDATE eee SET title = ?, content = ? WHERE section_key = ?", [title, content, section_key]);
    } else {
      const [maxId] = await promisePool.execute("SELECT COALESCE(MAX(id), 0) + 1 as next_id FROM eee");
      await promisePool.execute("INSERT INTO eee (id, section_key, title, content) VALUES (?, ?, ?, ?)", [maxId[0].next_id, section_key, title, content]);
    }
    res.redirect("/admin/eee");
  } catch (err) {
    console.error("Error updating EEE:", err);
    res.redirect("/admin/eee?error=update_failed");
  }
});

app.get("/admin/eee/edit/:section_key", async (req, res) => {
  try {
    const { section_key } = req.params;
    let data = null;
    if (section_key !== "new") {
      const [rows] = await promisePool.execute("SELECT * FROM eee WHERE section_key = ?", [section_key]);
      data = rows[0] || null;
    }
    res.render("admin/eee-edit", {
      data: data || { section_key: section_key === "new" ? "" : section_key },
      query: req.query
    });
  } catch (err) {
    console.error("Error loading EEE edit:", err);
    res.render("admin/eee-edit", { data: { section_key: req.params.section_key === "new" ? "" : req.params.section_key }, query: req.query });
  }
});

app.get("/admin/eee-gallery", async (req, res) => {
  try {
    const [gallery] = await promisePool.query("SELECT * FROM eee_gallery ORDER BY id DESC");
    res.render("admin/eee-gallery", { gallery });
  } catch (err) {
    console.error(err);
    res.send("EEE gallery error");
  }
});

app.post("/admin/eee-gallery/add", upload.single("image"), async (req, res) => {
  try {
    if (!req.file) return res.redirect("/admin/eee-gallery");
    const imagePath = `images/${req.file.filename}`;
    await promisePool.execute("INSERT INTO eee_gallery (image_path) VALUES (?)", [imagePath]);
    res.redirect("/admin/eee");
  } catch (err) {
    console.error("EEE gallery add error:", err);
    res.redirect("/admin/eee-gallery");
  }
});

app.get("/admin/eee-gallery/delete/:id", async (req, res) => {
  await promisePool.query("DELETE FROM eee_gallery WHERE id=?", [req.params.id]);
  res.redirect("/admin/eee");
});

app.get("/admin/eee-faculty_profile", async (req, res) => {
  try {
    const [faculty] = await promisePool.query("SELECT * FROM eee_faculty_profile ORDER BY id DESC");
    res.render("admin/eee-faculty_profile", { faculty });
  } catch (err) {
    console.error(err);
    res.send("EEE faculty page error");
  }
});

app.get("/admin/eee-faculty/add", (req, res) => {
  res.render("admin/eee-faculty_profile-edit", { data: null, query: req.query });
});

app.get("/admin/eee-faculty/edit/:id", async (req, res) => {
  try {
    const [rows] = await promisePool.query("SELECT * FROM eee_faculty_profile WHERE id=?", [req.params.id]);
    res.render("admin/eee-faculty_profile-edit", { data: rows[0] || null });
  } catch (err) {
    console.error("Error loading EEE faculty:", err);
    res.redirect("/admin/eee");
  }
});

app.post("/admin/eee-faculty/add", upload.single("teacher_image"), async (req, res) => {
  try {
    const { teacher_name, teacher_department } = req.body;
    if (!teacher_name || !teacher_department) return res.redirect("/admin/eee-faculty/add?error=fields_required");
    if (!req.file) return res.redirect("/admin/eee-faculty/add?error=image_required");
    const [maxId] = await promisePool.execute("SELECT COALESCE(MAX(id), 0) + 1 as next_id FROM eee_faculty_profile");
    await promisePool.execute(
      "INSERT INTO eee_faculty_profile (id, teacher_image, teacher_name, teacher_department) VALUES (?, ?, ?, ?)",
      [maxId[0].next_id, req.file.filename, teacher_name, teacher_department]
    );
    res.redirect("/admin/eee");
  } catch (err) {
    console.error("Error adding EEE faculty:", err);
    res.redirect("/admin/eee-faculty/add?error=add_failed");
  }
});

app.post("/admin/eee-faculty/update/:id", upload.single("teacher_image"), async (req, res) => {
  try {
    const { teacher_name, teacher_department, old_image } = req.body;
    const id = req.params.id;
    const filename = req.file ? req.file.filename : (old_image || null);
    await promisePool.query(
      "UPDATE eee_faculty_profile SET teacher_image=?, teacher_name=?, teacher_department=? WHERE id=?",
      [filename, teacher_name, teacher_department, id]
    );
    res.redirect("/admin/eee");
  } catch (err) {
    console.error("Error updating EEE faculty:", err);
    res.redirect("/admin/eee");
  }
});

app.get("/admin/eee-faculty/delete/:id", async (req, res) => {
  await promisePool.query("DELETE FROM eee_faculty_profile WHERE id=?", [req.params.id]);
  res.redirect("/admin/eee");
});

// ==================================================
// CSE (same logic & flow as civil)
// ==================================================

app.get("/cse", async (req, res) => {
  try {
    const [sections] = await promisePool.query("SELECT * FROM cse");
    const [gallery] = await promisePool.query("SELECT * FROM cse_gallery");
    const [faculty] = await promisePool.query("SELECT * FROM cse_faculty_profile");

    const sectionArrays = {
      about: [], vision: [], mission: [], objectives: [], outcomes: [], specific: [],
      highlights: [], placement: [], events: [], facult_achievements: [], student_achievements: [],
      newsletter: [], fees: []
    };
    sections.forEach(row => {
      if (sectionArrays[row.section_key]) sectionArrays[row.section_key].push(row);
    });

    res.render("cse", {
      ...sectionArrays,
      cseGallery: gallery || [],
      faculty: faculty || []
    });
  } catch (err) {
    console.error("CSE ERROR:", err);
    res.status(500).send(err.message);
  }
});

app.get("/admin/cse", async (req, res) => {
  try {
    const [data] = await promisePool.execute("SELECT * FROM cse ORDER BY id ASC");
    res.render("admin/cse", { data: data || [], query: req.query });
  } catch (err) {
    console.error("Error loading CSE:", err);
    res.render("admin/cse", { data: [], query: req.query });
  }
});

app.post("/admin/cse/update", async (req, res) => {
  try {
    const { section_key, title, content } = req.body;
    if (!section_key || section_key.trim() === "") {
      return res.redirect("/admin/cse?error=section_key_required");
    }
    const [existing] = await promisePool.execute("SELECT id FROM cse WHERE section_key = ?", [section_key]);
    if (existing.length > 0) {
      await promisePool.execute("UPDATE cse SET title = ?, content = ? WHERE section_key = ?", [title, content, section_key]);
    } else {
      const [maxId] = await promisePool.execute("SELECT COALESCE(MAX(id), 0) + 1 as next_id FROM cse");
      await promisePool.execute("INSERT INTO cse (id, section_key, title, content) VALUES (?, ?, ?, ?)", [maxId[0].next_id, section_key, title, content]);
    }
    res.redirect("/admin/cse");
  } catch (err) {
    console.error("Error updating CSE:", err);
    res.redirect("/admin/cse?error=update_failed");
  }
});

app.get("/admin/cse/edit/:section_key", async (req, res) => {
  try {
    const { section_key } = req.params;
    let data = null;
    if (section_key !== "new") {
      const [rows] = await promisePool.execute("SELECT * FROM cse WHERE section_key = ?", [section_key]);
      data = rows[0] || null;
    }
    res.render("admin/cse-edit", {
      data: data || { section_key: section_key === "new" ? "" : section_key },
      query: req.query
    });
  } catch (err) {
    console.error("Error loading CSE edit:", err);
    res.render("admin/cse-edit", { data: { section_key: req.params.section_key === "new" ? "" : req.params.section_key }, query: req.query });
  }
});

app.get("/admin/cse-gallery", async (req, res) => {
  try {
    const [gallery] = await promisePool.query("SELECT * FROM cse_gallery ORDER BY id DESC");
    res.render("admin/cse-gallery", { gallery });
  } catch (err) {
    console.error(err);
    res.send("CSE gallery error");
  }
});

app.post("/admin/cse-gallery/add", upload.single("image"), async (req, res) => {
  try {
    if (!req.file) return res.redirect("/admin/cse-gallery");
    const imagePath = `images/${req.file.filename}`;
    await promisePool.execute("INSERT INTO cse_gallery (image_path) VALUES (?)", [imagePath]);
    res.redirect("/admin/cse");
  } catch (err) {
    console.error("CSE gallery add error:", err);
    res.redirect("/admin/cse-gallery");
  }
});

app.get("/admin/cse-gallery/delete/:id", async (req, res) => {
  await promisePool.query("DELETE FROM cse_gallery WHERE id=?", [req.params.id]);
  res.redirect("/admin/cse");
});

app.get("/admin/cse-faculty_profile", async (req, res) => {
  try {
    const [faculty] = await promisePool.query("SELECT * FROM cse_faculty_profile ORDER BY id DESC");
    res.render("admin/cse-faculty_profile", { faculty });
  } catch (err) {
    console.error(err);
    res.send("CSE faculty page error");
  }
});

app.get("/admin/cse-faculty/add", (req, res) => {
  res.render("admin/cse-faculty_profile-edit", { data: null, query: req.query });
});

app.get("/admin/cse-faculty/edit/:id", async (req, res) => {
  try {
    const [rows] = await promisePool.query("SELECT * FROM cse_faculty_profile WHERE id=?", [req.params.id]);
    res.render("admin/cse-faculty_profile-edit", { data: rows[0] || null });
  } catch (err) {
    console.error("Error loading CSE faculty:", err);
    res.redirect("/admin/cse");
  }
});

app.post("/admin/cse-faculty/add", upload.single("teacher_image"), async (req, res) => {
  try {
    const { teacher_name, teacher_department } = req.body;
    if (!teacher_name || !teacher_department) return res.redirect("/admin/cse-faculty/add?error=fields_required");
    if (!req.file) return res.redirect("/admin/cse-faculty/add?error=image_required");
    const [maxId] = await promisePool.execute("SELECT COALESCE(MAX(id), 0) + 1 as next_id FROM cse_faculty_profile");
    await promisePool.execute(
      "INSERT INTO cse_faculty_profile (id, teacher_image, teacher_name, teacher_department) VALUES (?, ?, ?, ?)",
      [maxId[0].next_id, req.file.filename, teacher_name, teacher_department]
    );
    res.redirect("/admin/cse");
  } catch (err) {
    console.error("Error adding CSE faculty:", err);
    res.redirect("/admin/cse-faculty/add?error=add_failed");
  }
});

app.post("/admin/cse-faculty/update/:id", upload.single("teacher_image"), async (req, res) => {
  try {
    const { teacher_name, teacher_department, old_image } = req.body;
    const id = req.params.id;
    const filename = req.file ? req.file.filename : (old_image || null);
    await promisePool.query(
      "UPDATE cse_faculty_profile SET teacher_image=?, teacher_name=?, teacher_department=? WHERE id=?",
      [filename, teacher_name, teacher_department, id]
    );
    res.redirect("/admin/cse");
  } catch (err) {
    console.error("Error updating CSE faculty:", err);
    res.redirect("/admin/cse");
  }
});

app.get("/admin/cse-faculty/delete/:id", async (req, res) => {
  await promisePool.query("DELETE FROM cse_faculty_profile WHERE id=?", [req.params.id]);
  res.redirect("/admin/cse");
});

// ==================================================
// CSE Cyber Security (same logic & flow as civil)
// ==================================================

app.get("/cse-cyber-security", async (req, res) => {
  try {
    const [sections] = await promisePool.query("SELECT * FROM cse_cyber");
    const [gallery] = await promisePool.query("SELECT * FROM cse_cyber_gallery");
    const [faculty] = await promisePool.query("SELECT * FROM cse_cyber_faculty_profile");

    const sectionArrays = {
      about: [], vision: [], mission: [], objectives: [], outcomes: [], specific: [],
      highlights: [], placement: [], events: [], facult_achievements: [], student_achievements: [],
      newsletter: [], fees: []
    };
    sections.forEach(row => {
      if (sectionArrays[row.section_key]) sectionArrays[row.section_key].push(row);
    });

    res.render("cse-cyber-security", {
      ...sectionArrays,
      cseCyberGallery: gallery || [],
      faculty: faculty || []
    });
  } catch (err) {
    console.error("CSE Cyber Security ERROR:", err);
    res.status(500).send(err.message);
  }
});

app.get("/admin/cse-cyber-security", async (req, res) => {
  try {
    const [data] = await promisePool.execute("SELECT * FROM cse_cyber ORDER BY id ASC");
    res.render("admin/cse-cyber-security", { data: data || [], query: req.query });
  } catch (err) {
    console.error("Error loading CSE Cyber Security:", err);
    res.render("admin/cse-cyber-security", { data: [], query: req.query });
  }
});

app.post("/admin/cse-cyber-security/update", async (req, res) => {
  try {
    const { section_key, title, content } = req.body;
    if (!section_key || section_key.trim() === "") {
      return res.redirect("/admin/cse-cyber-security?error=section_key_required");
    }
    const [existing] = await promisePool.execute("SELECT id FROM cse_cyber WHERE section_key = ?", [section_key]);
    if (existing.length > 0) {
      await promisePool.execute("UPDATE cse_cyber SET title = ?, content = ? WHERE section_key = ?", [title, content, section_key]);
    } else {
      const [maxId] = await promisePool.execute("SELECT COALESCE(MAX(id), 0) + 1 as next_id FROM cse_cyber");
      await promisePool.execute("INSERT INTO cse_cyber (id, section_key, title, content) VALUES (?, ?, ?, ?)", [maxId[0].next_id, section_key, title, content]);
    }
    res.redirect("/admin/cse-cyber-security");
  } catch (err) {
    console.error("Error updating CSE Cyber Security:", err);
    res.redirect("/admin/cse-cyber-security?error=update_failed");
  }
});

app.get("/admin/cse-cyber-security/edit/:section_key", async (req, res) => {
  try {
    const { section_key } = req.params;
    let data = null;
    if (section_key !== "new") {
      const [rows] = await promisePool.execute("SELECT * FROM cse_cyber WHERE section_key = ?", [section_key]);
      data = rows[0] || null;
    }
    res.render("admin/cse-cyber-security-edit", {
      data: data || { section_key: section_key === "new" ? "" : section_key },
      query: req.query
    });
  } catch (err) {
    console.error("Error loading CSE Cyber Security edit:", err);
    res.render("admin/cse-cyber-security-edit", { data: { section_key: req.params.section_key === "new" ? "" : req.params.section_key }, query: req.query });
  }
});

app.get("/admin/cse-cyber-security-gallery", async (req, res) => {
  try {
    const [gallery] = await promisePool.query("SELECT * FROM cse_cyber_gallery ORDER BY id DESC");
    res.render("admin/cse-cyber-security-gallery", { gallery });
  } catch (err) {
    console.error(err);
    res.send("CSE Cyber Security gallery error");
  }
});

app.post("/admin/cse-cyber-security-gallery/add", upload.single("image"), async (req, res) => {
  try {
    if (!req.file) return res.redirect("/admin/cse-cyber-security-gallery");
    const imagePath = `images/${req.file.filename}`;
    await promisePool.execute("INSERT INTO cse_cyber_gallery (image_path) VALUES (?)", [imagePath]);
    res.redirect("/admin/cse-cyber-security");
  } catch (err) {
    console.error("CSE Cyber Security gallery add error:", err);
    res.redirect("/admin/cse-cyber-security-gallery");
  }
});

app.get("/admin/cse-cyber-security-gallery/delete/:id", async (req, res) => {
  await promisePool.query("DELETE FROM cse_cyber_gallery WHERE id=?", [req.params.id]);
  res.redirect("/admin/cse-cyber-security");
});

app.get("/admin/cse-cyber-security-faculty_profile", async (req, res) => {
  try {
    const [faculty] = await promisePool.query("SELECT * FROM cse_cyber_faculty_profile ORDER BY id DESC");
    res.render("admin/cse-cyber-security-faculty_profile", { faculty });
  } catch (err) {
    console.error(err);
    res.send("CSE Cyber Security faculty page error");
  }
});

app.get("/admin/cse-cyber-security-faculty/add", (req, res) => {
  res.render("admin/cse-cyber-security-faculty_profile-edit", { data: null, query: req.query });
});

app.get("/admin/cse-cyber-security-faculty/edit/:id", async (req, res) => {
  try {
    const [rows] = await promisePool.query("SELECT * FROM cse_cyber_faculty_profile WHERE id=?", [req.params.id]);
    res.render("admin/cse-cyber-security-faculty_profile-edit", { data: rows[0] || null });
  } catch (err) {
    console.error("Error loading CSE Cyber Security faculty:", err);
    res.redirect("/admin/cse-cyber-security");
  }
});

app.post("/admin/cse-cyber-security-faculty/add", upload.single("teacher_image"), async (req, res) => {
  try {
    const { teacher_name, teacher_department } = req.body;
    if (!teacher_name || !teacher_department) return res.redirect("/admin/cse-cyber-security-faculty/add?error=fields_required");
    if (!req.file) return res.redirect("/admin/cse-cyber-security-faculty/add?error=image_required");
    const [maxId] = await promisePool.execute("SELECT COALESCE(MAX(id), 0) + 1 as next_id FROM cse_cyber_faculty_profile");
    await promisePool.execute(
      "INSERT INTO cse_cyber_faculty_profile (id, teacher_image, teacher_name, teacher_department) VALUES (?, ?, ?, ?)",
      [maxId[0].next_id, req.file.filename, teacher_name, teacher_department]
    );
    res.redirect("/admin/cse-cyber-security");
  } catch (err) {
    console.error("Error adding CSE Cyber Security faculty:", err);
    res.redirect("/admin/cse-cyber-security-faculty/add?error=add_failed");
  }
});

app.post("/admin/cse-cyber-security-faculty/update/:id", upload.single("teacher_image"), async (req, res) => {
  try {
    const { teacher_name, teacher_department, old_image } = req.body;
    const id = req.params.id;
    const filename = req.file ? req.file.filename : (old_image || null);
    await promisePool.query(
      "UPDATE cse_cyber_faculty_profile SET teacher_image=?, teacher_name=?, teacher_department=? WHERE id=?",
      [filename, teacher_name, teacher_department, id]
    );
    res.redirect("/admin/cse-cyber-security");
  } catch (err) {
    console.error("Error updating CSE Cyber Security faculty:", err);
    res.redirect("/admin/cse-cyber-security");
  }
});

app.get("/admin/cse-cyber-security-faculty/delete/:id", async (req, res) => {
  await promisePool.query("DELETE FROM cse_cyber_faculty_profile WHERE id=?", [req.params.id]);
  res.redirect("/admin/cse-cyber-security");
});

// ==================================================
// B.Tech AI & Data Science (same logic & flow as civil)
// ==================================================

app.get("/btech-ai", async (req, res) => {
  try {
    const [sections] = await promisePool.query("SELECT * FROM btech_ai");
    const [gallery] = await promisePool.query("SELECT * FROM btech_ai_gallery");
    const [faculty] = await promisePool.query("SELECT * FROM btech_ai_faculty_profile");

    const sectionArrays = {
      about: [], vision: [], mission: [], objectives: [], outcomes: [], specific: [],
      highlights: [], placement: [], events: [], facult_achievements: [], student_achievements: [],
      newsletter: [], fees: []
    };
    sections.forEach(row => {
      if (sectionArrays[row.section_key]) sectionArrays[row.section_key].push(row);
    });

    res.render("btech-ai", {
      ...sectionArrays,
      btechAiGallery: gallery || [],
      faculty: faculty || []
    });
  } catch (err) {
    console.error("B.Tech AI ERROR:", err);
    res.status(500).send(err.message);
  }
});

app.get("/admin/btech-ai", async (req, res) => {
  try {
    const [data] = await promisePool.execute("SELECT * FROM btech_ai ORDER BY id ASC");
    res.render("admin/btech-ai", { data: data || [], query: req.query });
  } catch (err) {
    console.error("Error loading B.Tech AI:", err);
    res.render("admin/btech-ai", { data: [], query: req.query });
  }
});

app.post("/admin/btech-ai/update", async (req, res) => {
  try {
    const { section_key, title, content } = req.body;
    if (!section_key || section_key.trim() === "") {
      return res.redirect("/admin/btech-ai?error=section_key_required");
    }
    const [existing] = await promisePool.execute("SELECT id FROM btech_ai WHERE section_key = ?", [section_key]);
    if (existing.length > 0) {
      await promisePool.execute("UPDATE btech_ai SET title = ?, content = ? WHERE section_key = ?", [title, content, section_key]);
    } else {
      const [maxId] = await promisePool.execute("SELECT COALESCE(MAX(id), 0) + 1 as next_id FROM btech_ai");
      await promisePool.execute("INSERT INTO btech_ai (id, section_key, title, content) VALUES (?, ?, ?, ?)", [maxId[0].next_id, section_key, title, content]);
    }
    res.redirect("/admin/btech-ai");
  } catch (err) {
    console.error("Error updating B.Tech AI:", err);
    res.redirect("/admin/btech-ai?error=update_failed");
  }
});

app.get("/admin/btech-ai/edit/:section_key", async (req, res) => {
  try {
    const { section_key } = req.params;
    let data = null;
    if (section_key !== "new") {
      const [rows] = await promisePool.execute("SELECT * FROM btech_ai WHERE section_key = ?", [section_key]);
      data = rows[0] || null;
    }
    res.render("admin/btech-ai-edit", {
      data: data || { section_key: section_key === "new" ? "" : section_key },
      query: req.query
    });
  } catch (err) {
    console.error("Error loading B.Tech AI edit:", err);
    res.render("admin/btech-ai-edit", { data: { section_key: req.params.section_key === "new" ? "" : req.params.section_key }, query: req.query });
  }
});

app.get("/admin/btech-ai-gallery", async (req, res) => {
  try {
    const [gallery] = await promisePool.query("SELECT * FROM btech_ai_gallery ORDER BY id DESC");
    res.render("admin/btech-ai-gallery", { gallery });
  } catch (err) {
    console.error(err);
    res.send("B.Tech AI gallery error");
  }
});

app.post("/admin/btech-ai-gallery/add", upload.single("image"), async (req, res) => {
  try {
    if (!req.file) return res.redirect("/admin/btech-ai-gallery");
    const imagePath = `images/${req.file.filename}`;
    await promisePool.execute("INSERT INTO btech_ai_gallery (image_path) VALUES (?)", [imagePath]);
    res.redirect("/admin/btech-ai");
  } catch (err) {
    console.error("B.Tech AI gallery add error:", err);
    res.redirect("/admin/btech-ai-gallery");
  }
});

app.get("/admin/btech-ai-gallery/delete/:id", async (req, res) => {
  await promisePool.query("DELETE FROM btech_ai_gallery WHERE id=?", [req.params.id]);
  res.redirect("/admin/btech-ai");
});

app.get("/admin/btech-ai-faculty_profile", async (req, res) => {
  try {
    const [faculty] = await promisePool.query("SELECT * FROM btech_ai_faculty_profile ORDER BY id DESC");
    res.render("admin/btech-ai-faculty_profile", { faculty });
  } catch (err) {
    console.error(err);
    res.send("B.Tech AI faculty page error");
  }
});

app.get("/admin/btech-ai-faculty/add", (req, res) => {
  res.render("admin/btech-ai-faculty_profile-edit", { data: null, query: req.query });
});

app.get("/admin/btech-ai-faculty/edit/:id", async (req, res) => {
  try {
    const [rows] = await promisePool.query("SELECT * FROM btech_ai_faculty_profile WHERE id=?", [req.params.id]);
    res.render("admin/btech-ai-faculty_profile-edit", { data: rows[0] || null });
  } catch (err) {
    console.error("Error loading B.Tech AI faculty:", err);
    res.redirect("/admin/btech-ai");
  }
});

app.post("/admin/btech-ai-faculty/add", upload.single("teacher_image"), async (req, res) => {
  try {
    const { teacher_name, teacher_department } = req.body;
    if (!teacher_name || !teacher_department) return res.redirect("/admin/btech-ai-faculty/add?error=fields_required");
    if (!req.file) return res.redirect("/admin/btech-ai-faculty/add?error=image_required");
    const [maxId] = await promisePool.execute("SELECT COALESCE(MAX(id), 0) + 1 as next_id FROM btech_ai_faculty_profile");
    await promisePool.execute(
      "INSERT INTO btech_ai_faculty_profile (id, teacher_image, teacher_name, teacher_department) VALUES (?, ?, ?, ?)",
      [maxId[0].next_id, req.file.filename, teacher_name, teacher_department]
    );
    res.redirect("/admin/btech-ai");
  } catch (err) {
    console.error("Error adding B.Tech AI faculty:", err);
    res.redirect("/admin/btech-ai-faculty/add?error=add_failed");
  }
});

app.post("/admin/btech-ai-faculty/update/:id", upload.single("teacher_image"), async (req, res) => {
  try {
    const { teacher_name, teacher_department, old_image } = req.body;
    const id = req.params.id;
    const filename = req.file ? req.file.filename : (old_image || null);
    await promisePool.query(
      "UPDATE btech_ai_faculty_profile SET teacher_image=?, teacher_name=?, teacher_department=? WHERE id=?",
      [filename, teacher_name, teacher_department, id]
    );
    res.redirect("/admin/btech-ai");
  } catch (err) {
    console.error("Error updating B.Tech AI faculty:", err);
    res.redirect("/admin/btech-ai");
  }
});

app.get("/admin/btech-ai-faculty/delete/:id", async (req, res) => {
  await promisePool.query("DELETE FROM btech_ai_faculty_profile WHERE id=?", [req.params.id]);
  res.redirect("/admin/btech-ai");
});

// ==================================================
// B.Tech IT (same logic & flow as civil)
// ==================================================

app.get("/btech-it", async (req, res) => {
  try {
    const [sections] = await promisePool.query("SELECT * FROM btech_it");
    const [gallery] = await promisePool.query("SELECT * FROM btech_it_gallery");
    const [faculty] = await promisePool.query("SELECT * FROM btech_it_faculty_profile");

    const sectionArrays = {
      about: [], vision: [], mission: [], objectives: [], outcomes: [], specific: [],
      highlights: [], placement: [], events: [], facult_achievements: [], student_achievements: [],
      newsletter: [], fees: []
    };
    sections.forEach(row => {
      if (sectionArrays[row.section_key]) sectionArrays[row.section_key].push(row);
    });

    res.render("btech-it", {
      ...sectionArrays,
      btechItGallery: gallery || [],
      faculty: faculty || []
    });
  } catch (err) {
    console.error("B.Tech IT ERROR:", err);
    res.status(500).send(err.message);
  }
});

app.get("/admin/btech-it", async (req, res) => {
  try {
    const [data] = await promisePool.execute("SELECT * FROM btech_it ORDER BY id ASC");
    res.render("admin/btech-it", { data: data || [], query: req.query });
  } catch (err) {
    console.error("Error loading B.Tech IT:", err);
    res.render("admin/btech-it", { data: [], query: req.query });
  }
});

app.post("/admin/btech-it/update", async (req, res) => {
  try {
    const { section_key, title, content } = req.body;
    if (!section_key || section_key.trim() === "") {
      return res.redirect("/admin/btech-it?error=section_key_required");
    }
    const [existing] = await promisePool.execute("SELECT id FROM btech_it WHERE section_key = ?", [section_key]);
    if (existing.length > 0) {
      await promisePool.execute("UPDATE btech_it SET title = ?, content = ? WHERE section_key = ?", [title, content, section_key]);
    } else {
      const [maxId] = await promisePool.execute("SELECT COALESCE(MAX(id), 0) + 1 as next_id FROM btech_it");
      await promisePool.execute("INSERT INTO btech_it (id, section_key, title, content) VALUES (?, ?, ?, ?)", [maxId[0].next_id, section_key, title, content]);
    }
    res.redirect("/admin/btech-it");
  } catch (err) {
    console.error("Error updating B.Tech IT:", err);
    res.redirect("/admin/btech-it?error=update_failed");
  }
});

app.get("/admin/btech-it/edit/:section_key", async (req, res) => {
  try {
    const { section_key } = req.params;
    let data = null;
    if (section_key !== "new") {
      const [rows] = await promisePool.execute("SELECT * FROM btech_it WHERE section_key = ?", [section_key]);
      data = rows[0] || null;
    }
    res.render("admin/btech-it-edit", {
      data: data || { section_key: section_key === "new" ? "" : section_key },
      query: req.query
    });
  } catch (err) {
    console.error("Error loading B.Tech IT edit:", err);
    res.render("admin/btech-it-edit", { data: { section_key: req.params.section_key === "new" ? "" : req.params.section_key }, query: req.query });
  }
});

app.get("/admin/btech-it-gallery", async (req, res) => {
  try {
    const [gallery] = await promisePool.query("SELECT * FROM btech_it_gallery ORDER BY id DESC");
    res.render("admin/btech-it-gallery", { gallery });
  } catch (err) {
    console.error(err);
    res.send("B.Tech IT gallery error");
  }
});

app.post("/admin/btech-it-gallery/add", upload.single("image"), async (req, res) => {
  try {
    if (!req.file) return res.redirect("/admin/btech-it-gallery");
    const imagePath = `images/${req.file.filename}`;
    await promisePool.execute("INSERT INTO btech_it_gallery (image_path) VALUES (?)", [imagePath]);
    res.redirect("/admin/btech-it");
  } catch (err) {
    console.error("B.Tech IT gallery add error:", err);
    res.redirect("/admin/btech-it-gallery");
  }
});

app.get("/admin/btech-it-gallery/delete/:id", async (req, res) => {
  await promisePool.query("DELETE FROM btech_it_gallery WHERE id=?", [req.params.id]);
  res.redirect("/admin/btech-it");
});

app.get("/admin/btech-it-faculty_profile", async (req, res) => {
  try {
    const [faculty] = await promisePool.query("SELECT * FROM btech_it_faculty_profile ORDER BY id DESC");
    res.render("admin/btech-it-faculty_profile", { faculty });
  } catch (err) {
    console.error(err);
    res.send("B.Tech IT faculty page error");
  }
});

app.get("/admin/btech-it-faculty/add", (req, res) => {
  res.render("admin/btech-it-faculty_profile-edit", { data: null, query: req.query });
});

app.get("/admin/btech-it-faculty/edit/:id", async (req, res) => {
  try {
    const [rows] = await promisePool.query("SELECT * FROM btech_it_faculty_profile WHERE id=?", [req.params.id]);
    res.render("admin/btech-it-faculty_profile-edit", { data: rows[0] || null });
  } catch (err) {
    console.error("Error loading B.Tech IT faculty:", err);
    res.redirect("/admin/btech-it");
  }
});

app.post("/admin/btech-it-faculty/add", upload.single("teacher_image"), async (req, res) => {
  try {
    const { teacher_name, teacher_department } = req.body;
    if (!teacher_name || !teacher_department) return res.redirect("/admin/btech-it-faculty/add?error=fields_required");
    if (!req.file) return res.redirect("/admin/btech-it-faculty/add?error=image_required");
    const [maxId] = await promisePool.execute("SELECT COALESCE(MAX(id), 0) + 1 as next_id FROM btech_it_faculty_profile");
    await promisePool.execute(
      "INSERT INTO btech_it_faculty_profile (id, teacher_image, teacher_name, teacher_department) VALUES (?, ?, ?, ?)",
      [maxId[0].next_id, req.file.filename, teacher_name, teacher_department]
    );
    res.redirect("/admin/btech-it");
  } catch (err) {
    console.error("Error adding B.Tech IT faculty:", err);
    res.redirect("/admin/btech-it-faculty/add?error=add_failed");
  }
});

app.post("/admin/btech-it-faculty/update/:id", upload.single("teacher_image"), async (req, res) => {
  try {
    const { teacher_name, teacher_department, old_image } = req.body;
    const id = req.params.id;
    const filename = req.file ? req.file.filename : (old_image || null);
    await promisePool.query(
      "UPDATE btech_it_faculty_profile SET teacher_image=?, teacher_name=?, teacher_department=? WHERE id=?",
      [filename, teacher_name, teacher_department, id]
    );
    res.redirect("/admin/btech-it");
  } catch (err) {
    console.error("Error updating B.Tech IT faculty:", err);
    res.redirect("/admin/btech-it");
  }
});

app.get("/admin/btech-it-faculty/delete/:id", async (req, res) => {
  await promisePool.query("DELETE FROM btech_it_faculty_profile WHERE id=?", [req.params.id]);
  res.redirect("/admin/btech-it");
});

// ==================================================
// MBA (same logic & flow as civil)
// ==================================================

app.get("/mba", async (req, res) => {
  try {
    const [sections] = await promisePool.query("SELECT * FROM mba");
    const [gallery] = await promisePool.query("SELECT * FROM mba_gallery");
    const [faculty] = await promisePool.query("SELECT * FROM mba_faculty_profile");

    const sectionArrays = {
      about: [], mbaProgramme: [], experientialLearning: [], businessChallenge: [],
      leadershipSpark: [], eligibilityProcess: [], fees: []
    };
    sections.forEach(row => {
      if (sectionArrays[row.section_key]) sectionArrays[row.section_key].push(row);
    });

    res.render("mba", {
      ...sectionArrays,
      mbaGallery: gallery || [],
      faculty: faculty || []
    });
  } catch (err) {
    console.error("MBA ERROR:", err);
    res.status(500).send(err.message);
  }
});

app.get("/admin/mba", async (req, res) => {
  try {
    const [data] = await promisePool.execute("SELECT * FROM mba ORDER BY id ASC");
    res.render("admin/mba", { data: data || [], query: req.query });
  } catch (err) {
    console.error("Error loading MBA:", err);
    res.render("admin/mba", { data: [], query: req.query });
  }
});

app.post("/admin/mba/update", async (req, res) => {
  try {
    const { section_key, title, content } = req.body;
    if (!section_key || section_key.trim() === "") {
      return res.redirect("/admin/mba?error=section_key_required");
    }
    const [existing] = await promisePool.execute("SELECT id FROM mba WHERE section_key = ?", [section_key]);
    if (existing.length > 0) {
      await promisePool.execute("UPDATE mba SET title = ?, content = ? WHERE section_key = ?", [title, content, section_key]);
    } else {
      const [maxId] = await promisePool.execute("SELECT COALESCE(MAX(id), 0) + 1 as next_id FROM mba");
      await promisePool.execute("INSERT INTO mba (id, section_key, title, content) VALUES (?, ?, ?, ?)", [maxId[0].next_id, section_key, title, content]);
    }
    res.redirect("/admin/mba");
  } catch (err) {
    console.error("Error updating MBA:", err);
    res.redirect("/admin/mba?error=update_failed");
  }
});

app.get("/admin/mba/edit/:section_key", async (req, res) => {
  try {
    const { section_key } = req.params;
    let data = null;
    if (section_key !== "new") {
      const [rows] = await promisePool.execute("SELECT * FROM mba WHERE section_key = ?", [section_key]);
      data = rows[0] || null;
    }
    res.render("admin/mba-edit", {
      data: data || { section_key: section_key === "new" ? "" : section_key },
      query: req.query
    });
  } catch (err) {
    console.error("Error loading MBA edit:", err);
    res.render("admin/mba-edit", { data: { section_key: req.params.section_key === "new" ? "" : req.params.section_key }, query: req.query });
  }
});

app.get("/admin/mba-gallery", async (req, res) => {
  try {
    const [gallery] = await promisePool.query("SELECT * FROM mba_gallery ORDER BY id DESC");
    res.render("admin/mba-gallery", { gallery });
  } catch (err) {
    console.error(err);
    res.send("MBA gallery error");
  }
});

app.post("/admin/mba-gallery/add", upload.single("image"), async (req, res) => {
  try {
    if (!req.file) return res.redirect("/admin/mba-gallery");
    const imagePath = `images/${req.file.filename}`;
    await promisePool.execute("INSERT INTO mba_gallery (image_path) VALUES (?)", [imagePath]);
    res.redirect("/admin/mba");
  } catch (err) {
    console.error("MBA gallery add error:", err);
    res.redirect("/admin/mba-gallery");
  }
});

app.get("/admin/mba-gallery/delete/:id", async (req, res) => {
  await promisePool.query("DELETE FROM mba_gallery WHERE id=?", [req.params.id]);
  res.redirect("/admin/mba");
});

app.get("/admin/mba-faculty_profile", async (req, res) => {
  try {
    const [faculty] = await promisePool.query("SELECT * FROM mba_faculty_profile ORDER BY id DESC");
    res.render("admin/mba-faculty_profile", { faculty });
  } catch (err) {
    console.error(err);
    res.send("MBA faculty page error");
  }
});

app.get("/admin/mba-faculty/add", (req, res) => {
  res.render("admin/mba-faculty_profile-edit", { data: null, query: req.query });
});

app.get("/admin/mba-faculty/edit/:id", async (req, res) => {
  try {
    const [rows] = await promisePool.query("SELECT * FROM mba_faculty_profile WHERE id=?", [req.params.id]);
    res.render("admin/mba-faculty_profile-edit", { data: rows[0] || null });
  } catch (err) {
    console.error("Error loading MBA faculty:", err);
    res.redirect("/admin/mba");
  }
});

app.post("/admin/mba-faculty/add", upload.single("teacher_image"), async (req, res) => {
  try {
    const { teacher_name, teacher_department } = req.body;
    if (!teacher_name || !teacher_department) return res.redirect("/admin/mba-faculty/add?error=fields_required");
    if (!req.file) return res.redirect("/admin/mba-faculty/add?error=image_required");
    const [maxId] = await promisePool.execute("SELECT COALESCE(MAX(id), 0) + 1 as next_id FROM mba_faculty_profile");
    await promisePool.execute(
      "INSERT INTO mba_faculty_profile (id, teacher_image, teacher_name, teacher_department) VALUES (?, ?, ?, ?)",
      [maxId[0].next_id, req.file.filename, teacher_name, teacher_department]
    );
    res.redirect("/admin/mba");
  } catch (err) {
    console.error("Error adding MBA faculty:", err);
    res.redirect("/admin/mba-faculty/add?error=add_failed");
  }
});

app.post("/admin/mba-faculty/update/:id", upload.single("teacher_image"), async (req, res) => {
  try {
    const { teacher_name, teacher_department, old_image } = req.body;
    const id = req.params.id;
    const filename = req.file ? req.file.filename : (old_image || null);
    await promisePool.query(
      "UPDATE mba_faculty_profile SET teacher_image=?, teacher_name=?, teacher_department=? WHERE id=?",
      [filename, teacher_name, teacher_department, id]
    );
    res.redirect("/admin/mba");
  } catch (err) {
    console.error("Error updating MBA faculty:", err);
    res.redirect("/admin/mba");
  }
});

app.get("/admin/mba-faculty/delete/:id", async (req, res) => {
  await promisePool.query("DELETE FROM mba_faculty_profile WHERE id=?", [req.params.id]);
  res.redirect("/admin/mba");
});

app.get('/enquiry', (req, res) => {
  res.render('enquiry');
});



app.get("/:slug", async (req, res) => {

  const [rows] = await promisePool.execute(
    "SELECT * FROM pages WHERE page_slug=? AND status='active'",
    [req.params.slug]
  );

//   if (!rows.length) {
//     return res.render("404");
//   }

  const page = rows[0];

  res.render(`pages/${page.page_layout}`, { page });
});

app.get("/admin/pages/add", async (req, res) => {
  const [pages] = await promisePool.execute(
    "SELECT id, page_name FROM pages"
  );

  res.render("admin/add-new-page", { pages });
});
app.post("/admin/pages/save",
  upload.fields([
    { name: "image", maxCount: 1 },
    { name: "pdf", maxCount: 1 }
  ]),
  async (req, res) => {

    const {
      title,
      page_name,
      page_slug,
      page_layout,
      content
    } = req.body;

    const image = req.files?.image?.[0]?.filename || null;
    const pdf = req.files?.pdf?.[0]?.filename || null;

    await promisePool.execute(
      `INSERT INTO pages
      (title, page_name, page_slug, page_layout, content, image, pdf)
      VALUES (?, ?, ?, ?, ?, ?, ?)`,
      [
        title,
        page_name,
        page_slug,
        page_layout,
        content,
        image,
        pdf
      ]
    );

    res.redirect("/admin/pages/list");
  }
);
app.get("/admin/pages/list", async (req, res) => {

  const [pages] = await promisePool.execute(
    "SELECT * FROM pages ORDER BY id DESC"
  );

  res.render("admin/pages-list", { pages });
});
app.get("/admin/pages/edit/:id", async (req, res) => {

  const [rows] = await promisePool.execute(
    "SELECT * FROM pages WHERE id=?",
    [req.params.id]
  );

  if (!rows.length) return res.redirect("/admin/pages/list");

  res.render("admin/edit-page", {
    page: rows[0]
  });
});
app.post("/admin/pages/update/:id",
  upload.fields([
    { name: "image", maxCount: 1 },
    { name: "pdf", maxCount: 1 }
  ]),
  async (req, res) => {

    const { page_name, page_slug, page_layout, content } = req.body;

    const image = req.files?.image?.[0]?.filename || null;
    const pdf = req.files?.pdf?.[0]?.filename || null;

    let sql = `
      UPDATE pages SET
      page_name=?,
      page_slug=?,
      page_layout=?,
      content=?`;

    let params = [page_name, page_slug, page_layout, content];

    if (image) {
      sql += ", image=?";
      params.push(image);
    }

    if (pdf) {
      sql += ", pdf=?";
      params.push(pdf);
    }

    sql += " WHERE id=?";
    params.push(req.params.id);

    await promisePool.execute(sql, params);

    res.redirect("/admin/pages/list");
  }
);
app.post("/admin/pages/delete/:id", async (req, res) => {

  await promisePool.execute(
    "DELETE FROM pages WHERE id=?",
    [req.params.id]
  );

  res.redirect("/admin/pages/list");
});
